/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * RawMessages are a generalized form of message that can be created
 * and written to the protocol. They are are created by calling the 
 * a protocol's readMessage or newMessage methods.
 *
 * RawMessages contains the name of the message, it's source, any
 * connection settings, a set of the fields in the message, and 
 * the stream of trailing data.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 **/
package Freenet;

import Freenet.support.*;
import java.io.*;
import java.io.EOFException;
import java.util.*;

public abstract class RawMessage {

    // Public Fields
    /** The type of the message **/
    public String messageType;
    /** The set of fields **/
    public FieldSet fs;
    /** The length of the trailing data **/
    public long trailingFieldLength;
    /** The name of the trailing data **/
    public String trailingFieldName;
    /** The trailing data stream **/
    public InputStream trailingFieldStream;
    /** The source of the message **/
    public Address source;
    /** Whether to KeepAlive after reading **/
    public boolean keepAlive;
    /** Whether this message indicates that the connection should persist
     *	>0 indicates persist being set.
     *	<0 indicated persist being unset.
     *	0 indicates that persist was not specified (stay as is)
     **/
    public int persist;

    // Constructors
    /**
     *
     * @param messageType   The name of the message type.
     * @param source        The address from which this message claims to come.
     *                      may be null.
     * @param keepAlive     Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param persist       Whether the connection should persist from this
     *                      message, as above.
     * @param fs            A set of message specific fields.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param trailing       An inputstream containing the trailing data,
     *                       in a format that can be copied to destination
     *                       of this message.
     **/

    protected RawMessage(String messageType, Address source, boolean keepAlive,
			 int persist, FieldSet fs, long trailingLength, 
			 String trailingName, InputStream trailing) {
        this.messageType = messageType;
	this.source = source;
	this.keepAlive = keepAlive;
	this.persist = persist;
	this.fs = fs;
	this.trailingFieldLength = trailingLength;
	this.trailingFieldName = trailingName;
	this.trailingFieldStream = trailing;
    }

    protected RawMessage() {
    }

    // Public Methods


    public abstract void writeMessage(OutputStream out) throws IOException;
}
