package Freenet;
import Freenet.message.*;
import Freenet.support.*;
import java.util.*;

/**
 * The HandshakeHandler takes care of remembering HandshakeRequests, as well
 * as making new ones and accepting responses.
 *
 * @author oskar
 */

public class StandardHandshakeHandler implements HandshakeHandler {

    private volatile Hashtable handshakes;
    private Hashtable waiters;

    private class Waiter {
	public Address addr;
	public Waiter(Address addr) {
	    this.addr = addr;
	}
    }

    public StandardHandshakeHandler() {
	handshakes = new Hashtable();
	waiters = new Hashtable();
    }

    /**
     * Sends a handshake over a given connection. Please note that the
     * handshake will register as belonging to the port which this
     * connection is connected to, if you did not initiate the connection
     * this is probably not the same port the node listens on - and you should
     * use the method that connects itself instead.
     * @param ch   A ConnectionHandler serving the connection to send over
     * @returns    true if a handshake from this node has been served in
     *             the last Core.handshakeLife milliseconds, or if it replied
     *             to a HandshakeRequest within Core.handshakeTimeout
     *             milliseconds.  Else false.
     */

    public boolean getHandshake(Core n, ConnectionHandler ch)
    {
	Date d=(Date)handshakes.get(ch.peer());
	if(d!=null) {
	    if((d.getTime()+Core.handshakeLife) < new Date().getTime()) {
		handshakes.remove(ch.peer());
		d=null;
	    } else
		return true;
	}

	long id;
	do {
	    id = Math.abs(Core.randSource.nextLong());
	} while (id <= 0x10000);    

	HandshakeRequest hr=new HandshakeRequest(id);
	Waiter w = new Waiter(ch.peer());
	Long hkey = new Long(hr.id);
	synchronized(w) {
	    try {
		waiters.put(hkey,w);
		hr.sending(n, ch);
		ch.sendMessage(hr);
	    } catch(SendFailedException e) {
		Core.logger.log(this,
				"Couldn't send handshake request",
				Logger.MINOR); 
		return false;
	    }

	    try {
		w.wait(Core.handshakeTimeout);
	    } catch(InterruptedException e) {
	    }
	    waiters.remove(hkey);
	}
	d=(Date)handshakes.get(w.addr);
	Core.logger.log(this,"Done waiting for handshake, returning " + 
			(d != null),Logger.DEBUG);
	
	return d != null;
    }



    public boolean getHandshake(Core n, Address addr)
    {
	Date d=(Date)handshakes.get(addr);
	if(d!=null) {
	    if((d.getTime()+Core.handshakeLife) < new Date().getTime()) {
		handshakes.remove(addr);
		d=null;
	    } else
		return true;
	}
	try {
	    ConnectionHandler ch = n.connect(addr);
	    return getHandshake(n,ch);
	} catch (ConnectFailedException c) {
	    return false;
	}
    }

    public void setHandshake(long id) {	
	try {
	    Waiter w = (Waiter) waiters.get(new Long(id));
	    handshakes.put(w.addr, new Date()); 
	    synchronized(w) {
		w.notifyAll();
	    }
	} catch (ClassCastException e) {
	    Core.logger.log(this, "Something is putting bad objects in the waiter table",Logger.ERROR);
	} catch (NullPointerException e) {
	    Core.logger.log(this,"Got unexpected HandshakeReply",Logger.DEBUGGING);
	}
    }

    public void failHandshake(long id) {
	try {
	    Waiter w = (Waiter) waiters.get(new Long(id));
	    synchronized(w) {
		w.notifyAll();
	    }	
	} catch (ClassCastException e) {
	    Core.logger.log(this, "Something is putting bad objects in the waiter table",Logger.ERROR);
	} catch (NullPointerException e) {
	    Core.logger.log(this,"Got unexpected (bad) HandshakeReply",Logger.DEBUGGING);
	}
    }

}



