package Freenet.client;

import Freenet.*;
import Freenet.support.Bucket;
import java.net.BindException;

/** The AutoClient aims to provide two things:
  *     - an easy interface for client authors
  *       with selectable default metadata handling
  *     - total control over the internals for those who want it
  *
  * Easy recipes for using AutoClient:
  * 
  *     // one way to set up the Buckets
  *     File metaFile = new File("/path/to/metadata-file");
  *     File dataFile = new File("/path/to/data-file");
  *     FileBucket metaBucket = new FileBucket(metaFile);
  *     FileBucket dataBucket = new FileBucket(dataFile);
  *
  *     // get a new AutoClient with default settings and
  *     // fully automatic metadata handling
  *     AutoClient ac = new AutoClient();
  *
  *     // free-form info about node & protocol versions
  *     System.out.print(ac.nodeInfo());
  *
  *     // make a keypair for SSKs
  *     String[] keypair = ac.computeSVK();
  *     System.out.println("Private: " + keypair[0]);
  *     System.out.println("Public:  " + keypair[1]);
  * 
  *     // insert the contents of our buckets under that SSK
  *     String uri = "freenet:SSK@" + keypair[0] + "/docname";
  *     System.out.println("Inserting: " + uri);
  *     ac.put(uri, metaBucket, dataBucket);
  *
  *     // for kicks, retrieve that same SSK and put it in our buckets
  *     String uri = "freenet:SSK@" + keypair[1] + "/docname";
  *     System.out.println("Requesting: " + uri);
  *     ac.get(uri, metaBucket, dataBucket);
  *     
  *     // find out the CHK for what's in our buckets
  *     String uri = ac.computeCHK(meta, data);
  *     System.out.println("CHK is "+uri);
  *
  * Notes for advanced users:
  *
  * AutoClient deals with the *Request classes on its own because metadata
  * handling leads to the need to rewrite URIs and spawn derivative Requests.
  * The *Handler classes control it's behavior.
  *
  * It is possible to keep very fine track of what's going on with each
  * individual request performed by AutoClient, using ClientEventListeners.
  * AutoClient lets you define a set of ClientEventListeners that get
  * attached automatically to each Request.  In addition, you may give it
  * a listener for the ClientEvent SpawnedRequestEvent which will provide
  * information about each Request that AutoClient makes.
  * 
  * More on this later.
  *
  */
public class AutoClient {

    String serverAddress     = "127.0.0.1:19114";
    int listenPort           = 0;
    int defaultHTL           = 10;
    String defaultCipherName = "Twofish";
    boolean safer            = false;

    protected ClientFactory clientFactory;
    
    protected void doParams(Params p) {
        serverAddress      = p.getParam("serverAddress", serverAddress);
        listenPort         = p.getint("listenPort", listenPort);
        defaultHTL         = p.getint("htl", defaultHTL);
        defaultCipherName  = p.getParam("cipher", defaultCipherName);
        safer              = !p.getParam("safer","no").equalsIgnoreCase("no");
    }

    /** Create a new AutoClient using all default handlers,
      * settings, and the default ClientFactory (currently FNPClient).
      */
    public AutoClient() {
        this(new Params(), true);
    }
    
    /** Create a new AutoClient with the given settings and a choice
      * of whether to use all default handlers.
      * @param p
      *     listenPort      port to listen for replies on (default random)
      *     serverAddress   node to connect to (default "127.0.0.1:19114")
      *     htl             default Hops To Live (default 10)
      *     cipher          default cipher (default "Twofish")
      *     safer           use safer disk operations (default no)
      * @param useAllHandlers  whether to use all default metadata handlers
      */
    public AutoClient(Params p, boolean useAllHandlers) {
        doParams(p);
        // set up core and bind the listening port
        ClientFactory cf;
        try {
            ClientCore cc  = ClientUtil.getTCPCore(listenPort, p, Core.logger);
            Address target = ClientUtil.getAddress(serverAddress);
            cf             = new FNPClient(cc, target);
        } catch (BadAddressException e) {
            throw new IllegalArgumentException(
                "Could not parse address `" + serverAddress +
                "'.  It should be <host>:<port> or tcp/<host>:<port>" );
        } catch (BindException e) {
            throw new IllegalArgumentException(
                "Could not bind to port " + listenPort + ".  (Remember, users " +
                "only have access to ports >1024 on most systems.)" );
        }
        this.clientFactory = cf;
    }

    
    /** Create a new AutoClient with full control over its configuration.
      * @param cf  the ClientFactory to use
      * @param p   the settings
      * @param useAllHandlers  whether to use all default metadata handlers
      * @see public AutoClient(Params p, boolean useAllHandlers)
      */
    public AutoClient(ClientFactory cf, Params p, boolean useAllHandlers) {
        doParams(p);
        this.clientFactory = cf;
    }

    /** @return current default HTL setting
      */ 
    public int getDefaultHTL() {
        return defaultHTL;
    }

    /** @param htl new default HTL setting
      * @return    old default HTL setting
      */
    public int setDefaultHTL(int htl) {
        int ret = defaultHTL;
        defaultHTL = htl;
        return ret;
    }
    
    /** @return current default cipher
      */
    public String getDefaultCipherName() {
        return defaultCipherName;
    }

    /** @param cipher new default cipher name
      * @return       old default cipher name
      */
    public String setDefaultCipherName(String cipher) {
        String ret = defaultCipherName;
        defaultCipherName = cipher;
        return ret;
    }

    /** Causes all default metadata handlers to be used for future operations.
      */
    public void useAllHandlers() {
    }

    /** Adds a specific handler to the list for future operations.
      */
    public void useHandler(Handler h) {
    }

    /** @return the currently used handlers for operations
      */
    public Handler[] getCurrentHandlers() {
        return new Handler[0];
    }
    
    /** Removes a specific handler from the list for future operations.
      */
    public void clearHandler(Handler h) {
    }
    
    /** Causes no handlers to be used in future operations.
      */
    public void clearAllHandlers() {
    }

    /** Adds a specific ClientEventListener to the list of those that are
      * automatically attached to Requests spawned by AutoClient.
      */
    public void useEventListener(ClientEventListener l) {
    }

    /** @return the current set of automatically attached ClientEventListeners
      */
    public ClientEventListener[] getCurrentEventListeners() {
        return new ClientEventListener[0];
    }
    
    /** Removes a specific ClientEventListener from the set of automatically
      * attached listeners.
      */
    public void clearEventListener(ClientEventListener l) {
    }

    /** Causes no ClientEventListeners to be automatically attached in future
      * operations.
      */
    public void clearAllEventListeners() {
    }


    //=== request methods =====================================================

    /** @return free-form info about node & protocol version
      */
    public String nodeInfo() {
        return "foo";
    }

    /** @return a CHK calculated from the given buckets
      */
    public String computeCHK(Bucket meta, Bucket data) {
        return "foo";
    }
    
    /** @return a CHK calculated from the given buckets
      */
    public FreenetURI computeCHKasURI(Bucket meta, Bucket data) {
        return (FreenetURI) null;
    }

    /** @return a new SVK keypair as two Base64 encoded strings
      *         in the order private, public
      */
    public String[] computeSVK() {
        return new String[0];
    }
    
    /** @return a new SVK keypair as two byte arrays
      *         in the order private, public
      */
    public byte[][] computeSVKasByteArrays() {
        return new byte[0][0];
    }
    
    /** Do a blocking request for the given URI and store the result
      * in the given Buckets.  Uses the default HTL setting.
      * --- i am going to add some exceptions here for failure states --
      */
    public void get(String uri, Bucket meta, Bucket data) {
    }

    /** Do a blocking request for the given URI and store the result
      * in the given Buckets.
      * @param htl  use this value for HTL instead of the default
      * --- i am going to add some exceptions here for failure states --
      */
    public void get(String uri, Bucket meta, Bucket data, int htl) {
    }
    
    /** @see get(String uri, Bucket meta, Bucket data) */
    public void get(FreenetURI uri, Bucket meta, Bucket data) {
    }

    /** @see get(String uri, Bucket meta, Bucket data, int htl) */
    public void get(FreenetURI uri, Bucket meta, Bucket data, int htl) {
    }
    
    /** The "expert version."  Allows you to precisely control the Handlers
      * and ClientEventListeners, and allows the request to run asychronously
      * (meaning you must monitor the Requests for when they reach the DONE
      * or FAILED state).
      *
      * @see get(FreenetURI uri, Bucket meta, Bucket data, int htl)
      *
      * @param hy     set of handlers to use.  if null, use the current default
      *               handlers.  an empty array means use no handlers.
      * @param ly     set of listeners to use.  if null, use the current default
      *               listeners.  an empty array means use no listeners.
      * @param async  if true, runs the request on its own thread.
      */
    public void get(FreenetURI uri, Bucket meta, Bucket data, int htl,
                    Handler[] hy, ClientEventListener[] ly, boolean async) {
    }

    /** Do a blocking insert request for the given URI where the data is
      * in the given Buckets.  Uses the default HTL setting.
      * --- i am going to add some exceptions here for failure states --
      */
    public void put(String uri, Bucket meta, Bucket data) {
    }

    /** Do a blocking insert request for the given URI where the data is
      * in the given Buckets.
      * @param htl use this value for HTL instead of the default
      * --- i am going to add some exceptions here for failure states --
      */
    public void put(String uri, Bucket meta, Bucket data, int htl) {
    }

    /** @see put(String uri, Bucket meta, Bucket data) */
    public void put(FreenetURI uri, Bucket meta, Bucket data) {
    }
    
    /** @see put(String uri, Bucket meta, Bucket data, int htl) */
    public void put(FreenetURI uri, Bucket meta, Bucket data, int htl) {
    }

    /** The "expert version."  Allows you to precisely control the Handlers
      * and ClientEventListeners, and allows the request to run asychronously
      * (meaning you must monitor the Requests for when they reach the DONE
      * or FAILED state).
      *
      * @see put(FreenetURI uri, Bucket meta, Bucket data, int htl)
      *
      * @param hy     set of handlers to use.  if null, use the current default
      *               handlers.  an empty array means use no handlers.
      * @param ly     set of listeners to use.  if null, use the current default
      *               listeners.  an empty array means use no listeners.
      * @param async  if true, runs the request on its own thread.
      * @param cipherName  cipher name to use.  if null, use current default
      *                    ("Twofish" unless set otherwise).
      * @param ctBucket    temporary storage for encryption operations
      */
    public void put(FreenetURI uri, Bucket meta, Bucket data, int htl,
                    Handler[] hy, ClientEventListener[] ly,
                    String cipherName, Bucket ctBucket, boolean async) {
    }
}

