package Freenet.client;
import Freenet.client.events.*;
import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.ReadInputStream;
import java.io.*;
import java.net.BindException;
import java.util.Stack;
import Freenet.crypt.*;
import Freenet.support.Fields;

/**
 * This class contains methods for reading options in CLI clients.
 *
 * @author oskar (Mostly ripped out of old the Freenet clients though.)
 **/

public class CLI {

    /** set to on if the CLI is being used as a library instead of from the command line (yes, I know this is evil, it's temporary) **/
    //static public boolean library = false;
    /** default port to listen on for use with FNP**/
    static protected final int defaultListenPort = 0;
    /** default address of target node for use with FNP **/
    static protected final String defaultServerAddress = "tcp/127.0.0.1:19114";

    /** default address of target node FCP server**/
    static protected final String defaultFCPServerAddress = "tcp/127.0.0.1:8082";

    /** default hops to live to give request **/
    static protected final int defaultHopsToLive = 10;
    /** Default logging threshold **/
    static protected final String defaultLogging = "NORMAL";
    /** Default logging verbosity **/
    static protected final int defaultVerbosity = StandardLogger.SPARSE;
    /** Default cipher to use for encryption **/
    static protected final String defaultCipherName = "Twofish";

    /** The exit status to of the client **/
    public static int exitState = 0;

    /** When true FCP is used instead of FNP **/
    protected boolean useFCP = false;

    /** The ClientCore to use for requests when using FNP **/
    protected ClientCore cc;
    /** The Address of the target node **/
    protected Address target;
    /** The name of the cipher to use **/
    protected String cipherName; 
    /** The hops to live to give requests **/
    protected int htl;
    /** Whether to be careful with security issues like writing plaintext to
     *  disk
     **/
    protected boolean safer;

    /**
     * Creates a new CLI.
     * @param params  An object containing the settings.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  IllegalArgumentException is throw is the address is illegal.
     **/
    public CLI(Params params) throws IllegalArgumentException {
	this(params, null, false);
    }

    /**
     * Creates a new CLI with more control over logging.  This is just a 
     * temporary hack to minimize disruption; really we should make logging
     * a bit more configurable.
     *
     * @param params  An object containing the settings.
     * @param log     A Logger object.  If this is null, a new one will
     *                be created.
     * @param service If true, the doesn't clobber static members of Core.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  IllegalArgumentException is throw is the address is illegal.
     **/
    public CLI(Params params, Logger log) throws IllegalArgumentException {
	this(params, log, false);
    }


    /**
     * Creates a new CLI with more control over logging.  This is just a 
     * temporary hack to minimize disruption; really we should make logging
     * a bit more configurable.
     *
     * @param params  An object containing the settings.
     * @param log     A Logger object.  If this is null, a new one will
     *                be created.
     * @param service If true, the doesn't clobber static members of Core.
     * @exception  BindException is thrown if the client cannot listen on
     *             the given port.
     * @exception  BadAddressException is thrown if the address string cannot
     *             be parsed.
     * @exception  IllegalArgumentException is throw is the address is illegal.
     **/
    public CLI(Params params, Logger log, boolean service)
	throws IllegalArgumentException {

	// logging
	if (log == null) {
	    String logthresh = params.getParam("logging",defaultLogging);
	    int thresh = StandardLogger.priorityOf(logthresh);
	    int verb = params.getint("verbosity",defaultVerbosity);
	    log = new StandardLogger(System.err,verb,thresh);
	}
	if (params.getParam("jump","no").equalsIgnoreCase("yes")) 
	    jump();
	if (params.getParam("version","no").equalsIgnoreCase("yes")) 
	    version();


	// process options
	useFCP = params.getboolean("useFCP", false);
	int port = params.getint("listenPort", defaultListenPort);

	String addrstr = null;
	if (useFCP) {
	    addrstr = params.getParam("serverAddress", 
				      defaultFCPServerAddress);
	}
	else {
	    addrstr = params.getParam("serverAddress", 
				      defaultServerAddress);
	}

	htl = params.getint("htl", defaultHopsToLive);
	cipherName = params.getParam("cipher",defaultCipherName);
	safer = !params.getParam("safer","no").equalsIgnoreCase("no");
	
	// set up core and bind the listening port
	try {
	    if (!useFCP) {
		// Only need client core for FNP
		cc = service ? ClientUtil.getServiceCore(port) : ClientUtil.getTCPCore(port,params,log);
	    }

	    target = ClientUtil.getAddress(addrstr);

	} catch (BadAddressException e) {
	    throw new IllegalArgumentException
		("Could not parse address `" + addrstr + "'.  It should be "
		 + "<host>:<port> or tcp/<host>:<port>");
	} catch (BindException e) {
	    throw new IllegalArgumentException
		("Could not bind to port " + port + ".  (Remember, users "
		 + "only have access to ports >1024 on most systems.)");
	}
    }

    /**
     * Prints the options in the manner the constructor of this class expects
     * then to the PrintWriter, with 80 character lines.
     * @param p  Where to write the options.
     **/
    public void printOptions(PrintStream p) {
	p.println("");
	p.println("  -length bytes              Bytes to insert (if reading stdin)");
	p.println("  -metadata file             Attach private metadata to the document");
	p.println("  -cipher cipher-name        Use 'cipher-name' as the cipher");
	p.println("                             Choices are 'Twofish' (default) or 'Rijndael'");
	p.println("  -listenPort port           Local port to listen for replies on");
	p.println("  -safer yes|no              Avoid unnecessary risk by encrypting all data");
	p.println("                             written to disk. Slow and not perfect. Default off.");
	p.println("  -serverAddress address     Server node to connect to");
	p.println("  -htl hops-to-live          Hops-to-live for request");
	p.println("  -logging error|normal      Logging level");
	p.println("           |minor|debugging");
	p.println("  -verbosity 0-5             Verbosity of log messages");
    }

    /**
     * Returns an empty bucket.
     * @return A empty bucket whose type depends on the options (encrypted if
     *         "safer" was set).
     **/
    public Bucket getBucket() {
	if (safer)
	    return new CryptBucket(cipherName);
	else
	    return new FileBucket();
    }

    /**
     * Waits for all non-daemon threads to finish and then exits with the
     * exit state given in <b>exitState</b> (why doesn't java have this 
     * function??)
     **/
    public static void waitForThreadsAndExit() {
	int count;
	Thread[] ts;
	do {
	    ts = new Thread[Thread.activeCount()];
	    count = 0;
	    int n = Thread.enumerate(ts);
	    for (int i = 0; i < n ; i++) {
		if (!ts[i].isDaemon())
		    count++;
	    }
	    if (count > 1)
		Thread.currentThread().yield();
	} while (count > 1);
	OnExitCleanUp.doCleanUp();
//        if(!library)
	  System.exit(exitState);
    }

    /**
     * Prints version information about the client stdout.
     */
    public static void version() {
	System.out.println();
	System.out.println("                 Freenet Reference Client");
	System.out.println();
	int n = Math.max(Core.freenetVersion.length(),Core.protocolVersion.length());
	StringBuffer space;
	for (space = new StringBuffer(Core.protocolVersion) ; 
	     space.length() < 16 ;
	     space.append(' '));
	String proto = space.toString();
	for (space = new StringBuffer(Core.freenetVersion) ; 
	     space.length() < 16 ;
	     space.append(' '));
	String freenet = space.toString();
	System.out.println("Freenet Version:   " + freenet + 
                       "Client Version:    " + FNPClient.getVersion());  
	System.out.println("Protocol Version:  " + proto + 
                       "Build Number:      " + Core.buildNumber);
	System.exit(0);
    }

    /**
     * Like it says: jumps!
     */
    public static void jump() {
	try {
	    InputStream in = CLI.class.getResourceAsStream("dataThing");
	    if (in == null) {
		return;
	    }
	    ReadInputStream rin = 
		new ReadInputStream(new BufferedInputStream(in));
	    HexInputStream hin = new HexInputStream(rin);
	    BlockCipher bc = Util.getCipherByName("Twofish");
	    bc.initialize(new byte[] {97,98,-8,-75,2,-60,10,-52,46,-2,61,-68,108,-91,127,-33});
	    CipherInputStream cin = new CipherInputStream(bc,hin);
	    //System.out.println();
	    try {
		while (true) {
		    System.out.print((char) cin.read());
		}
	    } catch (EOFException e) {
	    }
	    System.exit(0);
	} catch (IOException e) {
	}
    }

    public static class HexInputStream extends InputStream {
	ReadInputStream rin;
	int c = 0;
	public HexInputStream(ReadInputStream rin) {
	    this.rin = rin;
	}
	public int read()  throws IOException {
	    String s = rin.readTo(' ');
	    c++;
	    return (int) Fields.stringToLong(s);
	}
    }
}





