package Freenet.client;
import Freenet.*;
import Freenet.keys.CHK;
import Freenet.crypt.*;
import Freenet.support.Fields;
import Freenet.support.Bucket;
import java.io.InputStream;
import java.io.IOException;
/**
 * ClientKey implementation for CHKs. 
 *
 * @author oskar
 **/

public class ClientCHK implements ClientKey {
    
    private InputStream in;
    private long plainLength;
    private byte[] cryptKey;
    private long totalLength = 0;
    private long partSize = 0;
    private CHK value;

    /**
     * Create a ClientCHK object, from which CHK values and data can be
     * generated from it's stream of data.
     * @param in   An InputStream of the plaintext data
     *             to read for a first content hash to use as
     *             the encryption key.
     * @param plainLength  the length of data to read;
     **/
    public ClientCHK(InputStream in, long plainLength) {
	this.in = in;
	this.plainLength = plainLength;
    }

    /**
     * Create a ClientCHK object from a Freenet CHK URI.
     * @param uri  The URI of a Freenet CHK.
     * @exception KeyException if the URI does not contain a CHK .
     **/
    public ClientCHK(FreenetURI uri) throws KeyException {
	if (!uri.getKeyType().equals("CHK"))
	    throw new KeyException("Not a CHK URI");
	value=new CHK(uri.getKeyVal());
	cryptKey=uri.getCryptoKey();
    }

    /**
     * Create a ClientCHK object from a given CHK and decrytion key.
     * @param value     The key value.
     * @param cryptKey  The key used to decrypt/encrypt data with the
     *                  given value.
     */
    public ClientCHK(CHK value, byte[] cryptKey) {
	this.value = value;
	this.cryptKey = cryptKey;
    }

    /**
     * Hashes the provided InputStream to provide a data specific
     * encryption key.
     * @exception KeyException is thrown if the read fails.
     **/
    public byte[] getEncryptionKey(int length) throws KeyException {

	if (cryptKey != null) {
	    if (length != cryptKey.length)
		throw new KeyException("Requested key length did not match that provided by URI (have " + cryptKey.length + " bytes, asked for " + 
				       length + ")");
	    else
		return cryptKey;
	} else if (in != null) {
	    byte[] entropy;
	    try {
		DigestInputStream dis = new DigestInputStream(new SHA1(),
							      in);
		byte[] buffer = new byte[512];
		    
		long rl = plainLength;
		while (rl > 0) {
		    int i = dis.read(buffer,0,(int) (buffer.length > rl ? 
						     rl : 
						     buffer.length));
		    rl -= i;
		}

		entropy = dis.getDigest().digest();
	    } catch (IOException e) {
		throw new KeyException("Generating encryption key for CHK data failed");
	    }
	    byte[] res = new byte[length];
	    Util.makeKey(entropy,res,0,length);
	    cryptKey = res;
	    return cryptKey;
	} else {
	    throw new KeyException("Cannot create key entropy from no data");
	}

    }

    public InputStream docToStream(Document doc, BlockCipher c, Bucket buck) 
	throws IOException {
	// 128 parts or 16 kBytes per part
	partSize = Math.max(plainLength >> 7,
			    16364);
	if (partSize >= plainLength)
	    partSize = 0; // single part
	ProgressiveHashOutputStream out = new 
	    ProgressiveHashOutputStream(partSize,
					new SHA1Factory(),
					buck);

	doc.write(out,c);
	out.finish();
	
	// get hash value
	int pos = 0;
	byte[] b = new byte[23];
	System.arraycopy(out.getDigest(),0,b,0,20);

	// set length
	totalLength = out.getLength();

	// set key value	
	//	for (b[20] = 0; (1 << b[20]) <= partSize ; b[20]++);  // size bound
	b[20] = Key.lengthLimit(partSize != 0 ? partSize : totalLength); 
	b[21] = (byte) (CHK.keyNumber >> 8 & 0xff); // keytype
	b[22] = (byte) (CHK.keyNumber & 0xff);
	value = new CHK(b);

	return out.getData();
    }

    public Key getKey(FieldSet storables) {
	// 20 are added since we now have a digest after the data of every part
	storables.add("PartSize",Fields.longToString(partSize == 0 ? partSize: 
						     partSize + 20));
	return value;
    }

    public Key getKey() {
	return value;
    }

    public long getDataLength() {
	return totalLength;
    }

    public long getPlainLength(long length, FieldSet storables) {
	String ps = storables.get("PartSize");
	if (ps == null)
	    return length - 1;
	try {
	    partSize = Fields.stringToLong(ps);
	    if (partSize == 0)
		return length - 1;
	} catch (NumberFormatException e) {
	    return 0;
	}
	int parts = (int) (length / (partSize + 1));
	long lastPart = length % (partSize + 1);

	return parts * (partSize - 20) + lastPart - 1;
    }

    public FreenetURI getURI() {
	return new FreenetURI("CHK",null,value.getVal(),cryptKey);
    }

    public String toString() {
	return value == null ? "new CHK" : getURI().toString();
    }
}
