package Freenet.client;

import Freenet.KeyException;
import Freenet.FieldSet;
import Freenet.Key;
import Freenet.keys.SVK;
import Freenet.crypt.*;
import Freenet.support.Bucket;
import java.io.*;
import java.util.Random;

/**
 * ClientKey implemenation for SVKs
 *
 * @author sgm
 */
public class ClientSVK implements ClientKey {
    protected long dataLen;
    protected InputStream dis;
    protected Random r;
    protected FreenetURI keyData;
    protected DSAKeyPair kp;
    protected byte[] contentHash;
    protected SVK key;

    protected ClientSVK() throws KeyException {} 

    public ClientSVK(FreenetURI key) throws KeyException {
	if (!key.getKeyType().equals("SVK"))
	    throw new KeyException("URI is not an SVK");
	keyData=key;
	this.key=new SVK(key.getKeyVal());
    }

    public ClientSVK(Random r) throws KeyException {
	this(r, null);
    }

    public ClientSVK(Random r, String docName) throws KeyException {
	this(r, docName, null);
    }

    public ClientSVK(Random r, String docName, 
		     DSAKeyPair kp) throws KeyException {
	this.kp=kp;
	this.r=r;
	keyData=new FreenetURI("SVK", docName, null, null);
    }

    public byte[] getEncryptionKey(int keyLength) throws KeyException {
	if (keyData.getCryptoKey() == null) {
	    byte[] keyBytes=new byte[keyLength];
	    r.nextBytes(keyBytes);
	    byte[] cryptoKey=new byte[keyLength];
	    Util.makeKey(keyBytes, cryptoKey, 0, cryptoKey.length);
	    keyData=new FreenetURI(keyData.getKeyType(),
				   keyData.getGuessableKey(),
				   keyData.getKeyVal(),
				   cryptoKey);
				   
	}
	return keyData.getCryptoKey();
    }
    
    public InputStream docToStream(Document doc, BlockCipher c, Bucket b) throws IOException {

	SHA1 sigctx=new SHA1();
	if (!keyData.getKeyType().equals("KSK") &&
	    keyData.getGuessableKey()!=null) {
	    byte[] docNameBytes=Util.hashString(new SHA1(), keyData.getGuessableKey());
	    sigctx.update(docNameBytes, 0, docNameBytes.length);
	}

	DigestOutputStream dos=
	    new DigestOutputStream(sigctx, 
				   b.getOutputStream());

	OutputStream out = new BufferedOutputStream(dos);
				       
	doc.write(out,c);
	out.close();
	contentHash=dos.getDigest().digest();
	dis=new 
	    SequenceInputStream(new BufferedInputStream(b.getInputStream()),
				new ByteArrayInputStream(new byte[] {0}));
	dataLen = doc.length() + 1;
	return dis;
    }
    
    public Key getKey() {
	if (key!=null) return key;
	if (kp!=null)
	    key=SVK.makeSVK(kp, keyData.getGuessableKey(), null);
	else {
	    key=SVK.makeSVK(r, keyData.getGuessableKey(), null);
 	    kp=key.getKeyPair();
	}
	keyData=new FreenetURI(keyData.getKeyType(), 
			       keyData.getGuessableKey(), 
			       key.getVal(), 
			       keyData.getCryptoKey());
	return key;
    }
    
    public Key getKey(FieldSet storables) {
	if (kp!=null) 
	    key=SVK.makeSVK(kp, keyData.getGuessableKey(), storables);
	else {
	    key=SVK.makeSVK(r, keyData.getGuessableKey(), storables);
	    kp=key.getKeyPair();
	}
	key.sign(contentHash, r, storables);
	keyData=new FreenetURI(keyData.getKeyType(), 
			       keyData.getGuessableKey(), 
			       key.getVal(), 
			       keyData.getCryptoKey());
	return key;
    }

    /** @return the public key for this SVK
      */
    public byte[] getPublicKey() {
        return key == null ? null : key.getKeyPair().getY().toByteArray();
    }

    /** @return the public key fingerprint for this SVK
      */
    public byte[] getPublicKeyFingerPrint() {
        return key == null ? null : key.getKeyPair().fingerprint();
    }

    /**
     * Return the Private Key for this SVK as a byte array
     */
    public byte[] getPrivateKey() {
	return key == null ? null : key.getKeyPair().getX().toByteArray();
    }
    /**
     * Returns the Private Key for this SVK encoded in Freenet's modified 
     * Base64.
     */
    public String getPrivateKeyString() {
	return key == null ? null : Base64.encode(getPrivateKey());
    }

    
    public long getDataLength() {
	return dataLen;
    }

    public long getPlainLength(long length, FieldSet storables) {
	return length - 1;
    }

    public FreenetURI getURI() {
	return keyData;
    }

    public static void main(String[] args) throws Exception {
	ClientSVK s=new ClientSVK(new Yarrow());
	s.getKey();
	s.getEncryptionKey(16);
	System.err.println(s.getURI());
	FreenetURI n=new FreenetURI(s.getURI().toString());
	n.decompose();
    }

    public String toString() {
	return key == null ? "new SVK" : getURI().toString();
    }
}


	
	





