package Freenet.client;
import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.WriteOutputStream;
import Freenet.presentation.FreenetProtocol;
import Freenet.transport.*;
import java.lang.reflect.InvocationTargetException;
import java.net.*;
import java.io.*;

import java.text.SimpleDateFormat;
import java.text.ParsePosition;
import java.util.GregorianCalendar;
import java.util.Date;
import java.util.TimeZone;

/**
 * Since the Freenet client libraries are based around the same code
 * as the reference node, they have the same protocol layer abstractions
 * allowing the application layer to work on top of multiple presentation
 * and transport layers. Since these abstraction will be of little use to
 * most Client authors, this class provides a couple of static methods to
 * produce instances of Freenet related objects in a normal tcp/FreenetProtol
 * environment.
 *
 * @author oskar
 **/
public abstract class ClientUtil {

    /**
     * Create a ClientCore for a TCP Freenet protocol. Most Internet Freenet
     * clients can retrieve a Core this way, and shouldn't have to bother with
     * it's actual constructor. This will create a silent Core with default 
     * options.
     *
     * @param port     The port to listen for replies on. 
     */
    public static ClientCore getTCPCore(int port) throws BindException {
        return getTCPCore(port, new Params(),new SuppressedLogger());
    }

    /**
     * Create a ClientCore for a TCP Freenet protocol. Most Internet Freenet
     * clients can retrieve a Core this way, and shouldn't have to bother with
     * its actual constructor.
     *
     * @param port     The port to listen for replies on. 
     * @param options  A Params set of Core parameters. See seperate 
     *                 documentation.
     * @param log      A Logger object to get log calls from the Core and 
     *                 Clients.
     **/
    public static ClientCore getTCPCore(int port, Params options, Logger log) 
        throws BindException {
        try {
            options.setParam("transient","yes"); // clients are transient
            Core.setLogger(log);
            Core.init(options);

            ClientCore core =  new 
                ClientCore(new ListeningAddress("tcp",
                                                new tcpListeningAddress(port)),
                           new FreenetProtocol(true));
            core.acceptConnections();
            return core;
        } catch (ListenException e) {
            throw new BindException("Error binding to port " + port);
        }
    }

    /**
     * Create a TCP ClientCore for use in the same JVM as a node.  Uses the params
     * and options of the hosting node.
     *
     * @param port      The port to listen for replies on.
     **/
    public static ClientCore getServiceCore(int port)
        throws BindException {
        try {
            ClientCore core = new
                ClientCore(new ListeningAddress("tcp",
                                                new tcpListeningAddress(port)),
                           new FreenetProtocol());
            core.acceptConnections();
            return core;
        } catch (ListenException e) {
            throw new BindException("Error binding to port " + port);
        }
    }


    /**
     * Returns a Freenet address object for an Internet address.
     * @param host  the host which the address should represent.
     * @param port  the port of that host the address should represent.
     * @exception IllegalArgumentException is thrown if port is outside range.
     **/
    public static Address getAddress(InetAddress host,int port) 
        throws IllegalArgumentException {
        return new Address("tcp",
                           new tcpAddress(host,
                                          port));
    }

    /**
     * Returns a Freenet address object for an Internet name.
     * @param host  the name of the internet host the address should represent.
     * @param port  the port of that host the address should represent.
     * @exception IllegalArgumentException is thrown if port is outside range.
     *            BadAddressException is thrown if the name cannot be resolved.
     **/
    public static Address getAddress(String host, int port) throws IllegalArgumentException, BadAddressException {
        return new Address("tcp",
                           new tcpAddress(host,
                                          port));
    }
    /**
     * Returns a Freenet address object for an Internet name.
     * @param address  the address is [tcp/]host:port notation. Note that port
     *                 is mandatory. Freenet has no standard port.
     **/
    public static Address getAddress(String s) throws IllegalArgumentException, BadAddressException {
        if (s.indexOf('/') == -1)
            s = "tcp/" + s;
        return new Address(s);
    }

    /**
     * Returns a ClientKey object from FreenetURI.
     * @param furi  The URI of a Key on Freenet.
     * @exception KeyException if the key contained in the URI is not correct.
     **/
    public static ClientKey getKeyFromURI(FreenetURI furi) {
        try {
            ClientKey k=
                (ClientKey) 
                Loader.getInstance("Freenet.client.Client" + 
                                   furi.getKeyType(),
                                   new Class[] {Freenet.client.FreenetURI.class},
                                   new Object[] {furi});
            return k;
        } catch (InvocationTargetException e) {
            e.printStackTrace();
            if (e.getTargetException() instanceof KeyException) {
                throw (KeyException) e.getTargetException();
            } else
                throw new KeyException(e.toString());
        } catch (ClassNotFoundException e) {
            System.out.println(e);
            throw new KeyException("Unknown keytype");
        } catch (ClassCastException e) {
            throw new KeyException("Class found, but didn't contain Key");
        } catch (NoSuchMethodException e) {
            throw new KeyException("Key not constructable from URI");
        } catch (InstantiationException e) {
            throw new KeyException("Something went wrong");
        } catch (IllegalAccessException e) {
            throw new KeyException("Illegal access");
        }
    }
    
    protected static FileBucket makeRedirectBucket(FreenetURI keyTo)
        throws IOException {
        return makeRedirectBucket(keyTo, new FieldSet());
    }
    
    protected static FileBucket makeRedirectBucket(FreenetURI keyTo, FieldSet fields)
        throws IOException {
        FileBucket b=new FileBucket();
        WriteOutputStream w=new WriteOutputStream(new FileOutputStream(b.getFile()));
        w.println("Redirect");
        fields.writeFields(w);
//      w.println("End");
        w.println(keyTo.toString());
        w.close();
        
        return b;
    }
    
    protected static File makeRedirect(FreenetURI keyTo)
        throws IOException {
        return makeRedirect(keyTo, new FieldSet());
    }

    protected static File makeRedirect(FreenetURI keyTo, FieldSet fields) 
        throws IOException {

        return makeRedirectBucket(keyTo, fields).getFile();
    }
    
    public static FreenetURI dateURI( FreenetURI uri, String base_string, String incr_string )
        throws MalformedURLException {
        
        return dateURI( uri, base_string, Long.parseLong(incr_string.trim()), 0);
    }
    
    public static FreenetURI dateURI( FreenetURI uri, String base_string, long incr, long offset) 
        throws MalformedURLException {
                                
        long increment = incr * 1000;
        
        if (uri.getGuessableKey() == null) {
            throw new MalformedURLException("Can't Date Redirect to non-guessable keytype.");
        }
        
        SimpleDateFormat df = new SimpleDateFormat("yyyyMMddHHmmss zzz");
        df.setTimeZone(TimeZone.getTimeZone("GMT"));

        ParsePosition pos = new ParsePosition(0);

        // append GMT to force interpretation as a GMT time.
        Date base_date = df.parse(base_string + " GMT", pos);
        
        long base_long = base_date.getTime();
        
        GregorianCalendar cal = new GregorianCalendar(TimeZone.getTimeZone("GMT"));
        long now = cal.getTime().getTime();
        
        long delta = (now + offset * increment ) - base_long;
        
        String latest = df.format( new Date(base_long + ( (long) (delta / increment) ) * increment ) );
        
        FreenetURI newuri = new FreenetURI(uri.getKeyType(),
                                           latest.substring(0,14)+"-"+uri.getGuessableKey(),
                                           uri.getKeyVal(),
                                           uri.getCryptoKey());
        
//      System.err.println("new URI:"+newuri);
        
        return newuri;
    }

    /** @return true if the string is an MSK uri
      * @deprecated
      */
    public static boolean isMSK(String uri) {
        if (uri.indexOf("//") < 0) return false;
        if (uri.toLowerCase().startsWith("freenet:")) uri = uri.substring(8);
        if (uri.toLowerCase().startsWith("ksk@")) return false;
        if (uri.toLowerCase().startsWith("ssk@")) return false;
        if (uri.toLowerCase().startsWith("svk@")) return false;
        if (uri.toLowerCase().startsWith("chk@")) return false;
        return true;
    }

    /** Returns the target URI string by requesting the MSK URI.
      * @param  uri   A string containing the uri.
      * @throws MalformedURLException if the URI cannot be parsed
      * @throws KeyException
      * @deprecated
      */
    public static String lookupMSK(String uri)
        throws MalformedURLException, KeyException {
        return lookupMSK(uri, new Params());
    }

    /** Returns the target URI string by requesting the MSK URI.
      * @param  uri     A string containing the uri.
      * @param  params
      * @throws MalformedURLException if the URI cannot be parsed
      * @throws KeyException
      * @deprecated
      */
    public static String lookupMSK(String uri, Params params)
        throws MalformedURLException, KeyException {
        // MapHandler.lookup can throw a KeyException if
        // if the MSK mapfile can't be read.
        uri = MapHandler.lookup(uri, params);
        if (uri == null) throw new KeyException("Filename not found in mapfile.");
        return uri;
    }
}

