package Freenet.client;

import Freenet.crypt.*;
import Freenet.crypt.ciphers.*;
import Freenet.FieldSet;
import Freenet.support.Fields;
import Freenet.support.OnExitCleanUp;
import java.util.*;
import java.io.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This class holds all relevant data about a Freenet document to support
 * document-level encryption and verification.
 */
public class Document {
    protected long metadatalength, doclength;
    protected byte[] encryptionKey, contentHash;
    protected InputStream metadata, data;

    /**
     * Construct a Document primitive with the data stored on the stream
     * <b>source</b>, and the encryption key <b>key<b>. 
     * Note that this does *not* mean that <b>source</b> is encrypted, but
     * rather that if it were, it would use <b>key</b> as its key.
     */
    public Document(InputStream source, byte[] key) {
	this(source, null, key, -1, 0);
    }

    public Document(InputStream source, InputStream metadata,
		    byte[] key, long dl, long ml) {
	this.encryptionKey=key;
	this.data=source;
	this.metadata=metadata;
	metadatalength=ml;
	doclength=dl;
    }

    public Document(File data, File metadata, byte[] key) throws IOException {
	this(new FileInputStream(data), new FileInputStream(metadata),
	     key, data.length(), metadata.length());
    }

    public void skipMetadata() throws IOException {
	long ic=0;
	while (ic<metadatalength) 
	    ic+=metadata.skip(metadatalength-ic);
    }

    public boolean hasMetadata() {
	return metadatalength!=0;
    }

    /**
     * Attach a metadata stream with the given length to the document
     */
    public void attachMetadata(InputStream metadata, long length) {
	metadatalength=length;
	this.metadata=metadata;
    }

    public long length() {
	return 2 + encryptionKey.length + metadatalength + doclength;
    }

    protected long metadataLength() {
	return metadatalength;
    }

    protected long dataLength() {
	return doclength;
    }

    protected long documentOffset() {
	return 2 + encryptionKey.length + metadatalength;
    }

    /**
     * Write this document to the specified stream, encrypted using its
     * encryption key and the specified block cipher.  This method
     * <i>must</i> be used by a client when inserting data into Freenet, 
     * as it generates the appropriate Freenet document structure.
     */
    public void write(OutputStream out, BlockCipher c) throws IOException {
	c.initialize(encryptionKey);
	write(new CipherOutputStream(c, out));
    }

    /**
     * Write this document to the specified stream in the clear.
     */
    public void write(OutputStream out) throws IOException {
	DataOutputStream dos=new DataOutputStream(out);
	dos.writeChar((char)encryptionKey.length);
	dos.write(encryptionKey);
	
	writeMetadata(out);
	writeData(out);
    }

    /**
     * Write only the metadata portion of this document to the given
     * stream in the clear.
     */
    public void writeMetadata(OutputStream out) throws IOException {
	if (metadata != null)
	    writeData(metadata, out, metadatalength);
    }

    /**
     * Write only the data portion of this document to the given
     * stream in the clear.
     */
    public void writeData(OutputStream out) throws IOException {
	if (data != null)
	    writeData(data, out, doclength);
    }

    protected void writeData(InputStream data, OutputStream out, long len) throws IOException {
	byte[] buffer=new byte[65536];
	int rc=0;

	do {
	    rc=data.read(buffer, 0, (int)Math.min(len, buffer.length));
	    if (rc>0) {
		len-=rc;
		out.write(buffer, 0, rc);
	    }
	} while (rc!=-1 && len!=0);
	out.flush();
    }

    public InputStream toInputStream(BlockCipher c) throws IOException {
	c.initialize(encryptionKey);
	byte[] header=new byte[encryptionKey.length+2];
	header[0]=(byte)(encryptionKey.length>>8 & 0xff);
	header[1]=(byte)encryptionKey.length;
	System.arraycopy(encryptionKey, 0, header, 2, encryptionKey.length);

	// Write to a temp file to encrypt the data and generate the 
	// hash for signing
	
	SequenceInputStream input=
	    new SequenceInputStream(new ByteArrayInputStream(header), 
				    new BufferedInputStream( 
				     (metadata == null) ? data :
				     new SequenceInputStream(metadata, data)));
	//	return new EncipherStream(input, c);
	
	
	File tempFile=
	    new File(Util.intToHexString(Freenet.Core.randSource.nextInt())+
		     ".tmp");
	OnExitCleanUp.deleteOnExit(tempFile);


	DigestOutputStream dos=
	    new DigestOutputStream(new SHA1(),
				   new FileOutputStream(tempFile));
	CipherOutputStream out=
	    new CipherOutputStream(c, new BufferedOutputStream(dos));

	byte[] buffer=new byte[32768];

	long bl=length();
	while (bl > 0) {
	    int rc=input.read(buffer);

	    if (rc!=-1) {
		out.write(buffer,0,rc);
		bl-=rc;
	    } else 
		throw new EOFException();
	}
	out.close();
	contentHash=dos.getDigest().digest();
	
	BufferedInputStream in=new BufferedInputStream(new FileInputStream(tempFile));
	
	return in;
	
    }

    public InputStream toInputStream() throws IOException {
	return toInputStream(null);
    }

    /**
     * This static method allows an unencrypted Document to be read from 
     * the specified input stream.  It reads a fully structured
     * Freenet document.
     */
    public static Document read(InputStream in, long dlen, 
				long mlen) throws IOException {
	DataInputStream dis=new DataInputStream(in);
	char keylen=dis.readChar();
	byte[] encryptionKey=new byte[keylen];
	dis.readFully(encryptionKey);
	return new Document(in, in, encryptionKey, dlen, mlen);
    }

    
    /**
     * Reads a document from the specified, encrypted stream.  It uses
     * <b>key</b> as the encryption key, and will also verify that the
     * document is valid by comparing the stored key against <b>key</b>.
     * 
     * This reads the structure only.  The data is left unread in the 
     * Document.data field.  The contents of the document can then be 
     * written to an output stream with d.writeData(OutputStream).
     *
     * This method throws a StreamCorruptedException immediately upon
     * finding that the document does not decrypt to a valid Freenet
     * document (i.e. when the stored key does not match the given key)
     */
    public static Document read(InputStream in, byte[] key, 
				BlockCipher c, long dlen,
				long mlen) throws IOException, StreamCorruptedException {
	c.initialize(key);
	CipherInputStream ins=new CipherInputStream(c, in, false);
	return read(ins, key, dlen, mlen);
    }

    /**
     * Reads a document from a clear stream, <b>in<b>, verifying that
     * the given key <b>key</b>, matches the stored key of the document 
     */
    public static Document read(InputStream in, byte[] key,
				long dlen, long mlen) throws IOException, StreamCorruptedException {
	Document d=read(in, dlen, mlen); 
	/*
	if (key.length==d.encryptionKey.length) {
	    for (int i=0; i<key.length; i++) 
		if (key[i]!=d.encryptionKey[i])
		    throw new StreamCorruptedException("Key mismatch");
	} else 
	    throw new StreamCorruptedException("Key mismatch");
	*/
	return d;
    }

    public static void main(String[] args) throws Exception {
	SHA1 ctx=new SHA1();
	Twofish c=new Twofish();
	byte[] key=new byte[c.getKeySize()>>3];
	File f=new File(args[1]);
	
	if (args[0].equals("create")) {
	    byte[] hash=Util.hashFile(ctx, f);
	    System.err.println(Fields.bytesToHex(hash,0,key.length));
	    Util.makeKey(hash, key, 0, key.length);
	    System.err.println(Fields.bytesToHex(key,0,key.length));

	    FileInputStream in=new FileInputStream(f);
	    Document d=new Document(in, key);
	    InputStream i=d.toInputStream(c);
	    byte[] buffer=new byte[65536];
	    int rc=0;
	    OutputStream out=System.out;

	    do {
		rc=i.read(buffer, 0, buffer.length);
		if (rc>0) {
		    out.write(buffer, 0, rc);
		}
	    } while (rc!=-1);
	    out.flush();	    
	} else if (args[0].equals("read")) {
	    Fields.hexToBytes(args[2],key,0);
	    c.initialize(key);
	    Document d=read(new FileInputStream(f), key, c, 0, 0);
	    d.writeData(System.out);
	} else if (args[0].equals("encipher")) {
	    Util.makeKey(Util.hashFile(ctx, f), key, 0, key.length);
	    System.err.println(Fields.bytesToHex(key,0,key.length));

	    FileInputStream in=new FileInputStream(f);
	    Document d=new Document(in, key);
	    InputStream inf=d.toInputStream(c);
	    int rc=0;
	    while (rc!=-1) {
		rc=inf.read();
		if (rc!=-1) System.out.write(rc);
	    }
	    System.out.flush();
	}	    
    }
}
