package Freenet.client;

/** 
 * FCP version of Client library
 *
 * @author <a href="mailto:rrkapitz@stud.informatik.uni-erlangen.de">Ruediger Kapitza</a>
 * @version 
 */

import java.io.*;
import java.net.*;
import java.math.BigInteger;
import Freenet.*;
import Freenet.client.events.*;
import Freenet.crypt.*;
import Freenet.presentation.*;
import Freenet.transport.tcpAddress;
import Freenet.support.io.ReadInputStream;
import Freenet.support.io.WriteOutputStream;

public class FCPClient implements ClientFactory {
 
    protected Address target;

    /** 
     * Create a new ClientFactory for spawning Clients to process fcp-requests.
     * @param target  The fcp-service to send messages to. This service should be 
     *                running locally.
     */    
    public FCPClient(Address target){
        this.target = target;
    }
    
    public Client obtainClient(Request req) throws UnsupportedRequestException, KeyException, IOException {
        
        if (req instanceof DataRequest) {
            FCPRequest i = new FCPRequest(req);
            (new Thread(i, "ClientGet: " + ((DataRequest) req).uri)).start();
            return i;
        }else if (req instanceof InsertRequest) {
            FCPInsert i = new FCPInsert(req);
            (new Thread(i, "ClientPut: " + ((InsertRequest) req).uri)).start();
            return i;
        }else if (req instanceof ComputeCHKRequest) {
            FCPComputeCHK i = new FCPComputeCHK(req);
            (new Thread(i, "GenerateCHK")).start();
            return i;
        }else if (req instanceof ComputeSVKRequest) {
            FCPComputeSVK i = new FCPComputeSVK(req);
            (new Thread(i, "GenerateSVKPair")).start();
            return i;
        }else if (req instanceof HandshakeRequest) {
            FCPHandshake i = new FCPHandshake(req);
            (new Thread(i, "ClientHello")).start();
            return i;
        }else{
            throw new UnsupportedRequestException();
        }
    }

    
    protected abstract class FCPInstance implements Client, Runnable, ClientEventListener {
        
        protected Request req;
        protected boolean ready = false;
        protected Socket soc;  
        protected InputStream in;
        protected OutputStream out;

        FCPInstance(Request req){
            this.req = req;
        }

        public int state() {
            return req.state;
        }

        public void state(int state) {
            if (state >= Request.FAILED && state <= Request.DONE) {
                req.state = state;
            }
            req.produceEvent(new StateReachedEvent(state));
        }

        protected synchronized void prepare() throws Exception {

            // new socket-connection
            soc = new Socket(((tcpAddress)(target.address())).host,((tcpAddress)target.address()).port);
            out = soc.getOutputStream();
            in = new BufferedInputStream( soc.getInputStream());

            state(Request.PREPARED);
            
            while (!ready) {
                this.wait();
            }
            doit();        
        }

        protected abstract void doit();
    
        public final synchronized void execute() {
            ready = true;
            if (req.state == Request.PREPARED)
                this.notify();
        }

    
        public final void run() {
            try {
                prepare();
            } catch (Exception e) {
                
                req.produceEvent(new ExceptionEvent(e));
            
                synchronized(this) {
                    state(Request.FAILED); 
                    this.notifyAll();
                }
            }
        }

        public void receive(ClientEvent ce) {
            req.produceEvent(ce);
        }

        // FCP-Methods 

        protected void close() throws IOException{
            in.close(); out.close(); soc.close();
        }
    
        protected void send(String messageType, FieldSet fs) throws IOException {
	    final String PROTOCOL_HEADER = "\000\000\000\002";
            WriteOutputStream writer = new WriteOutputStream(out);
            writer.writeUTF(PROTOCOL_HEADER + messageType, '\n');
            
            if (fs == null){
                writer.writeUTF("EndMessage", '\n');
            }else{
                fs.writeFields(writer, "EndMessage", '\n', '=', '.');
            }
            writer.flush();
        }
    
        protected FNPRawMessage getMessage() throws IOException{
            FNPRawMessage m = null; 
            ReadInputStream reader = new ReadInputStream(in);

            try{
                m = new FNPRawMessage(in);
            } catch (Exception e){
                req.produceEvent(new ErrorEvent("Could not parse message."));
                close();
		throw new IOException("Could not parse message." + e.toString());
            } 
            return m;
        }
        
    }

    protected class FCPHandshake extends FCPInstance {
        
        public FCPHandshake(Request req){
            super(req);
        }

        protected synchronized void doit() {
            try{
                state(Request.REQUESTING);
                sayHello();
                state(Request.DONE);
            } catch(Exception e){
                try{
                    close();
                }catch(IOException ioe){}
                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
        }

        protected void sayHello() throws IOException{
            FieldSet fs = new FieldSet();
            
            send("ClientHello",fs);

            FNPRawMessage m = getMessage();
            
            if (m.messageType.equals("NodeHello")){
                ((HandshakeRequest) req).prot  = m.fs.get("Protocol");
                ((HandshakeRequest) req).node  = m.fs.get("Node");
            }else{
                req.produceEvent(new ErrorEvent("Invalid message type " + m.messageType ));                      }
            
            close();
            return;
        }
    }

    protected class FCPComputeSVK extends FCPInstance {
        
        public FCPComputeSVK(Request req) throws IOException {
            super(req);
        }
        
        protected synchronized void doit() {
            try{
                state(Request.REQUESTING);
                computeSVK();
                state(Request.DONE);
            } catch(Exception e){
                try{
                    close();
                }catch(IOException ioe){}
                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
        }
        
        public void computeSVK() throws IOException, IllegalBase64Exception {
            FieldSet fs = new FieldSet();
            
            send("GenerateSVKPair",fs);

            FNPRawMessage m = getMessage();
            
            if (m.messageType.equals("Success")){
                
                BigInteger priv = new BigInteger(Base64.decode(m.fs.get("PrivateKey")));
                DSAKeyPair kp = new DSAKeyPair(priv, Global.DSAgroupB);
                ((ComputeSVKRequest) req).clientKey = new ClientSVK(null, null, kp);

            }else {
                req.produceEvent(new ErrorEvent("Invalid message typ " + m.messageType ));           
            }
            
            close();
            return;
        }
    }

    protected class FCPComputeCHK extends FCPInstance {
        
        public FCPComputeCHK(Request req) throws IOException {
            super(req);
        }
        
        protected synchronized void doit() {
            try{
                state(Request.REQUESTING);
                computeCHK(((ComputeCHKRequest) req).meta.size(),
                           (((ComputeCHKRequest) req).meta).getInputStream(),
                           ((ComputeCHKRequest) req).data.size(),
                           (((ComputeCHKRequest) req).data).getInputStream());
                state(Request.DONE);
            } catch(Exception e){
                try{
                    close();
                }catch(IOException ioe){}

                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
        }

        public void computeCHK(long mlen,InputStream meta, long  dlen, InputStream data) throws IOException, MalformedURLException {
            
            FieldSet fs = new FieldSet();
            byte buf[] = new byte[2048];
            long len;

            fs.put("DataLength",Long.toHexString(dlen+mlen));
            fs.put("MetadataLength",Long.toHexString(mlen)); 
            
            send("GenerateCHK",fs);
        
            if(mlen != 0){
                while ((len=meta.read(buf))!= -1){
                    out.write(buf,0,(int) len);
                    mlen = mlen - len;
                }
            }
            
            if(dlen !=0){
                while ((len = data.read(buf)) != -1){
                    out.write(buf,0,(int) len);
                    dlen = dlen - len;
                }
            }
            
            FNPRawMessage m = getMessage();

            if (m.messageType.equals("Success")){

                ((ComputeCHKRequest)req).clientKey = (ClientCHK) new ClientCHK(new FreenetURI(m.fs.get("URI")));
            }else {
                req.produceEvent(new ErrorEvent("Unkown message type "+ m.messageType ));           
            }
            
            close();
            return;
        }
    }

    
    protected class FCPInsert extends FCPInstance {
        
        public FCPInsert(Request req){
            super(req);
        }
        
        public void doit() {
            try{
                state(Request.REQUESTING);

                insert((((InsertRequest) req).uri).toString(),
                       ((InsertRequest) req).htl,
                       ((InsertRequest) req).meta.size(),
                       (((InsertRequest) req).meta).getInputStream(),
                       ((InsertRequest) req).data.size(),
                       (((InsertRequest) req).data).getInputStream());
                
            } catch(Exception e){
                try{
                    close();
                }catch(IOException ioe){}

                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
        }


        public void insert(String uri,int htl,long mlen, InputStream  m_data,long dlen, InputStream data) throws IOException {
            FieldSet fs = new FieldSet();
            long len;
            byte buf[] = new byte[2048];
            
            fs.put("URI",uri);
            fs.put("HopsToLive",Integer.toHexString(htl));
            fs.put("DataLength",Long.toHexString(dlen+mlen));
            fs.put("MetadataLength",Long.toHexString(mlen)); 
            
            state(Request.TRANSFERRING);

            send("ClientPut",fs);
            
            if(mlen != 0){
                while ((len=m_data.read(buf))!= -1){
                    out.write(buf,0,(int) len);
                    mlen = mlen - len;
                }
            }
        
            if(dlen != 0){
                while ((len = data.read(buf)) != -1){
                    out.write(buf,0,(int) len);
                    dlen = dlen - len;
                }
            }
            
            FNPRawMessage m = getMessage();
        
            if (m.messageType.equals("Success")){

                
                InsertRequest _req = (InsertRequest) req;
                FreenetURI _uri =  new FreenetURI(m.fs.get("URI"));
                
                if((_uri.getKeyType()).equals("KSK")){
                    _req.clientKey =  new ClientKSK(_uri);
                }else if ((_uri.getKeyType()).equals("CHK")){
                    _req.clientKey =  new ClientCHK(_uri);
                }else if ((_uri.getKeyType()).equals("SVK")){
                    _req.clientKey =  new ClientSVK(_uri);
                }else if ((_uri.getKeyType()).equals("SSK")){
                    _req.clientKey =  new ClientSSK(_uri);
                }
                state(Request.DONE);
                
            }else{
                if(m.messageType.equals("FormatError"))
                    req.produceEvent(new ErrorEvent("FormatError from ClientPut"));           
                else if(m.messageType.equals("URIError"))
                    req.produceEvent(new ErrorEvent("URIError from ClientPut: "+uri));           
                else if(m.messageType.equals("RouteNotFound"))
                    req.produceEvent(new RouteNotFoundEvent());           
                else if (m.messageType.equals("KeyCollision")) {
		    // IMPORTANT: Save the key in the request.
		    final InsertRequest _req = (InsertRequest) req;
		    _req.clientKey = ClientUtil.getKeyFromURI(new FreenetURI(m.fs.get("URI")));
		    req.produceEvent(new CollisionEvent( _req.clientKey));
		}
                else
                    req.produceEvent(new ErrorEvent("Invalid message type " + m.messageType ));

                state(Request.FAILED);
            }
            close();
            return;
        }
    
            public void insert(String uri,int htl,long dlen, InputStream data)throws IOException {
            insert(uri,htl,0,(InputStream) null,dlen,data);
            }
    }


    protected class FCPRequest extends FCPInstance {
        
        public FCPRequest(Request req){
            super(req);
        }

        public void doit() {
            try{
                state(Request.REQUESTING);

                request((((DataRequest) req).uri).toString(),
                        ((DataRequest) req).htl,
                        (((DataRequest) req).meta).getOutputStream(),
                        (((DataRequest) req).data).getOutputStream());
                
            } catch(Exception e){
                try{
                    close();
                }catch(IOException ioe){}
                
                req.produceEvent(new ExceptionEvent(e));

                state(Request.FAILED); 
            }
        }

        public void request(String uri, int htl,OutputStream meta, OutputStream data) throws IOException{
                            
            FieldSet fs = new FieldSet();

            // create & send ClientGet-message
            fs.put("URI", uri);
            fs.put("HopsToLive",Integer.toHexString(htl)); 
            send("ClientGet",fs);

            // check for DataFound or some kind of error message
            FNPRawMessage m = getMessage();
            
            if (m.messageType.equals("DataFound")){
            
                int status = 0;
                
                long dlen = m.trailingFieldLength;
                long mlen = Long.parseLong(m.fs.get("MetadataLength"),16);
                long q[] = {mlen, dlen-mlen, dlen};
                EventInputStream input = new EventInputStream(in, dlen/20);
                    ((ClientEventProducer) input).addEventListener(this);

                req.produceEvent(new TransferStartedEvent(q));
                
                // read metadata
                if( mlen != 0){
                    read_chunks(mlen,input,meta);
                }
		
		// Let client know meta dat is done.
		if (state() != Request.FAILED) {
		    req.produceEvent(new SegmentCompleteEvent(((DataRequest)req).meta));
		}

                // read data
                if( state() == Request.REQUESTING   && (dlen - mlen) > 0){
                    read_chunks(dlen - mlen, input, data);
                }
		
                if(state() != Request.FAILED){
		    // Let client know data is done
		    req.produceEvent(new SegmentCompleteEvent(((DataRequest)req).data));
                    req.produceEvent(new TransferCompletedEvent(dlen));
		    // TODO FIX FIXME
		    // REDFLAG: This should eventually be removed because it has been superceded by
		    //          TransferCompletedEvent but doing so
		    //          breaks old FNP clients.
		    req.produceEvent(new RequestCompleteEvent(null)); 
		    // COPIED from FNPClient -----------------^
                    state(Request.DONE);
                }

            }else { // error handling 
                if (m.messageType.equals("FormatError"))
                    req.produceEvent(new ErrorEvent("FormatError from ClientGet"));
                else if (m.messageType.equals("URIError"))
                    req.produceEvent(new ErrorEvent("Peer complains about URIError:"+uri));
                else if (m.messageType.equals("RouteNotFound")) { 
                    req.produceEvent(new RouteNotFoundEvent());
		    // TODO FIX FIXME
		    // REDFLAG: This should eventually be removed because it has been superceded by
		    //          RouteNotFoundEvent but doing so breaks old FNP clients.
		    req.produceEvent(
				     new RequestFailedEvent(ClientUtil.getKeyFromURI(((DataRequest)req).uri)
					 , null));
		}
                else if (m.messageType.equals("DataNotFound")) {
                    req.produceEvent(new DataNotFoundEvent());
		    // TODO FIX FIXME
		    // REDFLAG: This should eventually be removed because it has been superceded by
		    //          DataNotFoundEvent but doing so breaks old FNP clients.
		    req.produceEvent(
				     new RequestFailedEvent(ClientUtil.getKeyFromURI(((DataRequest)req).uri)
					 , null));

		}
                else if (m.messageType.equals("Failed"))
                    req.produceEvent(new ErrorEvent("Some kind of error on server-side, received a Failed  message"));
                else
                    req.produceEvent(new ErrorEvent("Invalid message type " + m.messageType));
                state(Request.FAILED); 
            }

            close();
            return;
        }

        public void request(String uri, int htl, OutputStream data) throws IOException {
            request(uri,htl,null,data);
        }

        private void read_chunks(long len, InputStream input, OutputStream output) throws IOException{
        
            byte buf[] = new byte[2048]; 
            long chunk_len = 0;
            FNPRawMessage m = null; 
            long len_accu = len;
            
            ReadInputStream reader = new ReadInputStream(input);
        
            while( len > 0 ){
                
                try{
                    m = new FNPRawMessage(input);
                } catch (Exception e){
                    req.produceEvent(new ErrorEvent("Invalid message type " + m.messageType ));
                    state(Request.FAILED); 
                    return;
                }
            
                if (m.messageType.equals("DataChunk")){
                    
                    chunk_len = Long.parseLong(m.fs.get("Length"),16);
                
                    if(chunk_len > buf.length){
                        buf = new byte[(int)chunk_len];
                    }
                
                }else{
                    
                    if(m.messageType.equals("Failed")){
                        req.produceEvent(new ErrorEvent("Some kind of error on server-side, received a Failed  message"));
                    }else{
                        req.produceEvent(new ErrorEvent("Invalid message type " + m.messageType ));
                    }
                    
                    state(Request.FAILED); 
                    return;
                }
            
                input.read(buf, 0, (int) chunk_len);
                
                if(output != null){
                    output.write(buf);
                }
                
                len = len - chunk_len;
            }

            return;
        }
        
    }
    
}





