package Freenet.client;

import Freenet.*;
import Freenet.support.Bucket;
import Freenet.support.io.*;
import Freenet.message.*;
import Freenet.client.events.*;
import Freenet.crypt.*;
import java.io.*;

/** FNP version of Client library
  * @author tavin
  */
public class FNPClient implements ClientFactory {

    /** The version of the client library **/
    private static final String clientVersion = "0.3.9.2";
    public static String getVersion() {
        return clientVersion;
    }

    protected ClientCore core;
    protected Address target;

    /** Create a new ClientFactory for spawning Clients to process requests.
      * @param core    A running ClientCore object that provides Freenet services.
      * @param target  The node to send messages to. Ideally this should be a
      *                node running locally.
      */    
    public FNPClient(ClientCore core, Address target) {
        this.core   = core;
        this.target = target;
    }

    public Client obtainClient(Request req)
        throws UnsupportedRequestException, IOException, KeyException
    {
        if (req instanceof DataRequest) {
            BRequestInstance b = new BRequestInstance((DataRequest) req);
            (new Thread(b, "Request: " + ((DataRequest) req).uri)).start();
            return b;
        }
        else if (req instanceof InsertRequest) {
            BInsertInstance b = new BInsertInstance((InsertRequest) req);
            (new Thread(b, "Insert: " + ((InsertRequest) req).uri)).start();
            return b;
        }
        else if (req instanceof ComputeCHKRequest) {
            BComputeCHKInstance b = new BComputeCHKInstance((ComputeCHKRequest) req);
            (new Thread(b, "ComputeCHK")).start();
            return b;
        }
        else if (req instanceof ComputeSVKRequest) {
            BComputeSVKInstance b = new BComputeSVKInstance((ComputeSVKRequest) req);
            (new Thread(b, "ComputeSVK")).start();
            return b;
        }
        else throw new UnsupportedRequestException();
    }

    private static InputStream joinStreams(InputStream in1, InputStream in2)
                               throws IOException {
        if (in1 == null && in2 == null)
            return new NullInputStream();
        else if (in1 == null)
            return in2;
        else if (in2 == null)
            return in1;
        else
            return new SequenceInputStream(in1, in2);
    }
    
    /** BInstance, BRequestInstance, and BInsertInstance implement Client
      * and do the grunt work of executing a request.  They handle all the
      * necessary negotiation with the Freenet node, and will
      * generate periodic events for any ClientEventListeners
      * registered with the Request object.
      *
      * At the end of a request, if it was sucessful, a 
      * TransferCompletedEvent will be generated, indicating the
      * end of a transfer.
      *
      * For Requests, a RequestCompletedEvent will be generated
      * following the TransferCompletedEvent that will contain
      * file descriptors of the data and metadata, if present.
      */
    protected abstract class BInstance
        implements Client, Runnable, ClientEventListener {

        protected boolean ready = false;
        protected ConnectionHandler current;
        protected long id;
        protected int initDepth;
        protected Key key;
        protected String cipherName;
        protected Request req;
        protected ClientKey ckey;

        protected BInstance(Request req) {
            this.req  = req;
        }

        public void state(int state) {
            if (state >= Request.FAILED && state <= Request.DONE) {
                req.state = state;
            }
            req.produceEvent(new StateReachedEvent(state));
        }

        protected void sendMessage(Message m) throws SendFailedException {
            sendMessage(m, null);
        }

        protected void sendMessage(Message m, ClientMessageObject mo)
            throws SendFailedException {
            try {
                if (current == null || !current.isOpen()) {
                    current = core.makeConnection(target);
                }
                m.sending(core,current); 
                current.sendMessage(m,mo);
            } catch (ConnectFailedException e) {
                throw new SendFailedException(target);
            }
            req.produceEvent(new SendEvent(target,m,""));
        }

        protected ClientMessageObject getNextReply(long id, long millis) 
            throws InterruptedException {
            ClientMessageObject m = core.cmh.getNextReply(id,millis);
            req.produceEvent(new ReceiveEvent(m instanceof Message
                                              ? target : null, m, ""));
            return m;
        }

        protected ClientMessageObject getQueryResponse(long id, long millis) {
            boolean b;
            ClientMessageObject m;
            try {
                do {
                    m = getNextReply(id,millis);
                    if (b = (m != null && m instanceof QueryRestarted)) {
                        req.produceEvent(new RestartedEvent(millis));
                    }
                } while (m != null && b);
            } catch (InterruptedException e) {
                m = null;
            }
            return m;
        }

        protected synchronized void prepare() throws Exception {
            // set message id and depth
            do {
                id = Math.abs(Core.randSource.nextLong());
            } while (id <= 0x10000);
            initDepth = Core.randSource.nextInt() & 32 + 4;
            current = core.makeConnection(target);
            state(Request.PREPARED);
            while (!ready) {
                this.wait();
            }
            doit();
        }

        public final synchronized void execute() {
            ready = true;
            if (req.state == Request.PREPARED)
                this.notify();
        }

        protected abstract void doit();

        public final void run() {
            try {
                prepare();
            } catch (Exception e) {
                e.printStackTrace();
                req.produceEvent(new ExceptionEvent(e));
                synchronized(this) {
                    state(Request.FAILED);
                    this.notifyAll();
                }
            }
        }

        public void receive(ClientEvent ce) {
            req.produceEvent(ce);
        }
    }

    protected class BComputeCHKInstance extends BInstance {

        public BComputeCHKInstance(ComputeCHKRequest req) throws IOException {
            super(req);
            ckey = new ClientCHK(
                joinStreams(req.meta.getInputStream(), req.data.getInputStream()),
                req.meta.size()+req.data.size()
            );
        }
        
        protected synchronized void prepare() throws Exception {
            ComputeCHKRequest _req = (ComputeCHKRequest) req;
            BlockCipher cipher = Util.getCipherByName(_req.cipherName);
            Document doc = new Document(
                _req.data.getInputStream(), _req.meta.getInputStream(),
                ckey.getEncryptionKey(cipher.getKeySize()>>3), 
                _req.data.size(), _req.meta.size()
            );
            // encrypt and calculate hash/signature
            ((ClientCHK) ckey).docToStream(doc, cipher, _req.ctBucket);
            // set ClientKey in Request object
            _req.clientKey = (ClientCHK) ckey;
            state(Request.PREPARED);
            while (!ready) this.wait();
            doit();
        }

        protected synchronized void doit() {
            state(Request.DONE);
        }
    }

    protected class BComputeSVKInstance extends BInstance {

        public BComputeSVKInstance(ComputeSVKRequest req) {
            super(req);
            ckey = new ClientSVK(core.randSource);
        }
        
        protected synchronized void prepare() throws Exception {
            ((ClientSVK) ckey).getKey();
            ((ComputeSVKRequest) req).clientKey = (ClientSVK) ckey;
            state(Request.PREPARED);
            while (!ready) this.wait();
            doit();
        }

        protected synchronized void doit() {
            state(Request.DONE);
        }
    }
    
    protected class BRequestInstance extends BInstance {

        private VerifyingInputStream vin;
        private FieldSet storables;

        public BRequestInstance(DataRequest req) {
            super(req);
            ckey = ClientUtil.getKeyFromURI(req.uri);
        }

        public void prepare() throws Exception {
            key = ckey.getKey();
            super.prepare();
        }

        public void doit() {
            try {
                state(Request.REQUESTING);
                Freenet.message.DataRequest dr =
                    new Freenet.message.DataRequest(id, ((DataRequest) req).htl, initDepth, key);
                sendMessage(dr);
                getResponse();
                if (req.state != Request.FAILED) {
                    ClientMessageObject m = getNextReply(id,5000);
                    if (! (m instanceof StoreData)) 
                        throw new Exception("Unexpected message: "+m);
                    state(Request.DONE);
                }
            } catch (Exception e) {
                //              e.printStackTrace();
                if (current != null)
                    current.forceClose();
                req.produceEvent(new ExceptionEvent(e));
                state(Request.FAILED);
            }
        }

        /* jumping so we can reenter here */
        public void getResponse() throws Exception {
            ClientMessageObject m = 
                getQueryResponse(id, RequestRestarted.getTime(((DataRequest) req).htl) + 
                                 Core.hopTimeExpected * 2); // add slight bonus for slow local nodes.
            if (m instanceof DataReply) {
                DataReply reply = (DataReply) m;
                storables=(reply.otherFields.isSet("Storable") ?
                           reply.otherFields.getSet("Storable") :
                           new FieldSet());
                InputStream fromConn = reply.in;
                vin = key.verifyStream(reply.in, 
                                       storables, reply.length);
                vin.stripControls(true);

                cipherName = storables.get("Symmetric-cipher")==null ? "Twofish" : storables.get("Symmetric-cipher");
                BlockCipher cipher = Util.getCipherByName(cipherName);
                byte[] cryptKey = ckey.getEncryptionKey(cipher.getKeySize()>>3);
                String mdl=storables.get("Metadata-length");
                long metaLength = (mdl == null ? 0 : Long.parseLong(mdl));
                state(Request.TRANSFERRING);
                long plainLen=ckey.getPlainLength(reply.length, storables);
                try {
                    Document d=Document.read(vin, cryptKey, cipher,
                                             plainLen - 2 - cryptKey.length 
                                             - metaLength, metaLength);
                    beginTransfer(d, storables);
                } catch (DataNotValidIOException dnv) {
                    dnv.printStackTrace();
                    int code=dnv.getCode();
                    synchronized(fromConn) {
                        fromConn.notify();
                    }
                    if (code == Presentation.CB_RESTARTED) {
                        state(Request.REQUESTING);
                        getResponse();
                        return;
                    } else throw dnv;
                }
                synchronized(fromConn) {
                    fromConn.notify();
                }
                req.produceEvent(new RequestCompleteEvent(null));
            } else {
                if (current != null)
                    current.forceClose();
                if (m == null)
                    req.produceEvent(new NoReplyEvent());
                else if (m instanceof Message)
                    req.produceEvent(new RequestFailedEvent(ckey, (Message) m));
                else
                    req.produceEvent(new ErrorEvent(
                        "Got an unexpected internal MessageObject: " + m ));
                state(Request.FAILED);
            }
        }

        public void beginTransfer(Document d, FieldSet storables) 
            throws IOException, DataNotValidIOException {
            long[] lens=new long[] {d.metadataLength(),
                                    d.dataLength(),
                                    d.length()};
            req.produceEvent(new TransferStartedEvent(lens));
            EventInputStream in=
                new EventInputStream(//new BufferedInputStream(d.data//),
                                     d.data,
                                     d.length() / 20);
            ((ClientEventProducer) in).addEventListener(this);
            
            BufferedOutputStream out = null;

            try {               
                if (d.metadataLength() > 0) {
                    out = new BufferedOutputStream(
                        ((DataRequest) req).meta.getOutputStream() );
                    readData(in, out, d.metadataLength());
                    out.close();
                }
                req.produceEvent(new SegmentCompleteEvent(((DataRequest) req).meta));
                out = new BufferedOutputStream(((DataRequest) req).data.getOutputStream());
                readData(in, out, d.dataLength());
                out.close();
                req.produceEvent(new SegmentCompleteEvent(((DataRequest) req).data));
            } catch (IOException dnv) {
                dnv.printStackTrace();
                ((DataRequest) req).data.resetWrite();
                ((DataRequest) req).meta.resetWrite();
                throw dnv;
            } finally {
                try { out.close(); } catch (Throwable e) {}
            }
        }

        protected void readData(InputStream in, OutputStream out, long len)
            throws IOException, DataNotValidIOException {
            byte[] buffer=new byte[Core.bufferSize];

            int rc=0;
            do {
                rc=in.read(buffer, 0, (int)Math.min(len, buffer.length));
                    
                if (rc>0) {
                    len-=rc;
                    out.write(buffer, 0, rc);
                }
            } while (rc!=-1 && len!=0);
         }
    }

    protected class BInsertInstance extends BInstance {

        private FieldSet storables;
        private InputStream processedInput;
        
        public BInsertInstance(InsertRequest req) throws IOException, KeyException {
            super(req);
            if (req.uri.getKeyType().equals("CHK"))
                ckey = new ClientCHK(
                    joinStreams(req.meta.getInputStream(), req.data.getInputStream()),
                    req.meta.size()+req.data.size()
                );
            else if (req.uri.getKeyType().equals("KSK"))
                ckey = new ClientKSK(core.randSource, req.uri.getGuessableKey());
            else if (req.uri.getKeyType().equals("SVK"))
                ckey = new ClientSVK(core.randSource);
            else if (req.uri.getKeyType().equals("SSK"))
                ckey = new ClientSSK(core.randSource, req.uri);
            else
                throw new KeyException("Unsupported keytype in URI");
            storables = new FieldSet();
            storables.add("Symmetric-cipher", req.cipherName);
            storables.add("Metadata-length", ""+req.meta.size());
        }
        
        protected synchronized void prepare() throws Exception {
            InsertRequest _req = (InsertRequest) req;
            BlockCipher cipher = Util.getCipherByName(_req.cipherName);
            Document doc = new Document(
                _req.data.getInputStream(), _req.meta.getInputStream(),
                ckey.getEncryptionKey(cipher.getKeySize()>>3), 
                _req.data.size(), _req.meta.size());
            // encrypt and calculate hash/signature
            processedInput = new EventInputStream(ckey.docToStream(doc, cipher, _req.ctBucket),
                                                  ckey.getDataLength() / 20);
            ((ClientEventProducer) processedInput).addEventListener(this);
            // set key and storables, set ClientKey in Request object
            key = ckey.getKey(storables);
            _req.clientKey = ckey;
            super.prepare();
        }

        protected synchronized void doit() {
            try {
                Freenet.message.InsertRequest ir
                    = new Freenet.message.InsertRequest(id, ((InsertRequest) req).htl, initDepth, key);
                sendMessage(ir);
                state(Request.REQUESTING);
                ClientMessageObject reply = 
                    getQueryResponse(id, RequestRestarted.getTime(((InsertRequest) req).htl) + 
                                     Core.hopTimeExpected * 2); // add slight bonus for slow local nodes.

                // if a new connection was started, I'll go over to that
                if (reply instanceof Message && 
                    ((Message)reply).receivedWith != current) {

                    if (current != null && current.isOpen())
                        current.close();
                    current = ((Message) reply).receivedWith;
                }

                if (reply instanceof InsertReply) {
                    FieldSet root=new FieldSet();
                    root.add("Storable",storables);
                    DataInsert di = new DataInsert(id, ((InsertRequest) req).htl, initDepth, root,
                                                   processedInput);
                    di.length = ckey.getDataLength();
                    SentInsert si = new SentInsert(id);
                    state(Request.TRANSFERRING);
                    sendMessage(di,si);
                    req.produceEvent(new TransferStartedEvent(di.length));
                    ClientMessageObject sent;
                    do {
                        sent = getNextReply(id, 0);
                    } while (sent != si);
                    
                    if (si.isSuccess()) {
                        StoreData sd = new StoreData(id,((InsertRequest) req).htl,initDepth,null);
                        try {
                            sendMessage(sd);
                            state(Request.DONE);
                        } catch (SendFailedException e) {
                            req.produceEvent(new ExceptionEvent(e));
                            state(Request.FAILED);
                        }
                    } else {
                        req.produceEvent(new ExceptionEvent(si.getException()));
                        state(Request.FAILED);
                    }
                    try {
                        processedInput.close();
                    } catch (IOException e) {
                    }
                } else {
                    if (current != null)
                        current.forceClose();
                    if (reply instanceof DataReply ||
                           reply instanceof TimedOut) {
                        req.produceEvent(new CollisionEvent(ckey));
                    } else if (reply instanceof RequestFailed) {
                        req.produceEvent(new RequestFailedEvent(ckey, 
                                                            (Message)reply));
                    } else {
                        req.produceEvent(new NoReplyEvent());
                        
                    }
                    state(Request.FAILED);
                }
            } catch (Exception e) {
                req.produceEvent(new ExceptionEvent(e));
                if (current != null)
                    current.forceClose();
                state(Request.FAILED);
            }
        }
    }

    private class SentInsert implements ClientMessageObject {
        Exception e;
        long id;
        public SentInsert(long id) {
            this.id = id;
        }
        public long id() {
            return id;
        }
        public void setException(Exception e) {
            this.e = e;
        }
        public Exception getException() {
            return e;
        }
        public boolean isSuccess() {
            return e == null;
        }
    }
}







