package Freenet.client;

import java.net.*;
import Freenet.crypt.Util;
import Freenet.keys.KSK;

/**
 * FreenetURI handles parsing and creation of the Freenet URI format, defined 
 * as follows:
 *
 * freenet:[KeyType@]KeyValue[+DecryptionKey]
 *
 * where
 * KeyType is the TLA of the key (currently SVK, KSK, or CHK).
 *   If ommited, KeyType defaults to KSK
 *
 * KeyValue is, for SVKs and CHKs, the modified Base64 encoded
 * key value.  For KSKs, it is the string keyword.
 *
 * DecryptionKey is the modified Base64 encoded decryption key.  This
 * is only valid for SVKs and CHKs
 */
public class FreenetURI {
    private String keyType, guessableName;
    private byte[] keyVal, cryptoKeyVal;

    public FreenetURI(String keyType, String guessableName,
		      byte[] keyVal, byte[] cryptoKey) {
	this.keyType=keyType.trim().toUpperCase();
	this.guessableName=guessableName;
	this.keyVal=keyVal;
	this.cryptoKeyVal=cryptoKey;
    }

    public FreenetURI(String URI) throws MalformedURLException {
	URI=urlDecode(URI);
	int colon = URI.indexOf(':');
	int amp = URI.indexOf('@');
	if (colon != -1 &&
	    !URI.substring(0, colon).equalsIgnoreCase("freenet"))
	    throw new MalformedURLException("Invalid protocol for Freenet URI");
	if (amp == -1) {
	    keyType="KSK";
	    amp=colon;
	} else keyType=URI.substring(colon+1, amp).toUpperCase().trim();

	parseKey(URI.substring(amp==-1 ? Math.max(colon+1, 0) : amp+1));
    }

    protected void parseKey(String rest) throws MalformedURLException {
	try {
	    if ("KSK".equals(keyType)) {
	        guessableName=rest;
	    } else if ("SVK".equals(keyType) ||
		       "CHK".equals(keyType)) {
		int cryptDelim=rest.indexOf(',');
		if (cryptDelim==-1) return;

		keyVal=Base64.decode(rest.substring(0,cryptDelim));
		cryptoKeyVal=Base64.decode(rest.substring(cryptDelim+1));
	    } else if ("SSK".equals(keyType)) {
		int cryptDelim=rest.indexOf(',');
		int ssDelim=rest.indexOf('/');

		if (cryptDelim!=-1) {
		    cryptoKeyVal=Base64.decode(rest.substring(cryptDelim+1,ssDelim));
		} else 
		    cryptDelim=ssDelim;

		if (ssDelim<1)
		    throw new MalformedURLException("Malformed SSK");

		guessableName=rest.substring(ssDelim+1);
		keyVal=Base64.decode(rest.substring(0, cryptDelim));
	    } else {
		throw new MalformedURLException("Invalid or unknown keytype");
	    }
	} catch (IllegalBase64Exception i) {
	    throw new MalformedURLException("Invalid Base64 quantity: "+i);
	}
    }

    protected void decompose() {
	System.out.println("Keytype: "+ keyType);
	System.out.println("Keyval : "+ 
			   (keyVal == null ? "none" :
			    Freenet.support.Fields
			    .bytesToHex((byte[])keyVal, 0,
					((byte[])keyVal).length)));
	System.out.println("Dc. Key: "+ (cryptoKeyVal == null ? "none" :
					Freenet.support.Fields
					 .bytesToHex(cryptoKeyVal, 0,
						     cryptoKeyVal.length)));
	System.out.println("Guess  : " + (guessableName == null ? "none" :
					  guessableName));
    }

    public String getGuessableKey() {
	return guessableName;
    }

    public byte[] getKeyVal() {
	return (byte[])keyVal;
    }
    
    public byte[] getCryptoKey() {
	return (byte[])cryptoKeyVal;
    }

    public String getKeyType() {
	return keyType;
    }

    protected static String urlDecode(String s) {
	StringBuffer b=new StringBuffer();
	for (int i=0; i<s.length(); i++) {
	    if (s.charAt(i)=='+')
		b.append(' ');
	    else if (s.charAt(i)=='%') {
		int n=Integer.parseInt(s.substring(i+1, i+3), 16);
		b.append((char)n);
		i+=2;
	    } else 
		b.append(s.charAt(i));
	}
	return b.toString();
    }

    protected static String urlEncode(String s) {
	StringBuffer b=new StringBuffer();
	for (int i=0; i<s.length(); i++) {
	    if (s.charAt(i)==' ')
		b.append('+');
	    else if (s.charAt(i)>128 || s.charAt(i)<44) {
		b.append('%').append(Integer.toString(s.charAt(i), 16));
	    } else 
		b.append(s.charAt(i));
	}
	return b.toString();
    }

    public String toURI() {
	return toString(true);
    }
    
    public String toString() {
	return toString(false);
    }

    public String toString(boolean asURI) {
	StringBuffer b=new StringBuffer("freenet:");
	b.append(keyType);
	if (asURI) b.append("%40");
	else b.append('@');
	if ("KSK".equals(keyType)) {
	    b.append(urlEncode(guessableName));
	} else if ("SSK".equals(keyType)) {
	    b.append(Base64.encode(keyVal));
	    if (cryptoKeyVal!=null) 
		b.append(',').append(Base64.encode(cryptoKeyVal));
	    b.append('/').append(urlEncode(guessableName));
	} else {
	    if (keyVal!=null) {
		b.append(Base64.encode((byte[])keyVal));
		if (cryptoKeyVal!=null)
		    b.append(',').append(Base64.encode(cryptoKeyVal));
	    }
	    if (guessableName!=null) 
		b.append(guessableName);
	}
	return b.toString();
    }

    public static void main(String[] args) throws Exception {
	FreenetURI r=new FreenetURI(args[0]);
	r.decompose();
	System.err.println(r);
	/*
	byte[] b=new byte[2];
	for (int i=0; i<65536; i++) {
	    b[0]=(byte)(i/256);
	    b[1]=(byte)(i%256);
	    System.err.println(Freenet.support.Fields.bytesToHex(b,0,b.length));
	    String m=Base64.encode(b);
	    System.err.println(m);
	    byte[] b2=Base64.decode(m);
	    if (b[0]!=b2[0] || b[1]!=b2[1]) {
		System.err.println("ERROR");
		break;
	    }
	}
		*/
    }
}
	
