package Freenet.client;

import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.client.events.*;
import Freenet.client.listeners.*;
import Freenet.crypt.*;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import Freenet.keys.*;

/**
 * Simple CLI for the Client library
 *
 * @author oskar
 **/
public class InsertClient extends CLI {

    protected ClientFactory bcf;

    public static void main(String[] args) {
        Params params = new Params();
	BlockingQueue nextArgs = new BlockingQueue();
	for (int i=0; i < args.length; i++) {
	    String curArg = args[i];
	    if (curArg.equals("--help")) {
		usage();
	    } else if (curArg.equals("--length")) { 
		nextArgs.enqueue("length");	
 	    } else if (curArg.equals("-l") || curArg.equals("--length")) {
 	    } else if (curArg.equals("-m") || curArg.equals("--metadata")) {
		nextArgs.enqueue("metadata");
 	    } else if (curArg.equals("-M") || curArg.equals("--mimeType")) {
		nextArgs.enqueue("mimeType");
 	    } else if (curArg.equals("-K") || curArg.equals("--makeKeypair")) {
		params.setParam("makeKeypair", "yes");
 	    } else if (curArg.equals("-r") || curArg.equals("--redirect")) {
		nextArgs.enqueue("redirect");
	    } else if (curArg.equals("-a") || curArg.equals("--autoRedirect")) {
		params.setParam("autoRedirect", "yes");
	    } else if (curArg.equals("-A") || curArg.equals("--noAutoRedirect")) {
 	    } else if (curArg.equals("--cipher")) {
		nextArgs.enqueue("cipher");
 	    } else if (curArg.equals("-p") || curArg.equals("--listenPort")) {
		nextArgs.enqueue("listenPort");
	    } else if (curArg.equals("-f") || curArg.equals("--safer")) {
		params.setParam("safer", "yes");
	    } else if (curArg.equals("-F") || curArg.equals("--noSafer")) {
		params.setParam("safer", "no");
	    } else if (curArg.equals("-c") || curArg.equals("--createUpdate")) {
		params.setParam("createUpdate", "yes");
	    } else if (curArg.equals("-C") || curArg.equals("--noCreateUpdate")) {
		params.setParam("createUpdate", "no");
	    } else if (curArg.equals("-u") || curArg.equals("--update")) {
		params.setParam("update", "yes");
	    } else if (curArg.equals("-U") || curArg.equals("--noUpdate")) {
		params.setParam("update", "no");
 	    } else if (curArg.equals("-i") || curArg.equals("--increment")) {
		nextArgs.enqueue("increment");
 	    } else if (curArg.equals("-b") || curArg.equals("--baseline")) {
		nextArgs.enqueue("baseline");
 	    } else if (curArg.equals("--future")) {
		if (params.getParam("past") != null) {
		    usage();
		    CLI.waitForThreadsAndExit();
		} else {
		    nextArgs.enqueue("future");
		}
 	    } else if (curArg.equals("--past")) {
		if (params.getParam("future") != null) {
		    usage();
		    CLI.waitForThreadsAndExit();
		} else {
		    nextArgs.enqueue("past");
		}
 	    } else if (curArg.equals("-s") || curArg.equals("--serverAddress")) {
		nextArgs.enqueue("serverAddress");
 	    } else if (curArg.equals("-h") || curArg.equals("--htl")) {
		nextArgs.enqueue("htl");
 	    } else if (curArg.equals("-g") || curArg.equals("--logging")) {
		nextArgs.enqueue("logging");
 	    } else if (curArg.equals("-v") || curArg.equals("--verbosity")) {
		nextArgs.enqueue("verbosity");
 	    } else if (curArg.equals("-P") || curArg.equals("--useFCP")) {
		params.setParam("useFCP", "yes");
	    } else if (curArg.equals("-k") || curArg.equals("--keyindex")) {
		nextArgs.enqueue("keyindex");
	    } else if (curArg.equals("-D") || curArg.equals("--debugSet")) {
		String variable = args[++i];
		nextArgs.enqueue(variable);
	    } else if (curArg.startsWith("-")) {
		System.out.println("Unknown argument: " + curArg);
		usage();
		CLI.waitForThreadsAndExit();
	    } else {
		if (! nextArgs.isEmpty()) {  
		    //it's associated with an argument
		    try {
			String argument = (String) nextArgs.dequeue();
			params.setParam ( argument, curArg);
		    } catch (InterruptedException e) {
			System.err.println("Some fatal error occurred with the queue processing in getting parameter:" + curArg);
		    }
		} else { //it's not with an option
		    params.addArg(args[i]);
		}
	    }
	}
	//        Params params = new Params(args);
        CLI.exitState = 1;
        try {
            InsertClient client = new InsertClient(params);
            if (params.getParam("makeKeypair") != null) {
                client.printKeypair();
            } else if (params.getNumArgs() < 1) {
                client.usage();
                CLI.exitState = 1;
            } else {
                boolean controlDoc=false;
                String uristring = params.getArg(0);
                FreenetURI uri = new FreenetURI(uristring);
                String file = params.getNumArgs() > 1 ? params.getArg(1) :
                    null;
                String metaDataFile = params.getParam("metadata");
                String mimeType = params.getParam("mimeType", "auto");
                String redir = params.getParam("redirect");
                boolean autoRedirect = params.getboolean("autoRedirect", true);
                boolean createUpdate = params.getboolean("createUpdate", false);
                
                String baseline = params.getParam("baseline");
                long increment = params.getlong("increment", 0);
                long offset = params.getlong("future", - params.getlong("past", 0) );
                
                boolean update = params.getboolean("update", (baseline != null) || (increment != 0) || (offset != 0) );

                String keyIndex = params.getParam("keyindex");
                
                if (baseline == null) {
                    baseline = "20000101000000";
                }

                if (increment == 0) {
                    increment = 86400;
                }
                
                FileBucket mfb=null;
                
                if (metaDataFile == null) {
                    if (file != null && mimeType.equals("auto") && file.indexOf('.') >= 0) {
                        mfb = MimeBucket(file.substring(file.lastIndexOf('.')+1), true);
                        metaDataFile = mfb.getFile().getCanonicalPath();
                    } else if (!mimeType.equals("auto") && !mimeType.equals("none")) {
                        mfb = MimeBucket(mimeType, false);
                        metaDataFile = mfb.getFile().getCanonicalPath();
                    }
                }
                
                if (createUpdate) {
                    FieldSet fs = new FieldSet();
                    fs.add("baseline", baseline);
                    fs.add("increment", String.valueOf(increment) );
                 
                    File updateFile;

                    if (redir == null) {
                        updateFile = ClientUtil.makeRedirect(publicURI(uri), fs);
                    } else {
                        updateFile = ClientUtil.makeRedirect(new FreenetURI(redir), fs);
                    }

                    file = null;
                    metaDataFile=updateFile.getCanonicalPath();
                    controlDoc=true;
                } else {
                    if (update) {
                        uri = ClientUtil.dateURI(uri, baseline, increment, offset);
                    }
                
                    if (redir!=null) {
                        if (params.getNumArgs() > 1) {
                            client.usage();
                            CLI.exitState = 1;
                            return;
                        }
                        File redirFile=
                            ClientUtil.makeRedirect(new FreenetURI(redir));
                        metaDataFile=redirFile.getCanonicalPath();
                        controlDoc=true;
                    }
                }
            uristring=client.execute(uri, file, metaDataFile, controlDoc, autoRedirect).toString();

            if(keyIndex!=null)
              {
                KeyIndexClient kic=new KeyIndexClient(params, Core.logger);
                kic.insertKey(keyIndex, uristring);
              }
            }
        } catch (Freenet.client.rdf.RDFException e) {
            System.err.println(e.getMessage());
            CLI.exitState = 1;
        } catch (IllegalArgumentException e) {
            System.err.println(e.getMessage());
            CLI.exitState = 1;
        } catch (MalformedURLException e) {
            System.err.println("Could not parse URL");
            CLI.exitState = 1;
        } catch (FileNotFoundException e) {
            System.err.println("File not found");
            CLI.exitState = 1;
        } catch (IOException e) {
            System.err.println("Error while reading file: " + e.getMessage());
            CLI.exitState = 1;
        } catch (Exception e) {
            System.err.println("Generic exception: " + e.getMessage());
            CLI.exitState = 1;
        } finally {
            CLI.waitForThreadsAndExit();
        }
    }

    /**
     * Creates a new InsertClient object.
     * @param params Options
     **/
    public InsertClient(Params params) throws IllegalArgumentException {
        super(params);
        if (useFCP) {
	    System.err.println("");
	    System.err.println("WARNING: FCP support is experimental.  It might work.  It might not.");
	    System.err.println("         Using Server: " + target.toString());
	    System.err.println("");
	    bcf = new FCPClient(target);
	    
	}
	else {
	    bcf = new FNPClient(cc, target);
	}
    }

    /**
     * Executes an Insert without private metadata.
     * @param uri      The value of the URI to attempt to insert as. 
     *                 This can be an incomplete URI for some keytypes
     *                 (example: freenet:CHK@)
     * @param filename The file to read the data from
     **/ 
    public FreenetURI execute(FreenetURI uri, String filename) 
        throws MalformedURLException, FileNotFoundException, IOException {
        return execute(uri, filename, null, false, true);
    }

    /**
     * Executes an Insert. AutoRedirects non CHK values to a CHK unless
         * controlDoc is true.
     * @param uri      The value of the URI to attempt to insert as. 
     *                 This can be an incomplete URI for some keytypes
     *                 (example: freenet:CHK@)
     * @param filename The file to read the data from
     * @param metaFilename The file to read the private metadata from
     **/ 
    public FreenetURI execute(FreenetURI uri, String filename, String metaFilename,
                        boolean controlDoc)
        throws MalformedURLException, FileNotFoundException, IOException {
                return execute(uri, filename, metaFilename, controlDoc, true);
        }
        
        /**
     * Executes an Insert.
     * @param uri      The value of the URI to attempt to insert as. 
     *                 This can be an incomplete URI for some keytypes
     *                 (example: freenet:CHK@)
     * @param filename The file to read the data from
     * @param metaFilename The file to read the private metadata from
         * @param controlDoc The document to insert is a control document
         * @param autoRedirect Whether to auto-redirect. If controlDoc is true,'
         *                 the value of this is not used.
     **/ 
        public FreenetURI execute(FreenetURI uri, String filename, String metaFilename,
                        boolean controlDoc, boolean autoRedirect)
        throws MalformedURLException, FileNotFoundException, IOException {
        // args
        Bucket plaintext;
        Bucket metaPlaintext;
        if (filename != null) {
            File f = new File(filename);
	    if (!f.canRead()) throw new FileNotFoundException();
	    plaintext = new FileBucket(f);
	}
	else {
            plaintext = getBucket();
            OutputStream out = plaintext.getOutputStream();
            if (! controlDoc ) {
                byte[] buffer = new byte[Core.bufferSize];
                for (int i = System.in.read(buffer) ; (i != -1) ;
                     i = System.in.read(buffer)) {
                    out.write(buffer,0,i);
                }
            }
            out.close();
        }
        if (metaFilename != null) {
            // for now; later, should be able to give both metadata and data
            // at stdin
            metaPlaintext = new FileBucket(new File(metaFilename));
        }
        else {
            metaPlaintext = getBucket();
        }
        FreenetURI wantedURI = uri;
        if (controlDoc || !autoRedirect ||
                wantedURI.getKeyType().equals("CHK")) {
            return dumpBuckets(wantedURI, plaintext, metaPlaintext);
        }
        else {
            // do redirect from chk
            FreenetURI insertedURI= dumpBuckets("CHK@", plaintext, metaPlaintext);
            Bucket redirBucket = ClientUtil.makeRedirectBucket(insertedURI);
                
            return dumpBuckets(wantedURI, new FileBucket(), redirBucket);
        }
    }

    public FreenetURI dumpBuckets(String uri, Bucket plaintext, 
                                  Bucket metaPlaintext) 
            throws MalformedURLException, IOException {
        return dumpBuckets(new FreenetURI(uri), plaintext, metaPlaintext);
    }

    public FreenetURI dumpBuckets(FreenetURI keyName, Bucket plaintext,
                                  Bucket metaPlaintext)
            throws MalformedURLException, IOException {

        InsertRequest req;
        try {
            req = new InsertRequest(htl, keyName, cipherName,
                                    metaPlaintext, plaintext, getBucket());
        } catch (InsertSizeException e) {
            System.err.println("SVKs, KSKs, and SSKs cannot be more than"+Freenet.keys.SVK.SVK_MAXSIZE+" bytes.");
            System.err.println("");
            System.err.println("Instead, insert your file as a CHK, then insert a redirect from the");
            System.err.println("SVK/KSK/SSK to the resulting CHK.");
            CLI.exitState = 1;
            return keyName;
        }

        DoneListener dl            = new DoneListener();
        ClientCollisionListener cl = new ClientCollisionListener();
        ExceptionListener el       = new ExceptionListener();
        req.addEventListener(cl);
        req.addEventListener(dl);
        req.addEventListener(el);
        req.addEventListener(new EventLogger(Core.logger));
        
        synchronized(dl) {
            Client bc;
            try {
                bc = bcf.obtainClient(req);
            }
            catch (Exception e) {
                e.printStackTrace();
                return keyName;
            }
            // request
            bc.execute();
            // wait
            try {
                dl.waitEvent();
            } catch (InterruptedException e) {}
        }

        keyName = req.getURI();   // read final URI back out
               
        if (req.state() < Request.DONE) {
            Exception[] es = el.getExceptions();
            if (es == null) {
                System.err.println("Insert failed gracefully.");
                if(keyName.getKeyType().equals("CHK") && cl.collisionHappened()) {
                    System.err.println("An exact copy of the data you tried " 
                                       + "to insert was already present.");
                    exitState = 0;        //The doc is allready in freenet
                    return keyName;
                }
            } else {
                System.err.println("Encountered the following exceptions while trying to Insert:");
                for (int i = 0; i < es.length ; i++) {
                    es[i].printStackTrace(System.err);
                }
            }
            exitState = 1;
            return null;
        } else {
            System.err.println("Inserted Key   : " + keyName);
            if (keyName.getKeyType().equals("SVK")) {
                System.err.println("SVK private key: " + Base64.encode(req.getPrivateKey()));
                System.err.println("Hang on to this key if you wish to update it in the future, or if you wish to\ninsert subspace keys (SSKs) under this SVK root.");
            }
            exitState = 0;
            return keyName;
        }
    }

    /**
     * returns priv as a public-key if it is an SSK, otherwise returns priv.
     **/
    static FreenetURI publicURI(FreenetURI priv)
    {
        if (priv.getKeyType().equals("SSK")) {
            ClientSSK ck = new ClientSSK(null, priv);
            return ck.getURI();
        } else {
            return priv;
        }
    }

    /**
     * Returns a FileBucket containing mime-type metadata, deduced from a file extension
     * or explicitly specified.
     **/
    private static FileBucket MimeBucket(String mt, boolean isExtension) throws IOException {
        if (isExtension) mt = MimeTypeUtils.getMimeType(mt);
        if (mt == null) mt = "application/octet-stream";
        return MimeBucket(mt);
    }

    private static FileBucket MimeBucket(String mt) throws IOException {
        FileBucket fb = new FileBucket();
        PrintWriter pw = new PrintWriter(fb.getOutputStream());
        pw.println("Content-Type="+mt);
        pw.println("End");
        pw.close();
        return fb;
    }

    private void printKeypair() {
        ClientSVK csvk = new ClientSVK(cc.randSource);
        Key k = csvk.getKey();
        String privkey = csvk.getPrivateKeyString();
        String pubkey = csvk.toString().substring(12);
        System.out.println("\nPrivate key : "+privkey);
        System.out.println("Public key  : "+pubkey+"\n");
    }
    
    /**
     * Prints the usage of this freenet_insert to standard out.
     **/
    public static void usage() {
        System.out.println("Usage: freenet_insert URL [input-file]");
        System.out.println("");
        System.out.println("  -l|--length bytes              Bytes to insert (if reading stdin)");
        System.out.println("  -m|--metadata file             Attach private metadata to the document");
        System.out.println("  -M|--mimeType auto|type|none   Attach specified mime-type. Default auto.");
        System.out.println("  -K|--makeKeypair               Create and print a new pair of keys.");
        System.out.println("  -r|--redirect URI              Create a redirect from this key to another");
        System.out.println("                                 (cannot be used with [input-file]");
        System.out.println("  -a|--autoRedirect              Redirects URI to a CHK automatically if set.");
        System.out.println("                                 Using redirect negates this value.");
	System.out.println("  -A|--noAutoRedirect            Turns off auto-redirection.");
        System.out.println("     --cipher cipher-name        Use 'cipher-name' as the cipher");
        System.out.println("                                 Choices are 'Twofish' (default) or 'Rijndael'");
        System.out.println("  -p|--listenPort port           Local port to listen for replies on");
        System.out.println("  -f|--safer yes|no              Avoid unnecessary risk by encrypting all data");
        System.out.println("                                 written to disk. Slow and not perfect. Default off.");
        System.out.println("  -c|--createUpdate              Set up a new date update redirect.");
	System.out.println("  -C|--noCreateUpdate            Force no creation of update redirect. (Def)");
        System.out.println("  -u|--update                    Insert an update.");
	System.out.println("  -U|--noUpdate                  Force not inserting an update (Def)");
        System.out.println("  -i|--increment secs            Number of seconds between updates. Default 86400.");
        System.out.println("  -b|--baseline yyyymmddhhmmss   Baseline date for update. Default 20000101000000");
        System.out.println("  --[future|past] n              Insert nth update from now, future or past.");
        System.out.println("  -s|--serverAddress address     Server node to connect to");
        System.out.println("  -h|--htl hops-to-live          Hops-to-live for request");
        System.out.println("  -l|--logging error|normal      Logging level");
        System.out.println("               |minor|debugging");
        System.out.println("  -v|--verbosity 0-5             Verbosity of log messages");
        System.out.println("  -P|--useFCP                    Use FCP instead of FNP. EXPERIMENTAL!");
        System.out.println("  -k|--keyindex index            The in-freenet key index to add the key to");
        System.out.println("                                 (Ex. -keyindex main-index)");
        System.out.println("");
        System.out.println("Examples:");
        System.out.println("  freenet_insert test-key test.txt");
        System.out.println("  freenet_insert freenet:KSK@test-key test.txt");
        System.out.println("  freenet_insert freenet:CHK@ test.txt");
        System.out.println();
        System.out.println("Send bug reports to freenet-dev@lists.sourceforge.net");
    }


}
