package Freenet.client;

import Freenet.support.Bucket;
import java.io.InputStream;
import java.net.MalformedURLException;

/** Represents an InsertRequest
  * @author tavin
  */
public class InsertRequest extends Request {

    int htl;
    FreenetURI uri;
    String cipherName;
    Bucket meta, data, ctBucket;
    ClientKey clientKey;

    /** Prepares an insert of the supplied data.
      * @param htl         The HopsToLive to give the request.
      * @param uri         The URI to insert under.
      * @param meta        A stream of Freenet standard client metadata.
      * @param metaLength  The length of the metadata.
      * @param data        A stream of the data to insert.
      * @param dataLength  The length of the data.
      * @param cipherName  The block cipher to use.  null defaults to Twofish
      * @param ctBucket    A place to store the CipherText during processing.
      * @throws MalformedURLException  if the URI string is bad
      */
    public InsertRequest(int htl, String uri, String cipherName,
                         Bucket meta, Bucket data, Bucket ctBucket)
        throws MalformedURLException, InsertSizeException
    {
        this(htl, new FreenetURI(uri), cipherName,
             meta, data, ctBucket);
    }

    /** Prepares an insert of the supplied data.
      * @param htl         The HopsToLive to give the request.
      * @param uri         The URI to insert under.
      * @param meta        A stream of Freenet standard client metadata.
      * @param metaLength  The length of the metadata.
      * @param data        A stream of the data to insert.
      * @param dataLength  The length of the data.
      * @param cipherName  The block cipher to use.  null defaults to Twofish
      * @param ctBucket    A place to store the CipherText during processing.
      */
    public InsertRequest(int htl, FreenetURI uri, String cipherName,
                         Bucket meta, Bucket data, Bucket ctBucket)
        throws InsertSizeException
    {
        super();
        if (!uri.getKeyType().equals("CHK") &&
            meta.size() + data.size() > Freenet.keys.SVK.SVK_MAXSIZE)
            throw new InsertSizeException(
                "SVKs, KSKs, and SSKs cannot be more than 32,768 bytes."
            );
        this.htl        = htl;
        this.uri        = uri;
        this.cipherName = (cipherName == null ? "Twofish" : cipherName);
        this.meta       = meta;
        this.data       = data;
        this.ctBucket   = ctBucket;
    }

    /** Returns the URI -- useful if the URI was set to just
      * SVK@ or CHK@ at insert and you want to see what it became.
      * Only valid after a Client instance has been obtained for
      * the request.
      * @returns the correct FreenetURI or null if the request is incomplete
      */
    public FreenetURI getURI() {
        return clientKey == null ? null : clientKey.getURI();
    }

    /** Get the private key for use in constructing an insert URI.
      * (Just Base64.encode() the return value).
      * @return the raw bytes which form the private key if it's an SVK
      *         or null if the request is incomplete
      */
    public byte[] getPrivateKey() {
        if (clientKey == null) return null;
        return clientKey instanceof ClientSVK
            ? ((ClientSVK) clientKey).getPrivateKey()
            : null;
    }

    /** Get the public key.
      * @return the raw bytes which form the public key if it's an SVK
      *         or null if the request is incomplete
      */
    public byte[] getPublicKey() {
        if (clientKey == null) return null;
        return clientKey instanceof ClientSVK
            ? ((ClientSVK) clientKey).getPublicKey()
            : null;
    }

    /** Get the public key fingerprint for use in constructing a request URI.
      * (Just Base64.encode() the return value).
      * @return the raw bytes which form the public key fingerprint
      *         if it's an SVK or null if the request is incomplete
      */
    public byte[] getPublicKeyFingerPrint() {
        if (clientKey == null) return null;
        return clientKey instanceof ClientSVK
            ? ((ClientSVK) clientKey).getPublicKeyFingerPrint()
            : null;
    }
}

