package Freenet.client;

import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.ReadInputStream;
import Freenet.client.events.*;
import Freenet.client.listeners.*;
import java.io.*;
import java.net.MalformedURLException;

/**
 * Simple CLI for the Client library
 *
 * @author oskar
 **/

public class RequestClient extends CLI {
    
    protected ClientFactory bcf;
    
    protected boolean autoRedirect=true;
    protected static Params params;
    public static void main(String[] args) throws Exception {
        params = new Params();
	BlockingQueue nextArgs = new BlockingQueue();
	for (int i=0; i < args.length; i++) {
	    String curArg = args[i];
	    if (curArg.equals("--help")) {
		usage();
 	    } else if (curArg.equals("-m") || curArg.equals("--metadata")) {
		nextArgs.enqueue("metadata");
	    } else if (curArg.equals("-a") || curArg.equals("--autoRedirect")) {
		params.setParam("autoRedirect", "yes");
	    } else if (curArg.equals("-A") || curArg.equals("--noAutoRedirect")) {
		params.setParam("autoRedirect", "no");
 	    } else if (curArg.equals("-c") || curArg.equals("--cipher")) {
		nextArgs.enqueue("cipher");
 	    } else if (curArg.equals("-p") || curArg.equals("--listenPort")) {
		nextArgs.enqueue("listenPort");
	    } else if (curArg.equals("-f") || curArg.equals("--safer")) {
		params.setParam("safer", "yes");
	    } else if (curArg.equals("-F") || curArg.equals("--noSafer")) {
		params.setParam("safer", "no");
 	    } else if (curArg.equals("-s") || curArg.equals("--serverAddress")) {
		nextArgs.enqueue("serverAddress");
 	    } else if (curArg.equals("-h") || curArg.equals("--htl")) {
		nextArgs.enqueue("htl");
 	    } else if (curArg.equals("-l") || curArg.equals("--logging")) {
		nextArgs.enqueue("logging");
 	    } else if (curArg.equals("-v") || curArg.equals("--verbosity")) {
		nextArgs.enqueue("verbosity");
  	    } else if (curArg.equals("-P") || curArg.equals("--useFCP")) {
	      		params.setParam("useFCP", "yes");
	    } else if (curArg.equals("-D") || curArg.equals("--debugSet")) {
		String variable = args[++i];
		nextArgs.enqueue(variable);
	    } else if (curArg.startsWith("-")) {
		System.out.println("Unknown argument: " + curArg);
		usage();
		CLI.waitForThreadsAndExit();
	    } else {
		if (! nextArgs.isEmpty()) {  
		    //it's associated with an argument
		    try {
			String argument = (String) nextArgs.dequeue();
			params.setParam ( argument, curArg);
		    } catch (InterruptedException e) {
			System.err.println("Some fatal error occurred with the queue processing in getting parameter:" + curArg);
		    }
		} else { //it's not with an option
		    params.addArg(args[i]);
		}
	    }
	}
	// process uncompleted arguments, if any
	while (!nextArgs.isEmpty()) {
	    try {
		params.setParam ( (String) nextArgs.dequeue(), "");
	    } catch (InterruptedException e) {
		System.err.println("Some fatal error occurred with the queue processing after the last paremeter");
	    }
	}
	
	
	CLI.exitState = 1;
	try {
	    RequestClient client = new RequestClient(params);
	    
	    if (params.getNumArgs() < 1) {
		client.usage();
		CLI.exitState = 1;
	    } else {
		String uri = params.getArg(0);
		String metaDataFile = params.getParam("metadata");
		client.autoRedirect = params.getboolean("autoRedirect", true);
		if (params.getNumArgs() > 1)
		    client.execute(uri, params.getArg(1), metaDataFile);
		else 
		    client.execute(uri, null, metaDataFile);
	    }
        } catch (IllegalArgumentException e) {
            System.err.println(e.getMessage());
            CLI.exitState = 1;
        } catch (MalformedURLException e) {
            System.err.println("Could not parse the provided URL");
            CLI.exitState = 1;
        } finally {
            CLI.waitForThreadsAndExit();
        }
    }

    /**
     * Creates a new RequestClient object.
     * @param params Options
     **/
    public RequestClient(Params args) {
        super(args);
        if (useFCP) {
	    System.err.println("");
	    System.err.println("WARNING: FCP support is experimental.  It might work.  It might not.");
	    System.err.println("         Using Server: " + target.toString());
	    System.err.println("");
	    bcf = new FCPClient(target);
	    
	}
	else {
	    bcf = new FNPClient(cc, target);
	}
    }

    /**
     * Executes a Request, discarding any private metadata.
     * @param uri      The value of the URI to request.
     * @param filename The file to write the data to.
     **/ 
    public void execute(String uri, String filename) 
        throws MalformedURLException {
        execute(uri, filename, null);
    }


    /**
     * Executes a Request.
     * @param uri      The value of the URI to request.
     * @param filename The file to write the data to.
     * @param metaFilename The file to write the private metadata to.
     **/ 
    public void execute(String uri, String filename, String metaFilename)
        throws MalformedURLException {

        Bucket data = (filename != null ?
                       new FileBucket(new File(filename)) :
                       getBucket());
        Bucket metaData = (metaFilename != null ?
                           new FileBucket(new File(metaFilename)) :
                           getBucket());

        // prepare
        execute(uri, filename, metaFilename, data, metaData);
    }

    /**
     * Executes a Request.
     **/ 
    protected void execute(String uri, 
                           String filename, String metaFilename,
                           Bucket data, Bucket metaData)
        throws MalformedURLException {
        boolean success=false;
        try {
            success = fillBuckets(uri, data, metaData);
        } catch(IOException e) {
            System.err.println("Error reading stream.");
            e.printStackTrace();
            exitState=1;
            return;
        } catch(NoDataException e) {
            System.err.println("Error, finished but received no data.");
            exitState = 1;
            return;
        }

        if(!success) {
            return;
        } else {
            if (metaFilename == null && metaData.size() > 0) {
                try {
                    InputStreamReader in = 
                        new InputStreamReader(metaData.getInputStream());
                    char[] buffer = new char[Core.bufferSize >> 1];
                    System.out.println("Metadata:"); 
                    for (int n = 0, i = 0 ; n < metaData.size() ; ) {
                        i = in.read(buffer);
                        n += i;
                        System.out.print(new String(buffer,0,i));
                    }
                } catch (IOException e) {
                    System.err.println("Error, failed to read metadata" +
                                       " file");
                } 
            }
            if (filename == null && data.size() > 0) {
                try {
                    InputStream in = data.getInputStream();
                    byte[] buffer = new byte[Core.bufferSize];
                    System.out.println("Data:"); 
                        for (int n = 0 ; n < data.size() ; ) {
                            int i = in.read(buffer);
                            n += i;
                            System.out.write(buffer,0,i);
                        }
                } catch (IOException e) {
                    exitState = 1;
                    return;
                }
            } 
            exitState = 0;
        }
    }

    public boolean fillBuckets(String uri, Bucket data, Bucket metaData)
        throws MalformedURLException, IOException, NoDataException {

        // *** FIX ME! ***
        if (ClientUtil.isMSK(uri)) uri = ClientUtil.lookupMSK(uri, params);

        DataRequest req = new DataRequest(htl, uri, metaData, data);
        DoneListener dl      = new DoneListener();
        ExceptionListener el = new ExceptionListener();
        req.addEventListener(dl);
        req.addEventListener(el);
        req.addEventListener(new EventLogger(Core.logger));
        
        synchronized (dl) {
            // request
            bcf.obtainClient(req).execute();
            // wait
            try {
                dl.waitEvent();
            } catch (InterruptedException e) {}
        }

        if (req.state() < Request.DONE) {
            Exception[] es = el.getExceptions();
            if (es == null) {
                System.err.println("Request failed gracefully.");
            } else {
                System.err.println("Encountered the following exceptions while attempting to Request:");
                for (int i = 0; i < es.length ; i++) {
                    es[i].printStackTrace(System.err);
                }
            }
            exitState = 1;
            return false;
        }

        if (data.size() == 0 && metaData.size() == 0){ 
            throw new NoDataException(); 
        }

        if (data.size() == 0 && metaData.size() > 0) { 
            handleControlDoc(data, metaData); 
        }

        return true;
    }

    public void handleControlDoc(Bucket fileData, Bucket metaData)     
        throws IOException, NoDataException {
                    ReadInputStream in = 
                        new ReadInputStream(metaData.getInputStream());
                    String type=in.readTo('\n','\r');
                    FieldSet f=new FieldSet();
                    f.parseFields(in);
                    handleControlDoc(type, f, in,
                                     fileData, metaData);
         }

    protected void handleControlDoc(String type, FieldSet fields, 
                                    ReadInputStream in,
                                    Bucket fileData, Bucket metaData)     
        throws IOException, NoDataException {
        /*
          try {
          Loader.getInstance("Freenet.client.controldocs."+type,
          new Class[] {FieldSet.class},
          new Object[] {fields});
        */
        if ("Redirect".equals(type)) {
            FreenetURI redirTo=new FreenetURI(in.readTo('\n','\r'));

            System.err.println("Redirect to: "+redirTo);

            // check for date redirect, modify redirTo
            String base = fields.get("baseline");
            String incr = fields.get("increment");
            if (base != null && incr != null) {
                redirTo = ClientUtil.dateURI(redirTo, base, incr);
            }
            
            metaData.resetWrite();
            System.err.println("Redirect to: "+redirTo);
            if (autoRedirect) {
                fillBuckets(redirTo.toString(), fileData, metaData);
            }
        }
    }
   
            
    /**
     * Prints the usage of this freenet_request to standard out.
     **/
    public static void usage() {
        System.out.println("Usage: freenet_request URL output-file");
        System.out.println("");
        System.out.println("  -m|--metadata file             Save private metadata in 'file'");
        System.out.println("  -a|--autoRedirect              Follows redirects automatically. Default on.");
	System.out.println("  -A|--noAutoRedirect            Turns off automatic following redirects.");
        System.out.println("  -c|--cipher cipher-name        Use 'cipher-name' as the cipher");
        System.out.println("                                 Choices are 'Twofish' (default) or 'Rijndael'");
        System.out.println("  -p|--listenPort port           Local port to listen for replies on");
        System.out.println("  -f|--safer                     Avoid unnecessary risk by encrypting all data");
        System.out.println("                                 written to disk. Slow and not perfect.");
	System.out.println("  -F|--noSafer                   Don't be too safe.  (Default)");
        System.out.println("  -s|--serverAddress address     Server node to connect to");
        System.out.println("  -h|--htl hops-to-live          Hops-to-live for request");
        System.out.println("  -l|--logging error|normal      Logging level");
        System.out.println("               |minor|debugging");
        System.out.println("  -v|--verbosity 0-5             Verbosity of log messages");
        System.out.println("  -P|--useFCP                    Use FCP instead of FNP. EXPERIMENTAL!");
	System.out.println("  -D|--debugSet <param> <val>    Set an internal parameter manually.");
        System.out.println("");
        System.out.println("Examples:");
        System.out.println("  freenet_request test-key test.txt -h 45");
        System.out.println("  freenet_request freenet:KSK@test-key test.txt --safer");
        System.out.println("  freenet_request freenet:CHK@yoZOL0nrUhqIdGPjCYjgwhTJQAEDAQ,cH4oZkbe");
        System.out.println("                  HS5wJijKBtKuPA test.txt");
        System.out.println();
        System.out.println("Send bug reports to freenet-dev@lists.sourceforge.net");
    }
}
