package Freenet.client;
import Freenet.*;
import Freenet.crypt.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.client.events.*;
import Freenet.client.listeners.*;
import java.io.*;
import java.net.MalformedURLException;
import Freenet.keys.SVK;

/**
 * Simplified client library support put and get with automatic redirect
 * handling for both inserts and requests.
 *
 * @author <a href=mailto:blanu@uts.cc.utexas.edu>Brandon Wiley</a>
 **/

public class SimplifiedClient extends CLI
{
    boolean autoRedirect;
    Logger logger=null;
    Params params;

    static public void main(String[] args) throws Exception
    {
      Core.logger=new StandardLogger(System.out, 5, 0);
      SimplifiedClient sc=new SimplifiedClient(new Params(args));
      sc.setLogger(Core.logger);
      Bucket file=new FileBucket();
      Bucket meta=new FileBucket();
      sc.getData(args[0], file, meta);
      InputStream is=file.getInputStream();
      Conduit c=new Conduit(is, System.out, null);
      c.run();
    }

  public SimplifiedClient()
  {
    this(new Params());
  }

  public SimplifiedClient(Params params)
  {
    this(params, Core.logger);
  }

  public SimplifiedClient(Params params, Logger logger)
  {
    super(params, logger, true);
    autoRedirect=params.getboolean("autoRedirect", true);
    this.params = params;
    Core.init(params);
  }

  public String getString() {return "!";}

  public Logger getLogger() {return (logger!=null) ? logger : cc.logger;}
  public void setLogger(Logger l) {logger=l;}

    /**
     * Sets the HTL used for searches.
     *
     * @param htl New hops-to-live value.
     */

    public void setHtl(int htl) {
	this.htl = htl;
    }

    /**
     * Gets the current HTL value.
     *
     * @return Current hops-to-live value.
     */

    public int getHtl() {
	return htl;
    }

  public Pair get(String uri) throws MalformedURLException, IOException, NoDataException
  {
    Bucket data=new PipedBucket();
    Bucket metaData=new FileBucket();
    getData(uri, data, metaData);
    return new Pair(data, metaData);
  }

  public Pair getMetadata(String uri) throws MalformedURLException, IOException, NoDataException
  {
    Bucket data=new PipedBucket();
    Bucket metaData=new FileBucket();
    getMetadata(uri, data, metaData);
    return new Pair(data, metaData);
  }

  public long getMetadata(String uri, Bucket data, Bucket metaData) throws MalformedURLException, IOException, NoDataException, FileNotFoundException
  {
    return fillBuckets(uri, data, metaData, false);
  }

  public void get(String uri, String filename, String metaFilename) throws IOException, MalformedURLException, NoDataException
  {
    Bucket data=new FileBucket(new File(filename));
    Bucket metaData=null;
    if(metaFilename==null)
      metaData=new PipedBucket();
    else
      metaData=new FileBucket(new File(metaFilename));
    getData(uri, data, metaData);
  }

  public String put(String filename, String metaFilename) throws IOException, CollisionException
  {
    return put(new FileBucket(new File(filename)), new FileBucket(new File(metaFilename)));
  }

  public String put(Bucket data, Bucket meta) throws IOException, CollisionException
  {
    try { return put("freenet:CHK@", data, meta); }
    catch(MalformedURLException e)
    { getLogger().log(this, "SimplifiedClient is broken. freenet:CHK@ is reported as an invalid URI.", logger.ERROR); return null; }
  }

  public String put(String uri, String filename, String metaFilename) throws IOException, MalformedURLException, CollisionException
  {
    Bucket data=null;
    Bucket meta=null;
    if(filename!=null) data=new FileBucket(new File(filename));
    else data=new FileBucket();
    if(metaFilename!=null) meta=new FileBucket(new File(metaFilename));
    else meta=new FileBucket();
    return put(uri, data, meta, true);
  }

  public String put(String uri, Bucket plaintext, Bucket metaPlaintext) throws IOException, MalformedURLException, CollisionException
  {
    return put(uri, plaintext, metaPlaintext, true);
  }

  public String put(String uri, Bucket plaintext, Bucket metaPlaintext, boolean everything) throws IOException, MalformedURLException, CollisionException
  {
      boolean control=false;
      if (uri.toLowerCase().startsWith("freenet:")) uri = uri.substring(8);
      String keytype=(String)StringUtils.split(uri, '@').elementAt(0);
      if(keytype.equals("CHK") || !autoRedirect || plaintext==null)
      {
        FreenetURI key=dumpBuckets(uri, plaintext, metaPlaintext, everything);
        return ((key == null) ? null : key.toString());
      }
      else
      {
        FreenetURI chk=dumpBuckets("freenet:CHK@", plaintext, metaPlaintext, true);
        return ((chk == null) ? null : putRedirect(uri, chk.toString()));
      }
  }

  public String putRedirect(String from, String to) throws IOException, CollisionException
  {
      return putRedirect(from, to, new FieldSet());
  }

  public String putRedirect(String from, String to, FieldSet fs) throws IOException, CollisionException
  {
      Bucket meta=ClientUtil.makeRedirectBucket(new FreenetURI(to), fs);
      FreenetURI key=dumpBuckets(from, new FileBucket(), meta);
      if(meta instanceof FileBucket)
      {
        ((FileBucket)meta).getFile().delete();
      }
      return ((key == null) ? null : key.toString());
  }

  public String putControlDoc(String uri, Bucket b) throws IOException, MalformedURLException, CollisionException
  {
      if (uri.toLowerCase().startsWith("freenet:")) uri = uri.substring(8);
      String keytype=(String)StringUtils.split(uri, '@').elementAt(0);
      if(keytype.equals("CHK") || !autoRedirect)
      {
	  FreenetURI key=dumpBuckets(uri, null, b);
	  return ((key == null) ? null : key.toString());
      }
      else
      {
	  FreenetURI chk=dumpBuckets("freenet:CHK@", null, b);
	  return ((chk == null) ? null : putRedirect(uri, chk.toString()));
      }
  }

  public void getLatest(String SVK, String outFile)
  {
    
  }

  protected String randFilename()
  {
    return "t"+Math.abs(new java.util.Random().nextInt() % 10000);
  }

    public void getData(String uri, Bucket data, Bucket metaData) throws MalformedURLException, IOException, NoDataException, FileNotFoundException
    {
      fillBuckets(uri, data, metaData, true);
    }

    public long fillBuckets(String uri, Bucket data, Bucket metaData, boolean everything) throws MalformedURLException, IOException, NoDataException, FileNotFoundException
    {
        // *** FIX ME! ***
        if (ClientUtil.isMSK(uri)) uri = ClientUtil.lookupMSK(uri, params);

	    ClientFactory bcf = new FNPClient(cc, target);
        DataRequest req = new DataRequest(htl, uri, metaData, data);

	    DoneListener dl = new DoneListener();
            SegmentCompleteListener scl = new SegmentCompleteListener();
            TransferStartedListener tsl = new TransferStartedListener();

	    // register listeners
        req.addEventListener(new EventLogger(Core.logger));
        req.addEventListener(dl);
        req.addEventListener(scl);
        req.addEventListener(tsl);
	    // request
	    bcf.obtainClient(req).execute();
            synchronized(tsl)
            {
              try { tsl.waitEvent(); }
              catch(InterruptedException e) {}
            }

            TransferStartedEvent tse=tsl.event;

            // Something happened. The request failed.
            if(tse==null)
              throw new FileNotFoundException();

            long len=tse.getDataLength();
            long mlen=tse.getMetadataLength();

	    if (len == 0 && mlen == 0){ 
	      throw new NoDataException(); 
	    }

            // If it's a control document
            if (len == 0 && mlen > 0)
            { 
              // Download the control file.
              synchronized(dl)
              {
                try { dl.waitEvent(); }
                catch(InterruptedException e) {}
              }
            
              // If we want to automatically redirect
              if(autoRedirect)
                // Fetch the actual file
	        return handleControlDoc(data, metaData, everything);

              return mlen;
	    }
            // It's a normal file
            else
            {
            // If they want the whole file
            if(everything)
              // Download the whole file
	      synchronized(dl)
              {
                try { dl.waitEvent(); } 
                catch(InterruptedException e) {}
	      }
            // If they want just the metadata and there is some
            else if(mlen>0)
              // Otherwise download just the metadata
	      synchronized(scl)
              {
                try { scl.waitEvent(); } 
                catch(InterruptedException e) {}
	      }
            // Otherwise, they want just the metadata and there isn't any,
            // so we're done.
            }
            return len;
    }

  public long handleControlDoc(Bucket fileData, Bucket metaData, boolean everything) throws IOException, NoDataException
  {
    ReadInputStream in = new ReadInputStream(metaData.getInputStream());
    String type=in.readTo('\n','\r');
    FieldSet f=new FieldSet();
    f.parseFields(in);
    
    if ("Redirect".equals(type)) {
	FreenetURI redirTo=new FreenetURI(in.readTo('\n','\r'));
	fileData.resetWrite();
	metaData.resetWrite();
	getLogger().log(this, "Redirect to: "+redirTo, Logger.MINOR);
	// check for date redirect, modify redirTo
	String base = f.get("baseline");
	String incr = f.get("increment");
	if (base != null && incr != null) {
	  redirTo = ClientUtil.dateURI(redirTo, base, incr);
	}
	getLogger().log(this, "Redirect to: "+redirTo, Logger.MINOR);
	if (autoRedirect) {
	    return fillBuckets(redirTo.toString(), fileData, metaData, everything);
	}

    }
    return 0;
  }
  
    public FreenetURI dumpBuckets(String uri, Bucket plaintext, 
				  Bucket metaPlaintext) 
	    throws MalformedURLException, IOException, CollisionException {
        return dumpBuckets(new FreenetURI(uri), plaintext, metaPlaintext);
    }    

    public FreenetURI dumpBuckets(String uri, Bucket plaintext, 
				  Bucket metaPlaintext, boolean everything) 
	    throws MalformedURLException, IOException, CollisionException
    {
      return dumpBuckets(new FreenetURI(uri), plaintext, metaPlaintext, everything);
    }

    public FreenetURI dumpBuckets(FreenetURI keyName, Bucket plaintext, Bucket metaPlaintext)
    throws MalformedURLException, IOException, CollisionException
    {
      return dumpBuckets(keyName, plaintext, metaPlaintext, true);
    }

    public FreenetURI dumpBuckets(FreenetURI keyName, Bucket plaintext,
				  Bucket metaPlaintext, boolean everything)
	    throws MalformedURLException, IOException, CollisionException {

        InsertRequest req;
        try {
            req = new InsertRequest(htl, keyName, cipherName,
                                    metaPlaintext, plaintext, getBucket());
        } catch (InsertSizeException e) {
    		getLogger().log(this,
    		    "SVKs, KSKs, and SSKs cannot be more than"+Freenet.keys.SVK.SVK_MAXSIZE+" bytes.",
    		    Logger.ERROR);
            throw new IOException("SVKs, KSKs, and SSKs cannot be more than"+Freenet.keys.SVK.SVK_MAXSIZE+" bytes.");
        }

        ClientFactory bcf = new FNPClient(cc, target);

        DoneListener dl            = new DoneListener();
        ClientCollisionListener cl = new ClientCollisionListener();
        ExceptionListener el       = new ExceptionListener();
	    req.addEventListener(cl);
	    req.addEventListener(dl);
	    req.addEventListener(el);
	    req.addEventListener(new EventLogger(getLogger()));

	synchronized(dl) {
	    // request
	    bcf.obtainClient(req).execute();
	    // wait
	    try {
		dl.waitEvent();
	    } catch (InterruptedException e) {
	    }
	}

        keyName = req.getURI();   // read final URI back out

	if (req.state() < Request.DONE) {
	    Exception[] es = el.getExceptions();
	    if (es == null)
            {
		getLogger().log(this, "Insert failed gracefully.", Logger.NORMAL);
                if(cl.collisionHappened())
                {
		  if (keyName.getKeyType().equals("CHK"))
                  {
		    getLogger().log(this,"An exact copy of the data you tried " 
				       + "to insert was already present.", Logger.NORMAL);
                    return keyName;
                  }
                  throw new CollisionException();
                }
                throw new IOException("Graceful failure.");
	    } 
            else
            {
		getLogger().log(this,"Encountered the following exceptions while trying to Insert:",Logger.ERROR);
		for (int i = 0; i < es.length ; i++)
                {
//		    es[i].printStackTrace(getLogger().getOut());
		    es[i].printStackTrace();
		}
                throw new IOException("Exceptions while inserting.");
	    }
	} else {
	    getLogger().log(this, "Inserted Key   : " + keyName, Logger.NORMAL);
	    if (keyName.getKeyType().equals("SVK")) {
		getLogger().log(this, "SVK private key: " + 
				   Base64.encode(req.getPrivateKey()), Logger.NORMAL);
		getLogger().log(this, "Hang on to this key if you wish to update it in the future, or if you wish to\ninsert subspace keys (SSKs) under this SVK root.", Logger.NORMAL);
	    }
	    exitState = 0;
            return keyName;
	}
    }

    public Pair makeKeypair() {
	ClientSVK csvk = new ClientSVK(cc.randSource);
	Key k = csvk.getKey();
	String privkey = csvk.getPrivateKeyString();
	String pubkey = csvk.toString().substring(12);
	return new Pair(privkey, pubkey);
    }
}
