/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.FieldSet;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Toolkit;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Rectangle;
import java.awt.SystemColor;
import javax.swing.JComponent;
import javax.swing.JScrollPane;

/**
 * A viewer for binary data, which is used when we don't know what type of
 * data we have received.  TO DO: Hex dumps are a bit geeky.  Make it more
 * 'user oriented' rather than geek oriented somehow.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class BinaryViewer
  extends JScrollPane
  implements Viewer
{
  private Document doc;
  private int heightInPixels = 1;
  private JComponent hexDumpComponent;
  private long lastDataLength = -1L;
  
  private static Font font = new Font("Monospaced", Font.PLAIN, 11);
  private static FontMetrics fontMetrics = Toolkit.getDefaultToolkit().getFontMetrics(font);
  
  public static final long maxDataLength = 100000000L;
  public static final int bytes_per_row = 16;

  public BinaryViewer()
  {
    setViewportView(hexDumpComponent = new JComponent() {
      public void update(Graphics g)
      {
        paint(g);
      }

      public void paint(Graphics g)
      {
        g.setFont(font);
        byte[] data = doc.getData();
        long dataLengthLong = doc.getDataLength();
        if (dataLengthLong > maxDataLength)
          dataLengthLong = maxDataLength;
        int dataLength = (int) dataLengthLong;
        
        Rectangle clip = g.getClipBounds();
        g.setColor(SystemColor.text);
	g.fillRect(clip.x, clip.y, clip.width, clip.height);
	g.setColor(SystemColor.textText);

        int height = fontMetrics.getHeight();
        int ascent = fontMetrics.getAscent();
        int startrow = clip.y / height;
        int endrow = (clip.y + clip.height + height + 1) / height;
        for (int row = startrow; row <= endrow; row++) {
          StringBuffer sb = new StringBuffer();
          int starti = row*bytes_per_row;
          int endi = (row+1)*bytes_per_row;
          
          if (endi > dataLength) endi = dataLength;
          if (starti < endi) { 
              // Show the offset address.
            String hexAddr = Integer.toHexString(starti);
            for (int i = hexAddr.length(); i < 8; i++)
              sb.append((char)'0');
            sb.append(hexAddr);
            sb.append((char)' ');
            
              // Dump the bytes in hex.
            for (int i = starti; i < endi; i++) {
              sb.append((char)' ');
              int ch = (int) data[i] & 0xFF;
              String hexCh = Integer.toHexString(ch);
              if (hexCh.length() == 1) sb.append((char)'0');
              sb.append(hexCh);
            }
            
              // Line things up if endi-starti is less than 16.
              // This happens only on the last line of all.
            for (int i = (endi-starti); i < 16; i++)
              sb.append("   ");
            sb.append("  ");
              
              // Add the ASCII representation on the right-hand-side.
            for (int i = starti; i < endi; i++) {
              int ch = (int) data[i] & 0xFF;
              if (ch >= 0x21 && ch <= 0x7E)
                sb.append((char)ch);
              else
                sb.append((char)'.');
            }
            
            String text = sb.toString();
            g.drawString(text, 2, row * height + ascent);
          }
        }
      }
      
      public Dimension getPreferredSize()
      {
        Dimension superSize = super.getPreferredSize();
        return new Dimension(superSize.width, heightInPixels);
      }
    });
  }

    // Hand the viewer a reference to the document so it can get efficient
    // access to the contents of the document rather than having to cache the
    // contents of the document itself (which would be inefficient).
  public void setDocument(Document doc)
  {
    this.doc = doc;
  }

  /**
   * Set the listener to its initial state in the middle of a running stream.  Clears the
   * metadata and any received text.  Received streams can be terminated by the source and
   * replaced with a new stream at any time.  All document listeners must be able to cope
   * with this.  This method needn't be efficient, because the implementor can assume that
   * this method won't be called for the initial transfer.  It's only for re-starting an
   * already running transfer.
   */
  public void reset()
  {
    refresh();
  }

  /**
   * Indicate the Freenet request state.  See Freenet.client.Client for values. 
   */
  public void setState(int state)
  {
  }

  /**
   * Let the viewer have a copy of the metaData for this document.  Caller is required
   * to call this method with a non-null value.  If there is no metaData, caller should
   * construct an empty Properties object and pass it.  This method MUST be called
   * before the first call to push(..).
   *
   * <p>contentLength is the content length of the data if known, or -1L if unknown.   
   */
  public void setMetaData(FieldSet metaData, long contentLength)
  {
  }

  /**
   * Blocks of data are pushed to the viewer through this method, and when this pushing
   * is complete, the caller calls finish().
   */
  public void push(byte[] data, int offset, int length)
  {
    refresh();
  }

  /**
   * Called after multiple calls to push to signify that it's the end of the data.
   */
  public void finish(boolean success, Exception[] exceptions)
  {
    refresh();
  }
  
  /**
   * Refresh the display - called every time something changes.
   */
  private void refresh()
  {
    long dataLengthLong = doc.getDataLength();
    if (dataLengthLong > maxDataLength)
      dataLengthLong = maxDataLength;
    this.heightInPixels = (int) (((dataLengthLong + (long) (bytes_per_row-1)) /
      (long) bytes_per_row) * (long) fontMetrics.getHeight());
    hexDumpComponent.invalidate();
    validate();
  }
}
