/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.*;
import Freenet.support.*;
import Freenet.client.*;
import Freenet.client.events.*;
import Freenet.presentation.FreenetProtocol;
import java.net.BindException;
import java.util.Vector;
import java.util.Enumeration;

/**
 * The model for a 'browser window', including URL text field, icons, and
 * multiple documents.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class Browser
{
    // TO DO: Make it so we can edit this.
  private String targetText = "tcp/localhost:50001";
  
    // Make it so this gets generated properly.
  private String listeningAddress = "tcp/localhost:50050";

  private ClientCore core;
  private Client client;
  private long htl;

  private Vector documents = new Vector();
  private Vector browserListeners = new Vector();
  
  private Configuration configuration;

  public Browser(Configuration configuration)
  {
    this.configuration = configuration;
    connect();
  }
  
  public Client getClient()
  {
    return client;
  }
  
  public long getHTL()
  {
    return htl;
  }
  
  public void connect()
  {
    try {
      doConnect();
    }
    catch (ListenException e) {
        // TO DO: Dialog box.
      e.printStackTrace();
    }
    catch (BadAddressException e) {
        // TO DO: Dialog box.
      e.printStackTrace();
    }
    catch (BindException e) {
        // TO DO: Dialog box.
      e.printStackTrace();
    }
  }

  private synchronized void doConnect()
    throws ListenException, BadAddressException, BindException
  {
    if (client != null)
      close();
      
    Params params = configuration.getParams();
    params.setParam("transient","yes"); // clients are always transient

      // logging
    String logthresh = params.getParam("logging","NORMAL");
    int thresh = StandardLogger.priorityOf(logthresh);
    int verb = params.getint("verbosity",StandardLogger.SPARSE);
    Logger log = new StandardLogger(System.err,verb,thresh);

    int serverListenPort = params.getint("listenPort", 19114);
    String addrstr = params.getParam("serverAddress", "tcp/127.0.0.1:"+serverListenPort);

    core = ClientUtil.getTCPCore(0,params,log);
  	Address addr = ClientUtil.getAddress(addrstr);
    client = new Client(core, addr);

    htl = params.getlong("htl", 10);
  }

  /**
   * Do whatever we have to do to close the connection.
   */
  public void close()
  {
      // TO DO: Put some code here.
    client = null;
    core = null;
  }
  
  public void add(Document doc)
  {
    documents.add(doc);
    
      // Notify listeners of added document.
    Vector browserListeners = (Vector) this.browserListeners.clone();
    Enumeration enum = browserListeners.elements();
    while (enum.hasMoreElements()) {
      BrowserListener listener = (BrowserListener) enum.nextElement();
      listener.documentAdded(doc);
    }
  }
  
  public void remove(Document doc)
  {
    synchronized (documents) {
      if (documents.contains(doc)) {
        doc.close();
        documents.remove(doc);
      }
    }

      // Notify listeners of removed document.
    Vector browserListeners = (Vector) this.browserListeners.clone();
    Enumeration enum = browserListeners.elements();
    while (enum.hasMoreElements()) {
      BrowserListener listener = (BrowserListener) enum.nextElement();
      listener.documentRemoved(doc);
    }
  }

  public void addBrowserListener(BrowserListener listener)
  {
    browserListeners.add(listener);
  }
  
  public void removeBrowserListener(BrowserListener listener)
  {
    browserListeners.remove(listener);
  }
}
