/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import javax.swing.JPanel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JTabbedPane;
import javax.swing.JLabel;
import javax.swing.Icon;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JSeparator;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JFileChooser;
import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.File;
import java.io.FileOutputStream;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import Freenet.FieldSet;

/**
 * The view for a Browser, including URL text field, icons, and
 * multiple documents.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class BrowserPane
  extends JPanel
  implements BrowserListener
{
  private Browser browser;
  
  /**
   * The colour of the top part of the display.
   */
  public static final Color topBackground = new Color(178, 212, 255);
  public static final Color topForeground = new Color(200, 60, 0);
  
  private JPanel topPanel = new JPanel();
  private JPanel topBars = new JPanel();
  private KeyEntryField kef;
  private JTabbedPane docTabber = new JTabbedPane();
  private JPanel iconPanel;
  
  /**
   * Maps document to DocumentPane.
   */
  private Hashtable documentPanes = new Hashtable();
  /**
   * Maps DocumentPanes to Documents.
   */
  private Hashtable documents = new Hashtable();
  
  public BrowserPane(Browser browser)
  {
    this.browser = browser;
    
    setBackground(topBackground);
    setLayout(new BorderLayout());
    add(BorderLayout.NORTH, topPanel);
    topPanel.setLayout(new BorderLayout());
    iconPanel = new IconPanel(new ImageIcon(getFileData("/Freenet/client/gui/hopstolive.jpg"), "Freenet"));
    iconPanel.setBackground(topBackground);
    topPanel.add(iconPanel, BorderLayout.EAST);
    topBars.setLayout(new GridLayout(3, 1));
    topBars.add(new JPanel());  // <-- Place holder for menu
                                //     Caller uses addMenuBar() to add their own menu.
    JLabel temporaryLabel = new JLabel("<- Buttons ->");
    topBars.setBackground(topBackground);
    topBars.setForeground(topForeground);
    temporaryLabel.setForeground(topForeground);
    topBars.add(temporaryLabel);
    kef = new KeyEntryField(browser);
    kef.setBackground(topBackground);
    kef.setForeground(topForeground);
    topBars.add(kef);
    topPanel.add(topBars, BorderLayout.CENTER);
    
    add(BorderLayout.CENTER, docTabber);
    docTabber.setBackground(topBackground);
    docTabber.addTab("Empty", new JPanel());
    
    browser.addBrowserListener(this);
  }
  
  public Image getIconImage()
  {
    return new ImageIcon(BrowserPane.getFileData("/Freenet/client/gui/hops32x32.gif")).getImage();
  }
  
  private static ImageIcon icon = new ImageIcon(BrowserPane.getFileData("/Freenet/client/gui/hops32x32.gif"));
  private static Image     iconImage = icon.getImage();
  
  /**
   * Caller must construct a menu bar and add it with this method if they want menus.
   */
  public void addMenuBar(JMenuBar menuBar)
  {
    topBars.remove(0);
    topBars.add(menuBar, 0);
  }

  /**
   * Fetch the contents of a file in the classpath.
   */
  static byte[] getFileData(String resourceName)
  {
    try {
      InputStream is = GUI.class.getResourceAsStream(resourceName);
      if (is == null)
        throw new InternalError("Can't find resource "+resourceName);
      ByteArrayOutputStream bos = new ByteArrayOutputStream();
      int n;
      byte[] buf = new byte[512];
      while ((n = is.read(buf)) > 0)
        bos.write(buf, 0, n);
      return bos.toByteArray();
    }
    catch (IOException e) {
      throw new InternalError(e.toString()+" while loading resource "+resourceName);
    }
  }

  private Vector uriSelectionListeners = new Vector();
  /**
   * Add a listener that gets notified when the user selects a URI either from the
   * pressing enter on the entry field, or through someone calling loadPage(..);
   */
  public void addURISelectionListener(final URISelectionListener listener)
  {
      // This won't work properly if there are multiple listeners, as only one
      // of the listeners will get any text.  Address this situation if it comes
      // up.
    ActionListener entryFieldAction = new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
          // Get the text that the user entered.
        String uriText = kef.getText();
        if (uriText.length() > 0) {
            // If the listener was successful in processing the entered key, then we
            // remove the text from the KeyEntryField.  This is different to the way
            // a normal web browser works, but trust me - you'll learn to love it.
            // (I think.)
          if (listener.uriSelected(uriText))
            kef.clearText();
        }
      }
    };
    kef.addActionListener(entryFieldAction);
    
    uriSelectionListeners.add(listener);
  }

  /**
   * Make the browser load a new page with the specified URL.
   */
  public void loadPage(String text)
  {
    Enumeration enum = ((Vector)uriSelectionListeners.clone()).elements();
    while (enum.hasMoreElements()) {
      URISelectionListener listener = (URISelectionListener)enum.nextElement();
      listener.uriSelected(text);
    }
  }

  /**
   * BrowserListener interface: Gets called when a document is added to the browser.
   */
  public synchronized void documentAdded(Document doc)
  {
      // If there are no real documents, then remove the empty document we use
      // as a placeholder.
    if (documentPanes.size() == 0)
      docTabber.removeTabAt(0);
    DocumentPane docPane = new DocumentPane(this, doc);
    documentPanes.put(doc, docPane);
    documents.put(docPane, doc);
    docTabber.addTab("loading...", docPane);
    docTabber.setSelectedComponent(docPane);
  }

  /**
   * BrowserListener interface: Gets called when a document is removed from the
   * browser.
   */
  public synchronized void documentRemoved(Document doc)
  {
    DocumentPane pane = (DocumentPane) documentPanes.get(doc);
    if (pane != null) {
      documentPanes.remove(doc);
      documents.remove(pane);
      docTabber.remove(pane);
      
        // If there are no document panes left, then add an empty one.
      if (documentPanes.size() == 0)
        docTabber.addTab("Empty", new JPanel());
    }
  }
  
  private static File currentDirectory = null;
  
  /**
   * Return the action to perform on selecting 'File' 'Open'.
   */
  public ActionListener getOpenAction()
  {
    return new ActionListener() {
      public void actionPerformed(ActionEvent evt)
      {
        JFileChooser dialog = new JFileChooser(currentDirectory);
        JFrame frame = new JFrame("Open File");
        try {
          frame.setIconImage(getIconImage());
          int answer = dialog.showOpenDialog(frame.getContentPane());
          if (answer == JFileChooser.CANCEL_OPTION) {
          }
          else
          if (answer == JFileChooser.APPROVE_OPTION) {
            try {
              loadPage("file:"+dialog.getSelectedFile().getCanonicalPath());
            }
              // TO DO: Do something that's not half-arsed with this exception, i.e.
              // pass it to the documentPane's error handler (once I write one).
            catch (IOException e) {
              e.printStackTrace();
            }
          }
        }
        finally {
          frame.dispose();
        }
      }
    };
  }

  /**
   * A factory method that creates a new progress indicator.  We can make this more
   * fancy if we want different kinds of progress indicators.
   */
  private ProgressIndicator createProgressIndicator(String title, String description)
  {
    ProgressDialog dialog = new ProgressDialog(title, description);
    dialog.setIconImage(getIconImage());
    dialog.setVisible(true);
    Point xy = getLocationOnScreen();
    dialog.setLocation(new Point(xy.x+50, xy.y+35));
    return dialog;
  }

  /**
   * Return the action to perform on selecting 'File' 'Save'.
   */
  public ActionListener getSaveAction()
  {
      // TO DO: 'Saving' dialog box with progress bar.
    class SaveAction implements ActionListener, DocumentListener, ProgressIndicatorListener {
      private String filename;
      private FileOutputStream fos;
      private Document doc;
      private IOException ioexception;

      public void actionPerformed(ActionEvent evt)
      {
          // Work out which document is the current one.
        Component c = docTabber.getSelectedComponent();
        if (c instanceof DocumentPane) {
          DocumentPane pane = (DocumentPane) c; 
          if (pane != null) {
            this.doc = (Document) documents.get(pane);
            if (doc != null) {
                // Ask user where they want to save it...
              JFileChooser dialog = new JFileChooser(currentDirectory);
              JFrame frame = new JFrame("Open File");
              try {
                frame.setIconImage(getIconImage());
                int answer = dialog.showSaveDialog(frame.getContentPane());
                if (answer == JFileChooser.CANCEL_OPTION) {
                }
                else
                if (answer == JFileChooser.APPROVE_OPTION) {
                  try {
                    filename = dialog.getSelectedFile().getCanonicalPath();
                    fos = new FileOutputStream(filename);
                    ProgressIndicator progressIndicator = createProgressIndicator("Saving...", "Saving "+filename);
                    progressIndicator.addProgressIndicatorListener(this);
                    doc.pushTo(this);
                    doc.pushTo(new ProgressIndicatorAdapter(progressIndicator));
                  }
                    // TO DO: Handle this properly (open a dialog box).
                  catch (IOException e) {
                    e.printStackTrace();
                  }
                }
              }
              finally {
                frame.dispose();
              }
            }
          }
        }
      }
      
      /**
       * Set the listener to its initial state in the middle of a running stream.  Clears the
       * metadata and any received text.  Received streams can be terminated by the source and
       * replaced with a new stream at any time.  All document listeners must be able to cope
       * with this.  This method needn't be efficient, because the implementor can assume that
       * this method won't be called for the initial transfer.  It's only for re-starting an
       * already running transfer.
       */
      public void reset()
      {
        if (ioexception == null) {
          try {fos.close();} catch (IOException e) {}
          try {
            fos = new FileOutputStream(filename);
          }
          catch (IOException e) {
            ioexception = e;
          }
        }
      }

      /**
       * Indicate the Freenet request state.  See Freenet.client.Client for values. 
       */
      public void setState(int state)
      {
      }

      /**
       * Let the listener have a copy of the metaData for this document.  Caller is required
       * to call this method with a non-null value.  If there is no metaData, caller should
       * construct an empty Properties object and pass it.  This method MUST be called
       * before the first call to push(..).
       *
       * <p>contentLength is the content length of the data if known, or -1L if unknown.   
       */
      public void setMetaData(FieldSet metaData, long contentLength)
      {
      }

      /**
       * Blocks of data are pushed to the listener through this method, and when this pushing
       * is complete, the caller calls finish().
       */
      public void push(byte[] data, int offset, int length)
      {
        if (ioexception == null) {
          try {
            fos.write(data, offset, length);
          }
          catch (IOException e) {
            ioexception = e;
          }
        }
      }

      /**
       * Called when the user has requested that the transfer be cancelled.
       */
      public void requestCancelTransfer(ProgressIndicator indicator)
      {
          // Tell the document to stop sending us stuff!
        doc.terminatePush(this);
      }

      /**
       * Called after multiple calls to push to signify that it's the end of the data.
       */
      public void finish(boolean success, Exception[] exceptions)
      {
        try {
          fos.close();
        }
        catch (IOException e) {
          if (ioexception == null)
            ioexception = e;
        }

          // If we failed to write for any reason (but were otherwise successful), then
          // change the resulting status to failure and pass the exception along. 
        if (success && ioexception != null) {
          success = false;
          exceptions = new Exception[] {ioexception};
        }

          // If it wasn't successful, then delete the broken file.
          // TO DO: Error dialog box.
        if (!success)
          new File(filename).delete();
      }
    }
    return new SaveAction();
  }
  
  /**
   * Return the action to perform on selecting 'File' 'Close'. 
   */
  public ActionListener getCloseAction()
  {
    return new ActionListener() {
      public void actionPerformed(ActionEvent evt)
      {
        Component c = docTabber.getSelectedComponent();
        if (c instanceof DocumentPane) {
          DocumentPane pane = (DocumentPane) c; 
          if (pane != null) {
            Document doc = (Document) documents.get(pane);
            if (doc != null)
              browser.remove(doc);
          }
        }
      }
    };
  }
}

/**
 * A little panel that contains nothing but an image.
 */
class IconPanel
  extends JPanel
{
  private Icon icon;
  
  public IconPanel(Icon icon)
  {
    this.icon = icon;
    setBorder(null);
  }
  
  public void paint(Graphics g)
  {
    Dimension size = getSize();
    g.setColor(getBackground());
    g.fillRect(0, 0, size.width, size.height);
    icon.paintIcon(this, g, 0, 0);
  }
  
  public Dimension getPreferredSize()
  {
    return new Dimension(
      icon.getIconWidth(),
      icon.getIconHeight());
  }
}
