/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.FieldSet;
import Freenet.support.io.ReadInputStream;
import java.util.Vector;
import java.util.Properties;
import java.util.Enumeration;

/**
 * Encapsulates a document that is viewed in the browser.
 *
 * <p>It listens to the DocumentLoader via a DocumentListener interface, and fetches the
 * contents of the document.  It then acts as a read-through cache, keeping the contents
 * in memory, and uses the same DocumentLoader interface to push the contents out to
 * the Viewer.
 *
 * <p>TO DO: (Eventually) Make it so Document supports files > 4 GB.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class Document
  implements DocumentListener
{
  private Browser browser;
  private URIExpert uriExpert;
  private DocumentLoader loader;
  
  private Vector documentListeners = new Vector();
  private FieldSet metaData = null;

    // Storage for incoming data.
  private byte[] data = new byte[2048];
    // Length of the valid text in the data array.
  private long dataLength = 0;

    // Total length of document - may be known at the beginning, in which case we
    // are able to draw progress bars and such.
  private long contentLength = -1L;

  private boolean finishedLoading = false;
  private boolean success;
  private Exception[] exceptions;
  
  private int currentState = -100;

  public Document(Browser browser, URIExpert uriExpert, DocumentLoader loader)
  {
    this.browser = browser;
    this.uriExpert = uriExpert;
    this.loader = loader;

    loader.addDocumentListener(this);
    loader.start();

  }
  
  public String getURI()
  {
    return uriExpert.getCanonicalURI();
  }
  
  public URIExpert getURIExpert()
  {
    return uriExpert;
  }

  public void close()
  {
    loader.removeDocumentListener(this);
    loader.close();
  }

  /**
   * Push the contents of this Document to the specified listener, which will usually
   * be a Viewer.
   *
   * <p>This method has the same effect on the viewer whether the contents of the document
   * have been fully fetched from the network or not.  If fully fetched, it will obviously
   * happen quicker.
   */
  public synchronized void pushTo(DocumentListener listener)
  {
    documentListeners.add(listener);

    if (currentState != -100)
      listener.setState(currentState);

      // If we've got metaData, then push that
    if (metaData != null || contentLength != -1L)
      listener.setMetaData(metaData, contentLength);
      // If we've got some data already, then push it all out to the listener immediately.
    if (!(finishedLoading && !success) && dataLength > 0)
        // TO DO: Add support for facking enormous files by caching them on disk if they
        // gets too big.
      listener.push(data, 0, (int) dataLength);

    if (finishedLoading)
      listener.finish(success, exceptions);
  }

  /**
   * Terminate the pushing of data to the specified document listener.
   *
   * <p>If there are no more listeners, then the transfer itself will be cancelled.
   */
  public synchronized void terminatePush(DocumentListener listener)
  {
    if (documentListeners.removeElement(listener)) {
      success = false;
      exceptions = new Exception[] {new CancelledByUserException()};    
      listener.finish(success, exceptions);
  
        // If no-one is listening any more, then end it all.
      if (documentListeners.size() == 0) {
        loader.removeDocumentListener(this);
        loader.close();
      }
    }
  }

  /**
   * Set the listener to its initial state in the middle of a running stream.  Clears the
   * metadata and any received text.  Received streams can be terminated by the source and
   * replaced with a new stream at any time.  All document listeners must be able to cope
   * with this.  This method needn't be efficient, because the implementor can assume that
   * this method won't be called for the initial transfer.  It's only for re-starting an
   * already running transfer.
   */
  public synchronized void reset()
  {
    this.finishedLoading = false;
    this.success = false;
    this.metaData = null;
    this.data = new byte[2048];
    this.dataLength = 0L;
    this.contentLength = -1L;
    Enumeration enum = documentListeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.reset();
    }
  }

  /**
   * Indicate the Freenet request state.  See Freenet.client.Client for values. 
   */
  public synchronized void setState(int state)
  {
    this.currentState = state;
    Enumeration enum = documentListeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.setState(state);
    }
  }

  /**
   * Let the listener have a copy of the metaData for this document.  Caller is required
   * to call this method with a non-null value.  If there is no metaData, caller should
   * construct an empty Properties object and pass it.
   *
   * <p>contentLength is the content length of the data if known, or -1L if unknown.
   *
   * <p>This class is more tolerant than the DocumentPane or Viewer classes.
   * It allows push methods to be called before setMetaData, but it queues the data
   * up and sends it all to allow the listeners to assume that setMetaData WILL be called
   * before the first call to push.
   */
  public synchronized void setMetaData(FieldSet metaData, long contentLength)
  {
    boolean wasNull = this.metaData == null;
    
    this.metaData = metaData;
    this.contentLength = contentLength;

    Enumeration enum = documentListeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.setMetaData(metaData, contentLength);
    }
  }

  /**
   * Blocks of data are pushed to the listener through this method, and when this pushing
   * is complete, the caller calls finish().
   */
  public synchronized void push(byte[] pushedData, int pushedOffset, int pushedLength)
  {
      // Store the data locally.
      // First ensure there is enough space in the data array.
      // TO DO: Add support for enormous files.
    if (dataLength + pushedLength > data.length) {
      int newCapacity = data.length;
      do {
        newCapacity = newCapacity * 2;
      } while ((int) dataLength + pushedLength > newCapacity);
      byte[] newData = new byte[newCapacity];
      System.arraycopy(data, 0, newData, 0, (int) dataLength);
      data = newData;
    }
    System.arraycopy(pushedData, pushedOffset, data, (int) dataLength, pushedLength);
    dataLength += pushedLength;

      // Push the data on to all listeners.
    Enumeration enum = documentListeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.push(pushedData, pushedOffset, pushedLength);
    }
  }

  /**
   * Called after multiple calls to push to signify that it's the end of the data.
   */
  public synchronized void finish(boolean success, Exception[] exceptions)
  {
    if (contentLength == -1L)
      contentLength = dataLength;
  
    this.finishedLoading = true;
    this.success = success;
    this.exceptions = exceptions;

      // Push the finish() event to all listeners.
    Enumeration enum = documentListeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.finish(success, exceptions);
    }
    documentListeners.removeAllElements();
  }

  
// ------ Methods to get data directly from the document

  /**
   * Get the byte data stored in this Document.  Useful if you want your Viewer to be
   * really efficient and not store its own copy of the data.  The length of the data
   * is determined by getDataLength(), NOT the length of the array.
   */
  public byte[] getData()
  {
    return data;
  }

  /**
   * The length of the array returned by getData(), which may be less than the ultimate
   * length of the document.
   */
  public long getDataLength()
  {
    return dataLength;
  }
  
  /**
   * The ultimate length of the data contents, or -1L if not known yet.  This may be
   * known early on, in which case we can draw progress bars.
   */
  public long getContentLength()
  {
    return contentLength;
  }
}
