/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.FieldSet;
import java.util.Enumeration;
import java.util.Vector;

/**
 * Class that loads a document.  How and from where it is loaded is defined by
 * the subclass.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public abstract class DocumentLoader
{
  public abstract void start();
  
  private Vector listeners = new Vector();
  
  public void addDocumentListener(DocumentListener listener)
  {
    listeners.add(listener);
  }
  
  public void removeDocumentListener(DocumentListener listener)
  {
    listeners.remove(listener);
  }

  /**
   * Call reset on all listeners.
   */
  public void fireReset()
  {
    Vector listeners = (Vector) this.listeners.clone();
    Enumeration enum = listeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.reset();
    }
  }

  /**
   * Call setState(state) on all listeners.
   */
  public void fireSetState(int state)
  {
    Vector listeners = (Vector) this.listeners.clone();
    Enumeration enum = listeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.setState(state);
    }
  }

  /**
   * Let the listener have a copy of the metaData for this document.  Caller is required
   * to call this method with a non-null value.  If there is no metaData, caller should
   * construct an empty Properties object and pass it.  This method MUST be called
   * before the first call to push(..).
   *
   * <p>contentLength is the content length of the data if known, or -1L if unknown.   
   */
  public void fireSetMetaData(FieldSet metaData, long contentLength)
  {
    Vector listeners = (Vector) this.listeners.clone();
    Enumeration enum = listeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.setMetaData(metaData, contentLength);
    }
  }

  /**
   * Blocks of data are pushed to the listener through this method, and when this pushing
   * is complete, the caller calls finish().
   */
  public void firePush(byte[] data, int offset, int length)
  {
    Vector listeners = (Vector) this.listeners.clone();
    Enumeration enum = listeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.push(data, offset, length);
    }
  }

  /**
   * Called after multiple calls to push to signify that it's the end of the data.
   */
  public void fireFinish(boolean success, Exception[] exceptions)
  {
    Vector listeners = (Vector) this.listeners.clone();
    Enumeration enum = listeners.elements();
    while (enum.hasMoreElements()) {
      DocumentListener listener = (DocumentListener) enum.nextElement();
      listener.finish(success, exceptions);
    }
  }

  /**
   * Free up any resources we're using, and, if the transfer is still in progress, terminate
   * it.
   */
  public abstract void close();
}
