/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.FieldSet;
import java.awt.Frame;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.JComponent;
import javax.swing.JComboBox;
import javax.swing.DefaultComboBoxModel;
import java.io.*;

/**
 * The view of a document.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class DocumentPane
  extends JPanel
  implements DocumentListener
{
  private BrowserPane browserPane;
  private Document doc;
  private JLabel uri;
  private JComboBox mimeTypeBox;
  private FieldSet metaData;
  private Viewer viewer;
  private String[] mimeTypes;
  private boolean userHasSelectedMIMEType = false;

  public DocumentPane(BrowserPane browserPane, Document doc)
  {
    this.browserPane = browserPane;
    this.doc = doc;

    setLayout(new BorderLayout());

    uri = new JLabel(doc.getURI());
    
      // Get the list of MIME types that have useful viewers attached to them.
      // Add '<unknown>' as the first item.
    String[] origMimeTypes = MIMEMapper.getMIMETypesWithViewers();
    mimeTypes = new String[origMimeTypes.length+1];
    System.arraycopy(origMimeTypes, 0, mimeTypes, 1, origMimeTypes.length);
    mimeTypes[0] = "<unknown>";
    mimeTypeBox = new JComboBox(mimeTypes);
    
    JPanel topPanel = new JPanel();
    topPanel.setLayout(new BorderLayout());
    topPanel.add(uri, BorderLayout.CENTER);
    topPanel.add(mimeTypeBox, BorderLayout.EAST);
    add(topPanel, BorderLayout.NORTH);

    DocumentProgressIndicator pi = new DocumentProgressIndicator();
    add(pi, BorderLayout.SOUTH);
    
      // Tell the document to push its contents at us.  We then delegate to the
      // right viewer (once we know which viewer to use).
    doc.pushTo(this);
    doc.pushTo(new ProgressIndicatorAdapter(pi));
    
      // If the user selects a MIME type, then change the document pane so it
      // uses the appropriate viewer for the MIME type selected.
    mimeTypeBox.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent evt)
      {
        int index = mimeTypeBox.getSelectedIndex();
        if (index >= 0 && index < mimeTypes.length) {
          String mimeType = mimeTypes[index];
          userHasSelectedMIMEType = true;
          changeViewerAccordingToMIMEType(mimeType);
        }
      }
    });
  }

  /**
   * Set the listener to its initial state in the middle of a running stream.  Clears the
   * metadata and any received text.  Received streams can be terminated by the source and
   * replaced with a new stream at any time.  All document listeners must be able to cope
   * with this.  This method needn't be efficient, because the implementor can assume that
   * this method won't be called for the initial transfer.  It's only for re-starting an
   * already running transfer.
   */
  public void reset()
  {
    this.metaData = null;
    if (viewer != null)
      viewer.reset();
  }

  /**
   * Indicate the Freenet request state.  See Freenet.client.Client for values. 
   */
  public void setState(int state)
  {
    // Don't tell the viewer the state.
  }

  /**
   * Let the listener have a copy of the metaData for this document.  Caller is required
   * to call this method with a non-null value.  If there is no metaData, caller should
   * construct an empty Properties object and pass it.  This method MUST be called
   * before the first call to push(..).
   */
  public void setMetaData(FieldSet metaData, long contentLength)
  {
    this.metaData = metaData;

    if (metaData != null && !userHasSelectedMIMEType) {
      String mimeType = metaData.get("Content-Type");

        // If no mime type was specified, then guess the mime type from the extension.
      if (mimeType == null) {
        String extension = doc.getURIExpert().getExtension();
        if (extension != null)
          mimeType = MIMEMapper.guessMIMEType(extension); 
      }

        // Show the MIME type in the Combo Box.
      setMIMETypeInComboBox(mimeType);

        // Change the viewer if necessary.
      changeViewerAccordingToMIMEType(mimeType);
    }
  }

  private void changeViewerAccordingToMIMEType(String mimeType)
  {
      // See what kind of viewer is required to display this data...
    Viewer newViewer = MIMEMapper.getViewer(mimeType);

      // If there's an existing viewer (which will happen if reset() has been
      // called, or if the user selects a MIME type) then we check to see if it's
      // the same class as the newly generated one.  If it is, then we keep the
      // old viewer, which is a a bit more efficient than thowing it away and
      // substituting the new one.
    if (viewer != null && viewer.getClass().equals(newViewer.getClass()))
      ;
    else {
      if (viewer != null)
        remove((JComponent)viewer);
      viewer = newViewer;
      viewer.setDocument(doc);
      add((JComponent)viewer, BorderLayout.CENTER);
      updateWindow();
      doc.pushTo(viewer);
    }
  }

  private void setMIMETypeInComboBox(String mimeType)
  {
      // Find the index of this mime type in the combo box...
    int foundIdx = -1;
    
      // If the MIME type is null, then it's <unknown>
    if (mimeType == null)
      foundIdx = 0;
    else
    for (int i = 1; i < mimeTypes.length; i++) {
      if (mimeType.equals(mimeTypes[i])) {
        foundIdx = i;
        break;
      }
    }
    
      // If this MIME type isn't found in the list, then add it to the end and
      // update the ComboBox accordingly.
    if (foundIdx == -1) {
      String[] newMimeTypes = new String[mimeTypes.length+1];
      System.arraycopy(mimeTypes, 0, newMimeTypes, 0, mimeTypes.length);
      newMimeTypes[mimeTypes.length] = mimeType;
      mimeTypes = newMimeTypes;
      foundIdx = mimeTypes.length-1;
      mimeTypeBox.setModel(new DefaultComboBoxModel(mimeTypes));
    }
    
    mimeTypeBox.setSelectedIndex(foundIdx);
  }

  /**
   * Blocks of data are pushed to the listener through this method, and when this pushing
   * is complete, the caller calls finish().
   */
  public void push(byte[] data, int offset, int length)
  {
  }

  /**
   * Called after multiple calls to push to signify that it's the end of the data.
   */
  public void finish(boolean success, Exception[] exceptions)
  {
      // If it failed, then replace the viewer with an error viewer. 
    if (!success) {
      System.err.println("transfer of "+doc.getURI()+" failed:");
      System.err.print("  ");
      if (exceptions != null && exceptions.length > 0)
        exceptions[0].printStackTrace();
      else
        System.err.println("no exception was returned");
      if (viewer != null)
        remove((JComponent)viewer);
      viewer = new ErrorViewer();
      add((JComponent)viewer, BorderLayout.CENTER);
      viewer.finish(success, exceptions);
      updateWindow();
    }
  }
  
  private void updateWindow()
  {
    invalidate();
    validate();
  }
}
