/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.client.*;
import Freenet.client.events.*;
import java.io.*;
import java.util.Vector;

public class FreenetDocumentLoader
  extends DocumentLoader
  implements ClientEventListener
{
  private Client client;
  private long htl;
  private FreenetURI uri;
  private Bucket metaDataBucket;
  private Bucket dataBucket;
  private Vector exceptions = new Vector();
  private boolean finished;
  private long length = -1L;

  public FreenetDocumentLoader(Client client, long htl, FreenetURI uri)
  {
    this.client = client;
    this.htl = htl;
    this.uri = uri;
  }

  public void start()
  {
      // prepare
    ClientKey ck = ClientUtil.getKeyFromURI(uri);

    metaDataBucket = new FileBucket();
    dataBucket = new DataBucket();

    RequestToken rt = client.prepare(htl, ck, dataBucket, metaDataBucket);
  
      // register listeners
    rt.addEventListener(new EventLogger(Core.logger));
    rt.addEventListener(this);
    
    client.makeRequest(rt);    
  }

  public void close()
  {
    if (!finished)
      System.err.println("TO DO: Implement terminating an in-progress freenet transfer");
  }

  /**
   * Handle client event. 
   */
  public void receive(ClientEvent ce) {
    if (ce instanceof StateReachedEvent) {
      StateReachedEvent sr = (StateReachedEvent) ce;
      if (sr.getState() == Client.FAILED ||
		      sr.getState() == Client.DONE) {
            
          // Freenet control document.
        if (sr.getState() == Client.FAILED) {
          exceptions.addElement(new FileNotFoundException(uri.toString()));
        }
        else
        if (dataBucket.size() == 0L && metaDataBucket.size() > 0L) {
          System.err.println("Receiving Freenet control document.");
          try {
              ReadInputStream in = new ReadInputStream(metaDataBucket.getInputStream());
              String type=in.readTo('\n','\r');
              FieldSet f=new FieldSet();
              f.parseFields(in);
              if ("Redirect".equals(type)) {
                uri=new FreenetURI(in.readTo('\n','\r'));
                System.err.println("Redirect to: "+uri);
                fireReset();
                start();
                return;
              }
              else
                throw new IOException("can't handle freenet control document of type "+type);
          } catch (IOException e) {
            exceptions.addElement(e);
          }
        }
          // Normal document.
        else {
          FieldSet metaData = new FieldSet();
          try {
            InputStream is = metaDataBucket.getInputStream();
            if (is != null) {
              try {
                metaData.parseFields(new ReadInputStream(is), '\n', '=', '/');
              }
              finally {
                try {is.close();} catch (IOException e) {}
              }
            }
          }
          catch (IOException e) {
            exceptions.addElement(e);
          }
            // TO DO: Push metadata as soon as it's available, and fetch the content length.
          fireSetMetaData(metaData, length);
        }
        Exception[] excs = new Exception[exceptions.size()];
        exceptions.copyInto(excs);
        finished = true;
        fireFinish(sr.getState() == Client.DONE && excs.length == 0, excs);
		  }
      else
        fireSetState(sr.getState());
    }
    else
    if (ce instanceof TransferStartedEvent) {
      TransferStartedEvent tse = (TransferStartedEvent) ce;
      length = tse.getLength();
      fireSetMetaData(null, length);
    }
    else
    if (ce instanceof ExceptionEvent) {
      System.out.println("description: "+((ExceptionEvent) ce).getDescription());
      try {
        ((ExceptionEvent) ce).rethrow();
      } catch (Throwable e) {
        e.printStackTrace();
        exceptions.addElement(e);
      }
    }
  }
  
  private class DataBucket
    extends OutputStream
    implements Bucket
  {
    private long size = 0L;

    // ------ Bucket interface
    
    public InputStream getInputStream()
    {
      throw new InternalError("DataBucket.getInputStream() not implemented");
    }
    
    public OutputStream getOutputStream()
    {
      return this;
    }
    
    public void resetWrite()
    {
      throw new InternalError("resetWrite() not implemented");
    }
    
    public long size()
    {
      return size;
    }
    
    // ------ OutputStream interface
    
    public void write(int b)
    {
      write(new byte[] {(byte)b});
    }
    
    public void write(byte[] b)
    {
      write(b, 0, b.length);
    }
    
    public void write(byte[] b, int off, int len)
    {
      size += len;
      firePush(b, off, len);
    }
    
    public void flush()
    {
    }
    
    public void close()
    {
    }
  }
}
