/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.Core;
import Freenet.ListenException;
import Freenet.BadAddressException;
import javax.swing.JFrame;
import javax.swing.WindowConstants;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JMenuBar;
import javax.swing.JSeparator;
import javax.swing.UIManager;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.net.MalformedURLException;

/**
 * The Freenet GUI Client.
 *
 * <p>You will observe that this is the fabulous model-view-controller pattern.
 * GUI is the controller, where you will find all the highest-level control of
 * program flow.  Classes that sound like widgets, such as 'DocumentPane' are the
 * view.  They interface with the user.  Classes that sound like things, e.g.
 * 'Browser', 'Document' are the model.  They represent things that are being
 * manipulated, and they have no GUI component.  (I don't actually know what I'm
 * doing, but I know all the right buzzwords.)
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class GUI
  implements BrowserListener
{
  public static void main(String[] args)
  {
    //set the LookAndFeel to be system specific
    try {
            UIManager.setLookAndFeel(
                UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) { }
    new GUI(args);
  }

  private JFrame frame;
  private Browser browser;
  private BrowserPane browserPane;
  private JMenuItem[] docRelatedMenuItems;

  public GUI(String[] args)
  {
    JFrame frame = new JFrame("Freenet Client "+Core.freenetVersion);
    
    /**
     * Handles requests to exit the program.
     */
    class ExitAction extends WindowAdapter implements ActionListener
    {
      public void windowClosing(WindowEvent evt)
      {
        System.exit(0);
      }
        
      public void actionPerformed(ActionEvent evt)
      {
        System.exit(0);
      }
    }
    
    ExitAction exitAction = new ExitAction();
    frame.addWindowListener(exitAction);
    frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

    Configuration configuration = new Configuration(args);
    browser = new Browser(configuration);
    
    browserPane = new BrowserPane(browser);
    frame.getContentPane().add(browserPane);
    frame.setBackground(BrowserPane.topBackground);

    frame.setIconImage(browserPane.getIconImage());
    
      // When the user selects a URI, we action it.
    browserPane.addURISelectionListener(new URIActioner());

    JMenuBar menu = new JMenuBar();
    menu.setBackground(browserPane.topBackground);
    menu.setForeground(browserPane.topForeground);
    JMenu fileMenu = new JMenu("File");
    fileMenu.setMnemonic('F');
    fileMenu.setBackground(browserPane.topBackground);
    fileMenu.setForeground(browserPane.topForeground);
    menu.add(fileMenu);
    JMenuItem mi;
    docRelatedMenuItems = new JMenuItem[2];
    fileMenu.add(mi = new JMenuItem("Open")); mi.addActionListener(browserPane.getOpenAction());
    mi.setMnemonic('O');
    fileMenu.add(mi = new JMenuItem("Save As")); mi.addActionListener(browserPane.getSaveAction());
    mi.setMnemonic('S');
    docRelatedMenuItems[0] = mi;
    mi.setEnabled(false);
    fileMenu.add(new JSeparator());
    fileMenu.add(mi = new JMenuItem("Close")); mi.addActionListener(browserPane.getCloseAction());
    mi.setMnemonic('C');
    docRelatedMenuItems[1] = mi;
    mi.setEnabled(false);
    fileMenu.add(mi = new JMenuItem("Exit")); mi.addActionListener(exitAction);
    mi.setMnemonic('x');
    browserPane.addMenuBar(menu);
    
      // Make it so we get notified when documents are added and removed so we can
      // grey out the right menu options.
    browser.addBrowserListener(this);

    frame.setSize(600, 550); 
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    frame.setLocation(screenSize.width/2-300, screenSize.height/2-225);
    frame.setVisible(true);
    frame.requestFocus();

  }

  /**
   * This code is in charge of greying out menu items that need greying out when
   * there are no documents.
   */
  private int noOfDocuments = 0;
  public void documentAdded(Document doc)
  {
    if (noOfDocuments == 0)
      for (int i = 0; i < docRelatedMenuItems.length; i++)
        docRelatedMenuItems[i].setEnabled(true);
    noOfDocuments++;
  }
  public void documentRemoved(Document doc)
  {
    noOfDocuments--;
    if (noOfDocuments == 0)
      for (int i = 0; i < docRelatedMenuItems.length; i++)
        docRelatedMenuItems[i].setEnabled(false);
  }

  /**
   * URISelectionListener that handles user pressing ENTER in URI selection box.
   */
  private class URIActioner
    implements URISelectionListener
  {
    public boolean uriSelected(String uriText)
    {
      try {
          // Let's ask the expert what we should do.
        URIExpert expert = new URIExpert(uriText);

          // OK, expert, gimme something that can load this document, depending on
          // what protocol was specified (freenet:.., http:.., etc...)
        DocumentLoader loader = expert.getDocumentLoader(browser.getClient(), browser.getHTL());
        
        System.out.println(expert.getCanonicalURI()+" loaded by "+loader.getClass().getName());
        
        Document doc = new Document(browser, expert, loader);
        browser.add(doc);

          // Tell caller we succeeded.
        return true;
      }
      catch (MalformedURLException e) {
          // TO DO: Dialog box.
        e.printStackTrace();
        return false;
      }
    }
  }
}

