/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.client.Client;
import Freenet.*;
import java.net.*;
import java.io.*;

/**
 * This class loads ordinary URLs through Java's built-in mechanisms.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class JavaDocumentLoader
  extends DocumentLoader
  implements Runnable
{
  private URL url;
  private InputStream is;
  private boolean terminated = false;

  public JavaDocumentLoader(URL url)
  {
    this.url = url;
  }

  public void start()
  {
    fireSetState(Client.REQUESTING);
    new Thread(this).start();
  }

  public void close()
  {
      // The closest thing to co-operatively killing the thread is to close off the
      // the Input Stream.  On a sensible JVM implementation, this causes the blocking
      // read on the reading thread to terminate immediately.  There are some stupid
      // implementations, unfortunately, but this is the only way of doing this.  The
      // worst case is that the thread will hang there for a while consuming resources.
    try {
      InputStream is = this.is;
      this.is = null;
      if (is != null) {
        terminated = true;
        is.close();
      }
    }
    catch (IOException e) {
    }
  }

  public void run()
  {
    try {
      URLConnection conn = url.openConnection();
      this.is = conn.getInputStream();
      try {
        FieldSet metaData = new FieldSet();
        String mimeType = conn.getHeaderField("Content-Type");
        if (mimeType == null)
          mimeType = MIMEMapper.guessMIMEType(url.getFile());
        metaData.add("Content-Type", mimeType); 
        fireSetMetaData(metaData, conn.getContentLength());
        byte[] buf = new byte[16384];
        int n;
        boolean started = false;
        while (true) {
          InputStream is = this.is;
          if (is == null)
            throw new IOException("transfer terminated");
          if ((n = is.read(buf)) < 0)
            break;
          if (!started) {
            started = true;
            fireSetState(Client.TRANSFERING);
          }
          firePush(buf, 0, n);
        }
        fireFinish(true, null);
      }
      finally {
        try {
          InputStream is = this.is;
          this.is = null;
          if (is != null)
            is.close();
        }
        catch (IOException e) {
        }
      }
    }
    catch (IOException e) {
      e.printStackTrace();
      if (terminated)
        System.err.println("Terminated transfer of "+url);
      fireFinish(false, new Exception[] {e});
    }
  }
}
