/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import java.awt.GridLayout;
import java.awt.FlowLayout;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.JFrame;
import javax.swing.JProgressBar;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JPanel;
import java.util.Vector;
import java.util.Enumeration;

/**
 * A dialog box in a frame that pops up to indicate the progress of a save or
 * insert. 
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class ProgressDialog
  extends JFrame
  implements ProgressIndicator, ActionListener
{
  private JProgressBar progressBar;
  private JButton      button;
  private Vector       listeners = new Vector();
  private long         maximum;
  private Object       cancelLock = new Object();

  public ProgressDialog(String title, String description)
  {
    super(title);
    getContentPane().setLayout(new GridLayout(3, 1));
    getContentPane().add(new JLabel(description));
    getContentPane().add(progressBar = new JProgressBar());
    progressBar.setMaximum(10000);
    JPanel panel = new JPanel();
    panel.setLayout(new FlowLayout(FlowLayout.RIGHT));
    panel.add(button = new JButton("Cancel"));
    getContentPane().add(panel);
    button.addActionListener(this);
    pack();
  }
  
  public Dimension getPreferredSize()
  {
      // Give the dialog box a minimum width of 450 pixels.
    Dimension dim = super.getPreferredSize();
    if (dim.width < 450)
      return new Dimension(450, dim.height);
    else
      return dim;
  }

  /**
   * Add a listener that gets notified when a command is issued by the progress
   * indicator.  The most obvious command is the user requesting that the transfer
   * be cancelled.
   */
  public void addProgressIndicatorListener(ProgressIndicatorListener listener)
  {
    listeners.addElement(listener);
  }

  public void removeProgressIndicatorListener(ProgressIndicatorListener listener)
  {
    listeners.removeElement(listener);
  }

  /**
   * Set the maximum value of the progress indicator. 
   */
  public void setMaximum(long maximum)
  {
    this.maximum = maximum;
  }

  /**
   * Set the current value of the progress indicator.
   */
  public void setValue(long value)
  {
    if (maximum != -1L)
      progressBar.setValue((int) ((value * 10000L)/maximum));
  }

  /**
   * Set some text that will appear on the indicator.
   */
  public void setText(String text)
  {
    progressBar.setString(text);
    progressBar.setStringPainted(true);
  }

  /**
   * Dispose of the progress indicator once the transfer has completed.
   */
  public void dispose()
  {
    synchronized (cancelLock) {
      super.dispose();
    }
  }

  /**
   * ActionListener method that's called if the user clicks on 'Cancel'.
   */
  public synchronized void actionPerformed(ActionEvent evt)
  {
    synchronized (cancelLock) {
      Enumeration enum = ((Vector)this.listeners.clone()).elements();
      while (enum.hasMoreElements()) {
        ProgressIndicatorListener listener = (ProgressIndicatorListener) enum.nextElement();
        listener.requestCancelTransfer(this);
      }
    }
  }
}
