/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

/**
 * A class whose job it is to limit the frequency of updates, as it would waste
 * CPU time if updates were too frequent.  This might happen if someone
 * transfers a huge file from local disk to local disk.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class ProgressIndicatorAgent
  implements ProgressIndicator
{
  private ProgressIndicator indicator;
  private long value;
  private long maximum;
  private long lastUpdateTime = 0L;

  public ProgressIndicatorAgent(ProgressIndicator indicator)
  {
    this.indicator = indicator;
    maximum = -1L;
    value = 0L;
  }

  /**
   * Add a listener that gets notified when a command is issued by the progress
   * indicator.  The most obvious command is the user requesting that the transfer
   * be cancelled.
   */
  public void addProgressIndicatorListener(ProgressIndicatorListener listener)
  {
    indicator.addProgressIndicatorListener(listener);
  }
  
  public void removeProgressIndicatorListener(ProgressIndicatorListener listener)
  {
    indicator.removeProgressIndicatorListener(listener);
  }

  public void setMaximum(long maximum)
  {
    if (this.maximum != maximum) {
      this.maximum = maximum;
      indicator.setMaximum(maximum);
      setValue(value);
    }
  }

  public void setValue(long value)
  {
    if (maximum != -1L) {
      long now = System.currentTimeMillis();
      long since = now - lastUpdateTime;
      if (since < 0 || since >= 500L) {
        indicator.setValue(value);
        lastUpdateTime = now;
      }
    }
  }
  
  public void setText(String text)
  {
    indicator.setText(text);
  }
  
  public void dispose()
  {
    indicator.dispose();
  }
}

