/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

 */

package Freenet.client.gui;

import Freenet.client.Client;
import Freenet.client.FreenetURI;
import java.net.URL;
import java.net.MalformedURLException;

/**
 * Class that knows all about raw URI's entered by a user, what they mean, and how they
 * are to be loaded.
 *
 * @author Stephen Blackheath (stephen@blacksapphire.com)
 */
public class URIExpert
{
  private String rawURI;
  private String protocol;
  private String path;

  public URIExpert(String rawURI)
  {
    this.rawURI = rawURI;
    
    int colonPos = rawURI.indexOf(':');

      // If no protocol is specified, then default to 'freenet:'.
    if (colonPos == -1) {
      protocol = "freenet";
      path = rawURI;
    }
    else {
      protocol = rawURI.substring(0, colonPos).toLowerCase();
        // If it's one of the protocols we know about, then take it as
        // the protocol, and the rest of the rawURI as the path.
      if (protocol.equals("file") ||
          protocol.equals("http") ||
          protocol.equals("freenet")) {
        path = rawURI.substring(colonPos+1);
      }
        // If we haven't heard of this protocol, then assume the protocol
        // is 'freenet' and the rawURI is just a freenet key that contains
        // a colon.
      else {
        protocol = "freenet";
        path = rawURI;
      }
    }
  }

  public String getCanonicalURI()
  {
    return protocol+":"+path;
  }

  public String getProtocol()
  {
    return protocol;
  }

  public String getPath()
  {
    return path;
  }

  /**
   * Get the object the knows how to load this document - different depending on the
   * protocol that was specified in the URI.
   */
  public DocumentLoader getDocumentLoader(Client client, long htl)
    throws MalformedURLException
  {
    if (protocol.equals("freenet"))
      return new FreenetDocumentLoader(client, htl, new FreenetURI(getCanonicalURI()));
    else
      return new JavaDocumentLoader(new URL(getCanonicalURI()));
  }

  /**
   * Get the file extension.  We are assuming that the key looks like a URL, which it may
   * not.  It is only used as a clue to guess the MIME type if one isn't supplied in the
   * MetaData.
   */
  public String getExtension()
  {
    int atIdx = path.indexOf('@');
    int pathIdx = atIdx < 0 ? 0 : atIdx+1;
    
      // If there's a dot after '@' (or if no '@', if there's a dot at all...)
    int dotIdx = path.lastIndexOf('.');
    if (dotIdx >= pathIdx) {
      int slashIdx = path.lastIndexOf('/');
        // And there's no slash after this dot...
      if (slashIdx < dotIdx)
          // Then we treat everything after the last dot as the file extension.
        return path.substring(dotIdx+1);
    }
    return null;
  }
}
