/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Literal.java
 *
 * Created on 26 July 2000, 14:33
 */

package Freenet.client.rdf;

/** An RDF Literal.
 * <p>Literal objects can be interpreted as typed values.  The literal string
 * is considered to be a serialization of the typed value.  Methods are
 * provided for retrieving built in java types which are represented by
 * the string produced by their <CODE>toString</CODE> method. A generic
 * method for creating an arbitrary object using a factory is also
 * provided.</p>
 *
 * <p>This interface provides methods supporting typed literals.  This means
 *    that methods are provided which will translate a built in type, or an
 *    object to an RDF Literal.  This translation is done by invoking the
 *    <CODE>toString()</CODE> method of the object, or its built in equivalent.
 *    The reverse translation is also supported.  This is built in for built
 *    in types.  Factory objects, provided by the application, are used
 *    for application objects.</p>
 * @author bwm
 */
public interface Literal extends RDFNode {
    /** Interpret the literal as a String representation of a Boolean
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a boolean
     */
    public boolean getBoolean() throws RDFException;
    
    /** Interpret the literal as a String representation of a byte
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a byte
     */
    public byte getByte() throws RDFException;
    
    /** Interpret the literal as a String representation of a short
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a short
     */
    public short getShort() throws RDFException;
    
    /** Interpret the literal as a String representation of an int
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as an int
     */
    public int getInt() throws RDFException;
    
    /** Interpret the literal as a String representation of a long
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a long
     */
    public long getLong() throws RDFException;
    
    /** Interpret the literal as a String representation of a char
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a char
     */
    public char getChar() throws RDFException;
    
    /** Interpret the literal as a String representation of a float
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a float
     */
    public float getFloat() throws RDFException;
    
    /** Interpret the literal as a String representation of a double
     * and return its value.
     * @throws RDFException Generic RDF Exception
     * @return the literal interpeted as a double
     */
    public double getDouble() throws RDFException;
    
    /** Return the String value of the literal
     * @throws RDFException Generic RDF Exception
     * @return the literal string
     */
    public String getString() throws RDFException;
    
    /** Interpret the literal as a String representation of an object.
     * @return the object created from the literal string
     * @param f A factory object for creating the returned object.
     * @throws RDFException Generic RDF Exception
     */
    public Object getObject(ObjectF f) throws RDFException;
    
    /** If a language is defined for this literal return it
     * @return the language for this literal if it exists, or null
     */    
    public String getLanguage();
}
