/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Model.java
 *
 * Created on 26 July 2000, 07:58
 */

package Freenet.client.rdf;

import java.io.PrintWriter;
import java.io.Reader;

/** An RDF Model.
 *
 * <p>An RDF model is a set of Statments.  Methods are provided for creating
 *    resources, properties and literals and the Statements which link them,
 *    for adding statements to and removing them from a model, for 
 *    querying a model and set operations for combining models.</p>
 * <p>This interface provides methods supporting typed literals.  This means
 *    that methods are provided which will translate a built in type, or an
 *    object to an RDF Literal.  This translation is done by invoking the
 *    <CODE>toString()</CODE> method of the object, or its built in equivalent.
 *    The reverse translation is also supported.  This is built in for built
 *    in types.  Factory objects, provided by the application, are used
 *    for application objects.</p>
 * <p>This interface provides methods for supporting enhanced resources.  An
 *    enhanced resource is a resource to which the application has added
 *    behaviour.  RDF containers are examples of enhanced resources built in
 *    to this package.  Enhanced resources are supported by encapsulating a
 *    resource created by an implementation in another class which adds
 *    the extra behaviour.  Factory objects are used to construct such
 *    enhanced resources.</p>
 * @author bwm
 */
public interface Model {

    /** computes the number of statements in the model.
     *
     * @throws RDFException Generic RDF Exception
     * @return the number of statements in the model
     */
    long  size() throws RDFException;
    
/** List all resources which are subjects of statements.
 *
 * <p>Subsequent operations on those resource may modify this model.</p>
 * @throws RDFException Generic RDF Exception
 * @return an iterator over a set of resources which are subjects of statements
 *         in the model.
 */
    ResIterator listSubjects() throws RDFException;

/** List all namespaces corresponding to predicates.
 * @throws RDFException Generic RDF Exception
 * @return an iterator over the set of namespaces associated with predicates in 
 *         the model.
 */
    NsIterator listNameSpaces() throws RDFException;

/** List all statements.
 *
 *  <p>Subsequent operations on those statements may modify this model.</p>
 * @throws RDFException Generic RDF Exception
 * @return an iterator over all statements in the model.
 */
    StmtIterator listStatements() throws RDFException;

/** List all reified statements in the model.
 *
 * <p> Subsequent operations on those statements may modify this model.</p>
 * @throws RDFException Generic RDF Exception
 * @return an iterator over the set of all reified statements in the model
*/
    StmtIterator listReifiedStatements() throws RDFException;

/** Return a Resource instance in this model.
 *
 * <p>Subsequent operations on the returned object may modify this model.</p>
 * <p>The resource is assumed to already exist in the model.  If it does not,
 * <CODE>createResource</CODE> should be used instead.</p>
 * @return a resource instance
 * @param uri the URI of the resource
 * @throws RDFException Generic RDF exception.
 */    
    Resource getResource(String uri) throws RDFException;

/** Return a Resource instance in this model.
 *
 * <p>Subsequent operations on the returned object may modify this model.</p>
 * <p>The resource is assumed to already exist in the model.  If it does not,
 * <CODE>createResource</CODE> should be used instead.</p>
 * @return a resource instance created by the factory provided
 * @param uri the URI of the resource
 * @param f the factory object
 * @throws RDFException Generic RDF Exception 
 */    
    Resource getResource(String uri, ResourceF f) throws RDFException;

/** Return a Property instance in this model.
 *
 * <p>Subsequent operations on the returned property may modify this model.</p>
 * <p>The property is assumed to already exist in the model.  If it does not,
 * <CODE>createProperty</CODE> should be used instead.</p>
 * @return a property object
 * @param uri the URI of the property
 * @throws RDFException Generic RDF Exception 
*/
    Property getProperty(String uri) throws RDFException;

/** Return a Property instance in this model.
 *
 * <p>Subsequent operations on the returned property may modify this model.</p>
 * <p>The property is assumed to already exist in the model.  If it does not,
 * <CODE>createProperty</CODE> should be used instead.</p>
 * @return a property linked to this model
 * @param nameSpace the RDF namespace of the property
 * @param localName the localName of the property in its namespace
 * @throws RDFException Generic RDF Exception
 */    
    Property getProperty(String nameSpace, String localName)
                          throws RDFException;

/** Return a Bag instance in this model.
 *
 * <p>Subsequent operations on the returned bag may modify this model.</p>
 * <p>The bag is assumed to already exist in the model.  If it does not,
 * <CODE>createBag</CODE> should be used instead.</p>
 * @return a bag instance
 * @param uri the URI of the bag.
 * @throws RDFException Generic RDF Exception
 */ 
    Bag getBag(String uri) throws RDFException;

/** Return a bag instance based on a given resource.
 *
 * <p> This method enables an application to treat any resource as a bag.
 *     It is in effect an unsafe downcast.</p>
 *
 * <p>Subsequent operations on the returned bag may modify this model.</p>
 * <p>The bag is assumed to already exist in the model.  If it does not,
 * <CODE>createBag</CODE> should be used instead.</p>
 * @return a bag instance
 * @param r an untyped Resource instance 
 * @throws RDFException Generic RDF Exception
 */ 
    Bag getBag(Resource r) throws RDFException;

/** Return an Alt instance in this model.
 *
 * <p>Subsequent operations on the returned object may modify this model.</p>
 * <p>The alt is assumed to already exist in the model.  If it does not,
 * <CODE>createAlt</CODE> should be used instead.</p>
 * @return an alt instance
 * @param uri the URI of the alt
 * @throws RDFException Generic RDF Exception
 */ 
    Alt getAlt(String uri) throws RDFException;

/** Return an Alt instance based on a given resource.
 *
 * <p> This method enables an application to treat any resource as an Alt.
 *     It is in effect an unsafe downcast.</p>
 *
 * <p>Subsequent operations on the returned Alt may modify this model.</p>
 * <p>The bag is assumed to already exist in the model.  If it does not,
 * <CODE>createAlt</CODE> should be used instead.</p>
 * @return an Alt instance
 * @param r an untyped Resource instance
 * @throws RDFException Generic RDF Exception
 */ 
    Alt getAlt(Resource r) throws RDFException;
/** Return a Seq instance in this model.
 *
 * <p>Subsequent operations on the returned bag may modify this model.</p>
 * <p>The seq is assumed to already exist in the model.  If it does not,
 * <CODE>createSeq</CODE> should be used instead.</p>
 * @return a seq instance
 * @param uri the URI of the seq
 * @throws RDFException Generic RDF Exception
 */ 
    Seq getSeq(String uri) throws RDFException;

/** Return a Seq instance based on a given resource.
 *
 * <p> This method enables an application to treat any resource as a Seq.
 *     It is in effect an unsafe downcast.</p>
 *
 * <p>Subsequent operations on the returned Seq may modify this model.</p>
 * <p>The Seq is assumed to already exist in the model.  If it does not,
 * <CODE>createAlt</CODE> should be used instead.</p>
 * @return an Alt instance
 * @param r an untyped Resource instance
 * @throws RDFException Generic RDF Exception
 */ 
    Seq getSeq(Resource r) throws RDFException;

/** Create a new anonymous resource.
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * @throws RDFException Generic RDF exception.
 * @return a new anonymous resource linked to this model.
 */
    public Resource createResource() throws RDFException;


/** Create a new anonymous resource with a given type.
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * <p> The resource is created and an rdf:type property added to the model
 * to specify its type. </p>
 * @ param type the type of the resource to be created.
 * @return a new anonymous resource linked to this model.
 * @param type A resource representing the RDF type of the new resource.
 * @throws RDFException Generic RDF exception
 */
    public Resource createResource(Resource type) throws RDFException;

/** Create a new resource.
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * @param uri the URI of the resource to be created
 * @throws RDFException Generic RDF exception.
 * @return a new resource linked to this model.
 */
    public Resource createResource(String uri) throws RDFException;

/** Create a new resource with a given type.
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * <p> The resource is created and an rdf:type property added to the model
 * to specify its type. </p>
 * @ param type the type of the resource to be created.
 * @return a new resource linked to this model.
 * @param uri The URI of the new resource.
 * @param type A resource representing the RDF type of the new resource.
 * @throws RDFException Generic RDF exception.
 */
    public Resource createResource(String uri, Resource type)
                                    throws RDFException;

/** Create a new anonymous resource using the supplied factory
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * @return a new anonymous resource linked to this model.
 * @param f A factory object to create the returned object.
 * @throws RDFException Generic RDF exception.
 */
    public Resource createResource(ResourceF f) throws RDFException;
 
/** Create a new resource using the supplied factory
 *
 * <p> Subsequent operations on the returned resource may modify this model.
 * </p>
 * @return a new resource linked to this model.
 * @param uri the URI of the resource
 * @param f A factory to create the returned object.
 * @throws RDFException Generic RDF exception.
 */   
    public Resource createResource(String uri, ResourceF f) throws RDFException;

/** Create a property
 *
 * <p> Subsequent operations on the returned property may modify this model.
 * </p>
 * @param uri the URI of the property
 * @throws RDFException Generic RDF exception
 * @return a property instance
 */
    public Property createProperty(String uri) throws RDFException;

/** Create a property
 *
 * <p> Subsequent operations on the returned property may modify this model.
 * </p>
 * @param nameSpace the nameSpace of the property
 * @param localName the name of the property within its namespace
 * @throws RDFException Generic RDF exception
 * @return a property instance
 */
    public Property createProperty(String nameSpace, String localName) 
                                     throws RDFException;

    /** create a literal from a boolean value.
     *
     * <p> The value is converted to a string using its <CODE>toString</CODE>
     * method. </p>
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */
    public Literal createLiteral(boolean v) throws RDFException; 
    /** create a literal from an integer value.
     *
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */   
    public Literal createLiteral(long v) throws RDFException;
    /** create a literal from a char value.
     *
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */
    public Literal createLiteral(char v) throws RDFException;
    /** create a literal from a float value.
     *
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */
    public Literal createLiteral(float v) throws RDFException;
    /** create a literal from a double value.
     *
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */
    public Literal createLiteral(double v) throws RDFException;
    /** create a literal from a String value.
     *
     * @param v the value of the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v
     */
    public Literal createLiteral(String v) throws RDFException;
    
    /** create a literal from a String value with a specified language
     *
     * @param v the value of the literal
     * @param language the language associated with the literal
     * @throws RDFException generic RDF exception
     * @return a new literal representing the value v with the given language
     */
    public Literal createLiteral(String v, String language) throws RDFException;
    
    /** create a literal from an Object.
     *
     * @return a new literal representing the value v
     * @param v the value of the literal.
     * @throws RDFException generic RDF exception 
     */
    public Literal createLiteral(Object v) throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts will
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, RDFNode o) 
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, boolean o) 
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, long o)
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, char o)
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, float o)
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, double o)
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The Object o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, String o)  
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The Object o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @param l the language associated with the object
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, String o, String l)  
                                      throws RDFException;
    
    /** Create a Statement instance.
     *
     * <p>Subsequent operations on the statement or any of its parts may
     * modify this model.</p>
     * <p>Creating a statement does not add it to the set of statements in the
     * model. </p>
     * <p>The value o will be converted to a Literal.</P>
     * @param s the subject of the statement
     * @param p the predicate of the statement
     * @param o is the value to be the object of the statement
     * @throws RDFException generic RDF exception
     * @return the new statement
     */
    public Statement createStatement(Resource s, Property p, Object o)  
                                      throws RDFException;
    
    /** Create a new anonymous bag.
     *
     * <p>Subsequent operations on the bag or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new bag is added to this model.
     * </p>
     * @throws RDFException Generic RDF exception.
     * @return a new anonymous bag.
     */
    public Bag createBag() throws RDFException;
    
    /** Create a new bag.
     *
     * <p>Subsequent operations on the bag or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new bag is added to this model.
     * </p>
     * @param uri The URI of the new Bag.
     * @throws RDFException Generic RDF exception.
     * @return a new bag.
     */
    public Bag createBag(String uri) throws RDFException;
       
    /** Create a new anonymous alt.
     *
     * <p>Subsequent operations on the alt or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new alt is added to this model.
     * </p>
     * @throws RDFException Generic RDF exception.
     * @return a new anonymous alt.
     */
    public Alt createAlt() throws RDFException;
    
    /** Create a new alt.
     *
     * <p>Subsequent operations on the alt or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new alt is added to this model.
     * </p>
     * @param uri The URI of the new alt.
     * @throws RDFException Generic RDF exception.
     * @return a new alt.
     */
    public Alt createAlt(String uri) throws RDFException;
       
    /** Create a new anonymous seq.
     *
     * <p>Subsequent operations on the seq or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new seq is added to this model.
     * </p>
     * @throws RDFException Generic RDF exception.
     * @return a new anonymous seq.
     */
    public Seq createSeq() throws RDFException;
    
    /** Create a new seq.
     *
     * <p>Subsequent operations on the seq or any of its parts may
     * modify this model.</p>
     * <p>A statement defining the type of the new seq is added to this model.
     * </p>
     * @param uri The URI of the new seq.
     * @throws RDFException Generic RDF exception.
     * @return a new seq.
     */
    public Seq createSeq(String uri) throws RDFException;

/** Add a statement to this model.
 * @return This model.
 * @param s The statement to be added.
 * @throws RDFException Generic RDF Exception
 */     
    Model add(Statement s) throws RDFException;

/** add a statement to this model
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, RDFNode o)     throws RDFException;

/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, boolean o) throws RDFException;
/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, long o) throws RDFException;
/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, char o) throws RDFException;
/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, float o) throws RDFException;
/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, double o) throws RDFException;

/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, String o) throws RDFException;

/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @param l the language associated with the object
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, String o, String l) throws RDFException;
/** add a statement to this model
 *
 * @return this model
 * @param s the subject of the statement to add
 * @param p the predicate of the statement to add
 * @param o the object of the statement to add
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Resource s, Property p, Object o) throws RDFException;
/** Add all the statements returned by an iterator to this model.
 * @return this model
 * @param iter An iterator which returns the statements to be added.
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(StmtIterator iter) throws RDFException;

/** Add all the statements in another model to this model.
 * @return this model
 * @param m The model whose statements are to be added.
 * @throws RDFException Generic RDF Exception
 */ 
    Model add(Model m) throws RDFException;

/** Add the RDF statements an XML document.
 * @return this model
 * @param url of the document containing the RDF statements.
 * @throws RDFException Generic RDF Exception
 */ 
    Model read(String url) throws RDFException;

/** Add RDF statements from an XML document.
 *
 * @return this model
 * @param reader a reader onto an xml document containing RDF statements
 * @param the base URI for resources define with ID attributes in the source
 * @throws RDFException Generic RDF Exception
 */ 
    Model read(Reader reader, String base) throws RDFException;

/** Write the model as an XML document.
 *
 * @return this model
 * @param writer a writer to which the XML will be written
 * @throws RDFException Generic RDF Exception
 */ 
    Model write(PrintWriter writer) throws RDFException;

/** Removes a statement.
 * @return this model
 * @param s The statement to be removed.
 * @throws RDFException Generic RDF Exception
 */ 
    Model remove(Statement s) throws RDFException;

/** Remove all the Statements returned by an iterator.
 * @return this model
 * @param iter the iterator which returns the statements to be removed.
 * @throws RDFException Generic RDF Exception
 */ 
    Model remove(StmtIterator iter) throws RDFException;

/** Remove all the Statements in a given model.
 * @return this model
 * @param m the model containing the statements to be removed.
 * @throws RDFException Generic RDF Exception
 */ 
    Model remove(Model m) throws RDFException;

/** Return a statement with given subject and property.
 *  <p>If more than one statement witht the given subject and property
 *  exists in the model, it is undefined which will be returned.</p>
 * @return A statement from the model with the given subject and property.
 * @param s The subject of the statement to be returned.
 * @param p The property of the statement to be returned.
 * @throws RDFException Generic RDF Exception
 */     
    Statement getProperty(Resource s, Property p) throws RDFException;

/** List all subjects with a given property.
 * @return an iterator over the subjects
 * @param p the property sought.
 * @throws RDFException Generic RDF Exception
 */     
    ResIterator listSubjectsWithProperty(Property p) throws RDFException;

/** List all subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, RDFNode o)
                                           throws RDFException;

/** List all subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, boolean o)
                                           throws RDFException;

/** List all the subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, long o)
                                           throws RDFException;

/** List all subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, char o)
                                          throws RDFException;

/** List all subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, float o)
                                           throws RDFException;
/** lists all subjects with a given property and property value
 * @return an iterator over the set of subjects
 * @param p The property sought.
 * @param o The property value sought.
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, double o)
                                           throws RDFException;
/** lists all subjects with a given property and property value
 * @return an iterator over the set of subjects
 * @param p The predicate sought.
 * @param o The property value sought.
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, String o)
                                          throws RDFException;
/** lists all subjects with a given property and property value
 * @return an iterator over the set of subjects
 * @param p The predicate sought.
 * @param o The property value sought.
 * @param l the language associated with the object
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, String o, String l)
                                          throws RDFException;

/** List all subjects with a given property and property value
 * @return an iterator over the subjects
 * @param p The predicate sought
 * @param o The value sought
 * @throws RDFException Generic RDF Exception
 */ 
    ResIterator listSubjectsWithProperty(Property p, Object o)
                                           throws RDFException;

/** List all objects in a model.
 * @return an iterator over the objects
 * @param p The predicate sought
 * @throws RDFException Generic RDF Exception
 */ 
    NodeIterator listObjects() throws RDFException;

/** List all objects of a given property.
 * @return an iterator over the objects
 * @param p The predicate sought
 * @throws RDFException Generic RDF Exception
 */ 
    NodeIterator listObjectsOfProperty(Property p) throws RDFException;

/** List the values of a property of a resource.
 * @return an iterator over the objects
 * @param p The predicate sought
 * @throws RDFException Generic RDF Exception
 */ 
    NodeIterator listObjectsOfProperty(Resource s, Property p)
      throws RDFException;

/** Determine whether this model contains any statements with a given subject
 *  and property.
 * @return true if there exists within this model a statement with
 * subject s and property p, false otherwise
 * @param s The subject sought.
 * @param p The predicate sought.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, RDFNode o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, boolean o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, long o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, char o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, float o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, double o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, String o) throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @param l the language associated with the object
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, String o, String l)
       throws RDFException;

/** Determine if a statement is present in this model.
 * @return true if the statement with subject s, property p and object o
 * is in the model, false otherwise
 * @param s The subject of the statment tested.
 * @param p The predicate of the statement tested.
 * @param o The object of the statement tested.
 * @throws RDFException Generic RDF Exception
 */ 
    boolean contains(Resource s, Property p, Object o) throws RDFException;

/** Determine if a statement is present in this model.
 * @param s The statement tested.
 * @throws RDFException Generic RDF Exception
 * @return true if the statement s is in this model, false otherwise
*/ 
    boolean contains(Statement s) throws RDFException;

/** Determine if any of the statements returned by an iterator are
 * contained in this model.
 * @param iter an iterator of the statements to be tested
 * @throws RDFException Generic RDF Exception
 * @return true if any of the statements returns by iter are contained
 *         in this model and false otherwise.
*/ 
    boolean containsAny(StmtIterator iter) throws RDFException;

/** Determine if all of the statements returned by an iterator are
 * contained in this model.
 * @param iter an iterator of the statements to be tested
 * @throws RDFException Generic RDF Exception
 * @return true if any of the statements returns by iter are contained
 *         in this model and false otherwise.
*/ 
    boolean containsAll(StmtIterator iter) throws RDFException;

/** Determine if any of the statements in a model are also contained
 *  in this model.
 * @param model the model containing the statements to be tested
 * @throws RDFException Generic RDF Exception
 * @return true if any of the statements in model are also contained
 *         in this model and false otherwise.
*/ 
    boolean containsAny(Model model) throws RDFException;

/** Determine if all of the statements in a model are also contained
 *  in this model.
 * @param model the model containing the statements to be tested
 * @throws RDFException Generic RDF Exception
 * @return true if all of the statements in model are also contained
 *         in this model and false otherwise.
*/ 
    boolean containsAll(Model model) throws RDFException;

/** Determine if this statement is the subject of any statements in this model
 * @param s The statement tested.
 * @throws RDFException Generic RDF Exception
 * @return true if the statement s is the subject of a statement in the model,
             false otherwise
*/ 
    boolean isReified(Statement s) throws RDFException;

/** List the statements matching a selector.
 *
 * <p>A statment is considered to match if the <CODE>test</CODE> method
 * of s returns true when called on s.</p>
 * @return an iterator over the matching statements
 * @param s A selector object.
 * @throws RDFException Generic RDF exception.
 */ 
    StmtIterator listStatements(Selector s) throws RDFException;

/** Create a new model containing the statements matching a query.
 *
 * <p>A statment is considered to match if the <CODE>test</CODE> method
 * of s returns true when called on s.</p>
 * @return an iterator over the matching statements
 * @param s A selector object.
 * @throws RDFException Generic RDF exception.
 */ 
    Model query(Selector s) throws RDFException;

/** Create a new model containing all the statements in this model 
 * together with all of those in another given model.
 * @return A new model containing all the statements that are in either model
 * @param model The other model whose statements are to be included.
 * @throws RDFException Generic RDF Exception
 */     
    Model union(Model model) throws RDFException;

/** Create a new model containing all the statements which are in both
 * this model and another.  As models are sets of statements, a statement
 * contained in both models will only appear once in the resulting model.
 * @return A new model containing all the statements that are in both models.
 * @param model The other model.
 * @throws RDFException Generic RDF Exception
 */     
    Model intersection(Model model) throws RDFException;

/** Create a new model containing all the statements in this model which
 * are not in another.
 * @return a new model containing all the statements in this model that
 *         are not in the given model.
 * @param model the other model whose statements are to be excluded.
 * @throws RDFException Generic RDF Exception
 */     
    Model difference(Model model) throws RDFException;

/** Begin a new transation.
 *
 * <p> All changes made to a model within a transaction, will either
 * be made, or none of them will be made.</p>
 * @return this model to enable cascading.
 * @throws RDFException Generic RDF Exception
 */     
    Model begin() throws RDFException;

/** Abort the current transaction and abandon any changes in progress.
 * @return this model to enable cascading.
 * @throws RDFException Generic RDF Exception
 */     
    Model abort() throws RDFException;

/** Commit the current transaction.
 * @return this model to enable cascading.
 * @throws RDFException Generic RDF Exception
 */     
    Model commit() throws RDFException;

/** Determine whether this model is independent.
 *
 *  <p>For efficiency reasons, some implementations may create models which
 *  which are dependent on others, i.e. a change in one model may cause
 *  a change in another.  If this is the case this method will return false,
 *  otherwise it will return true.</p>
 *
 * @return  true if this model is indepdent of others
 */
    boolean independent();

/** Determine whether this model supports transactions.
 * @return  true if this model supports transactions.
 */
    boolean supportsTransactions();

/** Determine whether this model supports set operations.
 * @return true if this model supports set operations.
 */
    boolean supportsSetOperations();
}
