/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * RDFException.java
 *
 * Created on 26 July 2000, 07:00
 */

package Freenet.client.rdf;

import Freenet.contrib.xmlrpc.xml.sax.*;
import java.lang.Integer;

/** Generic RDF Exception class.
 * @author bwm
 */
public class RDFException extends SAXParseException {
    
    protected int       errorCode = 0;
    protected int       otherCode = 0;
    protected Exception nestedException = null;

    /** A method which would access a model has been invoked on an object
     * with no associated model.
     */
    public static final int NOTRELATEDTOMODEL                 =   1;
    /** The URI supplied for a Property is invalid.
     */
    public static final int INVALIDPROPERTYURI                =   2;
    /** A method which is unsupported has been called.
     */
    public static final int UNSUPPORTEDOPERATION              =   3;
    /** The object of a statement is not a Resource.
     */
    public static final int OBJECTNOTRESOURCE                 =   4;
    /** The object of a Statement is not a Literal.
     */
    public static final int OBJECTNOTLITERAL                  =   4;
    /** A property was not found in the model.
     */
    public static final int PROPERTYNOTFOUND                  =   5;
    /** The resource is no anonymous.
     */
    public static final int NOTANONRESOURCE                   =   6;
    /** The iterator is closed.
     */
    public static final int ITERATORCLOSED                    =   7;
    /** The error code is invalid.
     */
    public static final int INVALIDERRORCODE                  =   8;
    /** This exception contains another which has been caught and encapsulated as
     * an RDF exception.
     */
    public static final int NESTEDEXCEPTION                   =   9;
    /** A literal did not represent a boolean value.
     */
    public static final int INVALIDBOOLEANFORMAT              =  10;
    /** A literal did not contain a valid char.
     */
    public static final int LITERALNOTCHAR                    =  11;
    /** An Alt resource has no default value.
     */
    public static final int ALTHASNODEFAULT                   =  12;
    /** An application supplied Selector has raised an exception, which is
     * encapsulated in this exception.
     */
    public static final int SELECTOREXCEPTION                 =  13;
    /** The object of a statement is not of the expected type.
     */
    public static final int OBJECTWRONGTYPE                   =  14;
    /** A required element does not exist.
     */
    public static final int NOSUCHELEMENT                     =  15;
    /** Internal Error - an assertion has failed.
     */
    public static final int ASSERTIONFAILURE                  =  16;
    /** A sequence index is out of the valid range for that sequence.
     */
    public static final int SEQINDEXBOUNDS                    =  17;
    /** An enhanced resource does not have a constructor of the form
        foo(Resource r)
     */
    public static final int NORESOURCECONSTRUCTOR             =  18;
    
    protected static final int MAXERRORCODE                   =  18;

    
    protected static final String[] errorMessage = 
        { "Object is not related to a model",
          "Invalid property URI", 
          "Unsupported operation",
          "Object of statement is not a resource",
          "Object of statement is not a literal",
          "Subject does not have that property",
          "Resource is not anonymous",
          "Iterator has been closed",
          "Invalid error code",
          "Nested Exception",
          "Literal is not a boolean",
          "Literal is not a single character",
          "Alt container has no default",
          "Selector threw exception",
          "Statement object does match requested type",
          "Iterator does not have an element",
          "Assertion failure",
          "Sequence index out of range",
          "Enhanced Resource lacks Resource constructor"
        };
    
    public RDFException(String s) {super(s, null);}
    public RDFException(String s, Locator l) {super(s,l);}

    /** Create a new RDF exception with the given error code.
     * @param errorCode The code number of the error which has occurred.
     */
    public RDFException(int errorCode) {
        super(new Integer(errorCode).toString(), null);
        if (1<=errorCode && errorCode<=MAXERRORCODE) {
            this.errorCode = errorCode;
        } else {
            this.errorCode = INVALIDERRORCODE;
            this.otherCode = errorCode;
        }
    }
    
    /** Encapsulate an exception in an RDFException.
     * @param e The exception to be encapsulated.
     */
    public RDFException(Exception e) {
        super(e.toString(), null);
        errorCode = NESTEDEXCEPTION;
        nestedException = e;
    }
    
    /** Create a new RDFException with a given error code and encapsulate another
     * exception.
     * @param errorCode The code number of the error which has occurred.
     * @param e The exception to be encapsulated.
     */
    public RDFException(int errorCode, Exception e) {
        super(new Integer(errorCode).toString(), null);
        errorCode = SELECTOREXCEPTION;
        nestedException = e;
    }
    
    /** Return an error message describing the exception.
     * @return the error message
     */
    public String toString() {
        String message;
        
        message = "RDF Error code:" 
                + Integer.toString(errorCode)
                + " " + errorMessage[errorCode];        
        if (errorCode == INVALIDERRORCODE) {
            message = message + " = " + Integer.toString(otherCode);
        } else if (errorCode == NESTEDEXCEPTION
                || errorCode == SELECTOREXCEPTION) {
            message = message + " = " + nestedException.toString();
        }
        return message;
    }
    
    public int getErrorCode() {
        return errorCode;
    }
}
