/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ModelMem.java
 *
 * Created on 03 August 2000, 15:02
 */

package Freenet.client.rdf.impl;

import Freenet.client.rdf.util.*;
import Freenet.client.rdf.*;
import Freenet.client.rdf.vocabulary.RDF;

import java.io.FileReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.HashSet;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Vector;


/** A main memory implemention of the RDF API.
 *
 * <p>This implementation of the RDF API stores its all its data in main memory.
 *   It supports all methods except the transaction calls.</p>
 * <p>This is a prototype implementation, and should be considered to be alpha
 *   quality code.  It has not yet been performance optimised and is known to be
 *   wasteful of memory.  It has been tested by loading the wordnet database
 *   (see Sample3) which results in a model containing over 600,000 statements.
 *   </p>
 * @author bwm
 */
public class ModelMem extends Object implements Model, ModelI {    
    
    // next free error code = 3
    
    protected Store store = new Store();

    /** Create an empty model.
     */
    public ModelMem() {
    }
    
    public long size() throws RDFException {
        return store.size();
    }
    
    public ResIterator listSubjects() throws RDFException {
        try {
            HashSet subjects = new HashSet();
            Iterator iter = store.list();
            while (iter.hasNext()) {
                subjects.add(((Statement) iter.next()).getSubject());
            }
            return new ResIteratorImpl(subjects.iterator(), subjects);
        } catch (Exception e) {
            ErrorHelper.logInternalError(this.getClass().getName(), 1, e);
            throw new RDFException(e);
        }
    }
    
    public NsIterator listNameSpaces() throws RDFException {
        try {
            HashSet nameSpaces = new HashSet();
            Iterator iter = store.list();
            while (iter.hasNext()) {
                nameSpaces.add(((Statement) iter.next()).getPredicate()
                                                        .getNameSpace());
            }
            return new NsIteratorImpl(nameSpaces.iterator(), nameSpaces);
        } catch (Exception e) {
            ErrorHelper.logInternalError(this.getClass().getName(), 2, e);
            throw new RDFException(e);
        }
    }
    
    public StmtIterator listStatements() throws RDFException {
        return new StmtIteratorImpl(store.list(), null);
    }
    
    public StmtIterator listReifiedStatements() throws RDFException {
        Vector reifiedStatements = new Vector();
        Iterator iter = store.list();
        Object subject;
        while (iter.hasNext()) {
            subject = ((Statement) iter.next()).getSubject();
            if (subject instanceof Statement) {
                reifiedStatements.add(subject);
            }
        }
        return new StmtIteratorImpl(reifiedStatements.iterator(),
                                    reifiedStatements);
    }
    
    public Resource getResource(String uri) throws RDFException {
        return new ResourceImpl(uri, this);        
    }
    
    public Resource getResource(String uri, ResourceF f) throws RDFException {
        try {
            return f.createResource(new ResourceImpl(uri, this));
        } catch (Exception e) {
            throw new RDFException(e);
        }
    }
    
    public Property getProperty(String uri) throws RDFException {
        return new PropertyImpl(uri, this);
    }
    
    public Property getProperty(String nameSpace,String localName)
      throws RDFException {
        return new PropertyImpl(nameSpace, localName, this);                                                            
    }
    
    public Bag getBag(String uri) throws RDFException {
      return new BagImpl(uri, this);
    }
    public Bag getBag(Resource r) throws RDFException {
        return new BagImpl(r, this);
    }
    
    public Alt getAlt(String uri) throws RDFException {
        return new AltImpl(uri, this);
    }
    
    public Alt getAlt(Resource r) throws RDFException {
        return new AltImpl(r, this);
    }
    
    public Seq getSeq(String uri) throws RDFException {
        return new SeqImpl(uri, this);
    }
    public Seq getSeq(Resource r) throws RDFException {
        return new SeqImpl(r, this);
    }
    
    public Model add(Statement s) throws RDFException {
        if (! (s instanceof StatementImpl)
          || (((StatementImpl)s).getModel() != this) ) {
            s = new StatementImpl(s.getSubject(),
                                  s.getPredicate(), 
                                  s.getObject(),
                                  this);
        }
        store.add(s);
        return this;
    }
    
    public Model add(Resource s,Property p,RDFNode o) throws RDFException {
        store.add(new StatementImpl(s, p, o, this));
        return this;
    }
    
    public Model add(Resource s, Property p, boolean o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, long o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, char o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, float o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, double o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, String o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(Resource s, Property p, String o, String l)
       throws RDFException {
        add(s, p, new LiteralImpl(o,l));
        return this;
    }
    
    public Model add(Resource s, Property p, Object o) throws RDFException {
        add(s, p, new LiteralImpl(o));
        return this;
    }
    
    public Model add(StmtIterator iter) throws RDFException {
        try {
            while (iter.hasNext()) {
                add(iter.next());
            }
            return this;
        } finally {
            iter.close();
        }
    }
    
    public Model add(Model m) throws RDFException {
        return add(m.listStatements());
    }
    
    public Model read(String url) throws RDFException {
        RDFLoader.load(this, url);
        return this;
    }
    
    public Model read(Reader reader, String base) throws RDFException {
        RDFLoader.load(this, reader, base);
        return this;
    }
 
    public Model write(PrintWriter writer) throws RDFException {
        try {
            RDFWriter.write(this, writer);
            return this;
        } catch (Exception e) {
            throw new RDFException(e);
        }
    }
    
    public Model remove(Statement s) throws RDFException {
        store.remove(s);
        return this;
    }
    
    public Model remove(StmtIterator iter) throws RDFException {
        try {
            while (iter.hasNext()) {
                store.remove((Statement)iter.next());
            }
            return this;
        } finally {
            iter.close();
        }
    }
    
    public Model remove(Model m) throws RDFException {
        remove(m.listStatements());
        return this;
    }
    
    public boolean contains(Statement s) throws RDFException {
        return store.contains(s);
    } 
    
    public boolean contains(Resource s, Property p) throws RDFException {
        StmtIterator iter = null;
        try {
            iter = listStatements(new SelectorImpl(s, p, (RDFNode) null));
            return iter.hasNext();
        } finally {
            iter.close();
        }
    }
    
    public boolean contains(Resource s, Property p, RDFNode o)
      throws RDFException {
       return contains(new StatementImpl(s, p, o));
    } 
    
    public boolean contains(Resource s, Property p, boolean o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    }
    
    public boolean contains(Resource s, Property p, long o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    }
    
    public boolean contains(Resource s, Property p, char o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    }
    
    public boolean contains(Resource s, Property p, float o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    } 
    
    public boolean contains(Resource s, Property p, double o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    } 
    
    public boolean contains(Resource s, Property p, String o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    } 
    
    public boolean contains(Resource s, Property p, String o, String l)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o,l));
    }
    
    public boolean contains(Resource s, Property p, Object o)
      throws RDFException {
        return contains(s, p, new LiteralImpl(o));
    }
    
    public boolean containsAny(StmtIterator iter) throws RDFException {
        while (iter.hasNext()) {
            if (contains(iter.next())) return true;
        }
        return false;
    }
    
    public boolean containsAll(StmtIterator iter) throws RDFException {
        while (iter.hasNext()) {
            if (!contains(iter.next())) return false;
        }
        return true;
    }
    
    public boolean containsAny(Model model) throws RDFException {
        return containsAny(model.listStatements());
    }
    
    public boolean containsAll(Model model) throws RDFException {
        return containsAll(model.listStatements());
    }
    
    public boolean isReified(Statement s) throws RDFException {
        StmtIterator iter = null;
        try {
            iter = listStatements(new SelectorImpl(s, null, (RDFNode) null));
            if (iter.hasNext()) {
                return true;
            }
            iter = listStatements(new SelectorImpl(null, null, s));
            return iter.hasNext();
        } finally {
            iter.close();
        }
    }
    
    public Statement getProperty(Resource s,Property p) throws RDFException {
        StmtIterator iter = null;
        try {
            iter = listStatements(new SelectorImpl(s, p, (RDFNode) null));
            if (iter.hasNext()) {
                return iter.next();
            } else {
                throw new RDFException(RDFException.PROPERTYNOTFOUND);
            }
        } finally {
            iter.close();
        }
    }
    
    public ResIterator listSubjectsWithProperty(Property p) 
    throws RDFException {
        HashSet subjects = new HashSet();
        Iterator iter = store.listByPredicateHash(p);
        Statement stmt;
        while (iter.hasNext())
        {
            stmt = (Statement)(iter.next());
            if (stmt.getPredicate().equals(p))
            {
                subjects.add(stmt.getSubject());
            }
        }
        return new ResIteratorImpl(subjects.iterator(), subjects);            
    }
    public ResIterator listSubjectsWithProperty(Property p, boolean o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, long o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, char o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, float o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, double o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, String o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, String o, String l) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o,l));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, Object o) 
    throws RDFException {
        return listSubjectsWithProperty(p, new LiteralImpl(o));
    }
    
    public ResIterator listSubjectsWithProperty(Property p, RDFNode o) 
    throws RDFException {
        HashSet subjects = new HashSet();
        Iterator iter = store.listByPredicateHash(p);
        Statement stmt;
        while (iter.hasNext())
        {
            stmt = (Statement)(iter.next());
            if (stmt.getPredicate().equals(p) && stmt.getObject().equals(o))
            {
                subjects.add(stmt.getSubject());
            }
        }
        return new ResIteratorImpl(subjects.iterator(), subjects);         
    }
    
    public NodeIterator listObjects() throws RDFException {
        HashSet objects = new HashSet();
        Iterator iter = store.list();
        Statement stmt = null;
        while (iter.hasNext()) {
            objects.add(((Statement) iter.next()).getObject());
        }
        return new NodeIteratorImpl(objects.iterator(), objects);
    }
    
    public NodeIterator listObjectsOfProperty(Property p) throws RDFException {
        HashSet objects = new HashSet();
        Iterator iter = store.listByPredicateHash(p);
        Statement stmt = null;
        while (iter.hasNext()) {
            objects.add(((Statement) iter.next()).getObject());
        }
        return new NodeIteratorImpl(objects.iterator(), objects);         
    }
    
    public NodeIterator listObjectsOfProperty(Resource s, Property p) 
      throws RDFException {    
        HashSet objects = new HashSet();
        Iterator iter = store.listBySubjectHash(s);
        Statement stmt = null;
        while (iter.hasNext()) {
            stmt = (Statement) iter.next();
            if (stmt.getPredicate().equals(p))
            {
                objects.add(stmt.getObject());
            }
        }
        return new NodeIteratorImpl(objects.iterator(), objects);         
    }
    
    public StmtIterator listStatements(Selector selector)
      throws RDFException {
        Iterator iter;
        
        if (selector instanceof SelectorImpl) {
            Resource s = ((SelectorImpl)selector).getSubject();
            Property p = ((SelectorImpl)selector).getPredicate();
            RDFNode  o = ((SelectorImpl)selector).getObject();            
            if (s != null) {
                iter = store.listBySubjectHash(s);
            } else if (o != null) {
                iter = store.listByObjectHash(o);
            } else if (p != null) {
                iter = store.listByPredicateHash(p);
            } else {
                iter = store.list();
            }
        } else {
            iter = store.list();
        }
        
        if (iter == null) {
            return new StmtIteratorImpl((new HashSet()).iterator(), null);
        }
        
        return listStatements(selector, iter);
    }
    
    protected StmtIterator listStatements(Selector s, Iterator iter)
      throws RDFException {
        Vector matching = new Vector();
        Statement stmt;
        while (iter.hasNext()) {
            stmt = (Statement) iter.next();
            try {
                if (s.test(stmt)) {
                    matching.add(stmt);
                }
            } catch (Exception e) {
                throw new RDFException(RDFException.SELECTOREXCEPTION, e);
            }
        }
        return new StmtIteratorImpl(matching.iterator(), matching);
    }
    
    public Model query(Selector selector) throws RDFException {
        ModelMem model = new ModelMem();
        StmtIterator iter = null;
        try {
            iter = listStatements(selector);
            while (iter.hasNext()) {
                model.add(iter.next());
            }
            return model;
        } finally {
            iter.close();
        }
    }
    
    public Model union(Model model) throws RDFException {
        return (new ModelMem()).add(this)
                               .add(model);
    }
    
    public Model intersection(Model model) throws RDFException {
        Model largerModel = this;
        Model smallerModel  = model;
        ModelMem resultModel = new ModelMem();
        StmtIterator iter = null;
        Statement stmt;
        
        if (model.size() > this.size()) {
            largerModel = model;
            smallerModel = this;
        }
        try {
            iter = smallerModel.listStatements();
            while (iter.hasNext()) {
                stmt = iter.next();
                if (largerModel.contains(stmt)) {
                    resultModel.add(stmt);
                }
            }
            return resultModel;
        } finally {
            iter.close();
        }
    }
    
    public Model difference(Model model) throws RDFException {
        ModelMem resultModel = new ModelMem();
        StmtIterator iter = null;
        Statement stmt;
        try {
            iter = listStatements();
            while (iter.hasNext()) {
                stmt = iter.next();
                if (! model.contains(stmt)) {
                    resultModel.add(stmt);
                }
            }
            return resultModel;
        } finally {
            iter.close();
        }
    }
    
    public Model begin() throws RDFException {
        throw new RDFException(RDFException.UNSUPPORTEDOPERATION);
    }
    
    public Model abort() throws RDFException {
        throw new RDFException(RDFException.UNSUPPORTEDOPERATION);
    }
    
    public Model commit() throws RDFException {
        throw new RDFException(RDFException.UNSUPPORTEDOPERATION);
    }
    
    public boolean independent() {
        return true;
    }
    public Resource createResource() throws RDFException {
        return new ResourceImpl(this);
    }
    public Resource createResource(Resource type) throws RDFException {
        return getResource(null).addProperty(RDF.type, type);
    }
    public Resource createResource(String uri) throws RDFException {
        return getResource(uri);
    }
    public Resource createResource(String uri,Resource type) 
    throws RDFException {
        return getResource(uri)
                   .addProperty(RDF.type, type);
    }
    public Resource createResource(ResourceF f) throws RDFException {
        return createResource("", f);
    }
    public Resource createResource(String uri, ResourceF f) throws RDFException {
        try {
            return f.createResource(new ResourceImpl(uri, this));
        } catch (Exception e) {
            throw new RDFException(RDFException.NESTEDEXCEPTION, e);
        }
    }
    public Property createProperty(String uri) throws RDFException {
        return new PropertyImpl(uri, this);
    }
    public Property createProperty(String nameSpace, String localName)
    throws RDFException {
        return new PropertyImpl(nameSpace, localName, this);
    }
    public Literal createLiteral(boolean v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(long v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(char v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(float v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(double v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(String v) throws RDFException {
        return new LiteralImpl(v);
    }
    public Literal createLiteral(String v, String l) throws RDFException {
        return new LiteralImpl(v, l);
    }
    public Literal createLiteral(Object v) throws RDFException {
        return new LiteralImpl(v);        
    }
    public Statement createStatement(Resource r, Property p, RDFNode o) 
    throws RDFException {
        return new StatementImpl(r, p, o, this);
    }
    public Statement createStatement(Resource r, Property p, boolean o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, long o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, char o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, float o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, double o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, String o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Statement createStatement(Resource r, Property p, String o, String l) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o,l));
    }
    public Statement createStatement(Resource r, Property p, Object o) 
    throws RDFException {
        return createStatement(r, p, new LiteralImpl(o));
    }
    public Bag createBag() throws RDFException {
        return createBag(null);
    }
    public Bag createBag(String uri) throws RDFException {
        Bag bag = new BagImpl(uri, this);
        bag.addProperty(RDF.type, RDF.Bag);
        return bag;
    }
    public Alt createAlt() throws RDFException {
        return createAlt(null);
    }
    public Alt createAlt(String uri) throws RDFException {
        Alt alt = new AltImpl(uri, this);
        alt.addProperty(RDF.type, RDF.Alt);
        return alt;
    }
    public Seq createSeq() throws RDFException {
        return createSeq(null);
    }
    public Seq createSeq(String uri) throws RDFException {
        Seq seq = new SeqImpl(uri, this);
        seq.addProperty(RDF.type, RDF.Seq);
        return seq;
    }
        
    public NodeIterator listContainerMembers(Container cont,
                                             NodeIteratorFactory f)
                                                  throws RDFException {
        Iterator iter = store.listBySubjectHash(cont);
        Statement    stmt;
        String       rdfURI = RDF.getURI();
        Vector       result = new Vector();
        int          maxOrdinal = 0;
        int          ordinal;
        while (iter.hasNext()) {
            stmt = (Statement) iter.next();
            ordinal = stmt.getPredicate().getOrdinal();
            if (stmt.getSubject().equals(cont) && ordinal != 0) {
                if (ordinal > maxOrdinal) {
                    maxOrdinal = ordinal;
                    result.setSize(ordinal);
                }
                result.setElementAt(stmt, ordinal-1); 
            }
        }
        try {
             return f.createIterator(result.iterator(), result, cont);
        } catch (Exception e) {
            throw new RDFException(e);
        }
    }
    
    public int containerSize(Container cont) throws RDFException {
        int result = 0;
        Iterator iter = store.listBySubjectHash(cont);
        Property     predicate;
        Statement    stmt;
        String       rdfURI = RDF.getURI();
        while (iter.hasNext()) {
            stmt = (Statement) iter.next();
            predicate = stmt.getPredicate();
            if (stmt.getSubject().equals(cont)
             && predicate.getOrdinal() != 0
               ) {
                result++;
            }
        }
        return result;
    }
    
    public int containerIndexOf(Container cont, RDFNode n) throws RDFException {
        int result = 0;
        Iterator iter = store.listBySubjectHash(cont);
        Property     predicate;
        Statement    stmt;
        String       rdfURI = RDF.getURI();
        while (iter.hasNext()) {
            stmt = (Statement) iter.next();
            predicate = stmt.getPredicate();
            if (stmt.getSubject().equals(cont)
             && predicate.getOrdinal() != 0
             && n.equals(stmt.getObject())
              ) {
                return predicate.getOrdinal();
            }
        }
        return 0;
    }
    
   public boolean containerContains(Container cont, RDFNode n)
    throws RDFException {
        return containerIndexOf(cont, n) != 0;
    }
    
    public Resource convert(Resource r) throws RDFException {
        if (r.getModel() == this) {
            return r;
        } else {
            return ((ResourceI)r).port(this);
        }
    }
    
    public Property convert(Property p) throws RDFException {
        if (p.getModel() == this) {
            return p;
        } else {
            return (Property) ((ResourceI)p).port(this);
        }
    }
    
    public RDFNode convert(RDFNode n) throws RDFException {
        if (n instanceof Property) {
            return convert((Property) n);
        } else if (n instanceof Resource) {
            return convert((Resource) n);
        } else {
            return n;
        }
    }
    
    public boolean supportsTransactions() {return false;}
    
    public boolean supportsSetOperations() {return true;}
}
