/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * EnhResource.java
 *
 * Created on 16 August 2000, 14:06
 */

package Freenet.client.rdf.util;

import Freenet.client.rdf.*;

/** A base class for supporting enhanced resources.  An enhanced resource is a
 * resource with extended functionality, provided either by the application or
 * by the implementation.  The RDF containers provided by this implementation
 * are examples of enhanced resources.
 *
 * <p>Enhanced resources are implemented using encapsulation.  An enhanced
 *   resource contains an embedded resource which provides the basic
 *   functionality of the resource.  The application code provides any extended
 *   behaviour.  This method was chosen, rather than subclassing, so as not
 *   to constrain the applications need to create its own class hierarchy.</p>
 *
 * <p>This class is intended to be subclassed by the application to provide the
 *   specific enhanced behaviour.  It provides some basic housekeeping
 *   functions to support the encapsulation.</p>
 * @author bwm
 */
public class EnhResource implements Resource, ResourceI {
    
    static final String rc = Util.CLASSPATH + ".model.Resource";
        
    protected Resource resource;

    public EnhResource(String uri, Model m) throws RDFException {
        resource = m.createResource(uri);
    }
    
    public EnhResource(Model m) throws RDFException {
        resource = m.createResource();
    }

    protected EnhResource(Resource r) {
        resource = r;
    }

    public int     getId() throws RDFException {
        return resource.getId();
    }
    public String  getURI() {
        return resource.getURI();
    }
    public String  toString() {
        return resource.toString();
    }
    public int hashCode() {
        return resource.hashCode();
    }
    public boolean equals(Object o) {
        return ( o instanceof EnhResource) 
             && resource.equals(((EnhResource)o).resource);
    }
    public boolean isAnon() {
        return resource.isAnon();
    }
    public Statement getProperty(Property p) throws RDFException {
        return resource.getProperty(p);
    }
    public StmtIterator listProperties(Property p) throws RDFException {
        return resource.listProperties(p);
    }
    public StmtIterator listProperties() throws RDFException {
        return resource.listProperties();
    }
    public Resource addProperty(Property p, boolean o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, byte o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, short o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, int o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, long o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, char o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, float o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, double o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, String o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, String o, String l)
      throws RDFException {
        return resource.addProperty(p, o, l);
    }
    public Resource addProperty(Property p, Object o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public Resource addProperty(Property p, RDFNode o) throws RDFException {
        return resource.addProperty(p, o);
    }
    public boolean hasProperty(Property p) throws RDFException {
        return resource.hasProperty(p);
    }
    public boolean hasProperty(Property p, boolean o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, byte o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, short o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, int o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, long o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, char o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, float o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, double o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, String o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, String o, String l)
       throws RDFException {
        return resource.hasProperty(p, o, l);
    }
    public boolean hasProperty(Property p, Object o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public boolean hasProperty(Property p, RDFNode o) throws RDFException {
        return resource.hasProperty(p, o);
    }
    public Resource removeProperties() throws RDFException {
        return resource.removeProperties();
    }
    public Resource begin() throws RDFException {
        return resource.begin();
    }
    public Resource abort() throws RDFException {
        return resource.abort();
    }
    public Resource commit() throws RDFException {
        return resource.commit();
    }
    public Model getModel() {
        return resource.getModel();
    }        
    public Resource getEmbeddedResource() {
        return ((ResourceI)resource).getEmbeddedResource();
    }    
    public Resource port(Model m) throws RDFException {
        if (resource.getModel()==m) {
            return this;
        } else {
            try {
                Class[] paramTypes = {Class.forName(rc)};
                Object[] params = {((ResourceI)resource).port(m)};
                return (Resource) this.getClass()
                                      .getConstructor(paramTypes)
                                      .newInstance(params);
            } catch (java.lang.NoSuchMethodException e1) {
                throw new RDFException(RDFException.NORESOURCECONSTRUCTOR);
            } catch (Exception e2) {
                throw new RDFException(e2);
            }
        }
    }
}
