/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * PropertyImpl.java
 *
 * Created on 03 August 2000, 13:47
 */

package Freenet.client.rdf.util;

import Freenet.client.rdf.*;
import Freenet.client.rdf.vocabulary.RDF;
import java.lang.Integer;
import java.lang.NumberFormatException;


/** An implementation of Property.
 *
 * @author  bwm
 */

// next free error code = 2

public class PropertyImpl extends ResourceImpl implements Property {
    
    protected String nameSpace;
    protected String localName;
    protected int    ordinal   = 0;

    /** Creates new PropertyImpl */
    public PropertyImpl(String uri) throws RDFException {
        super(uri);
        splitNameSpace();
        checkOrdinal();
    }
    
    public PropertyImpl(String nameSpace, String localName)
    throws RDFException {
        super(nameSpace + localName);
        this.nameSpace = nameSpace;
        this.localName = localName;
        checkOrdinal();
    }
    
    public PropertyImpl(String uri, Model m) throws RDFException {
        super(uri, m);
        splitNameSpace();
        checkOrdinal();
    }
    
    public PropertyImpl(String nameSpace, String localName, Model m) 
    throws RDFException {
        super(nameSpace + localName, m);
        this.nameSpace = nameSpace;
        this.localName = localName;
        checkOrdinal();
    }
    
    public PropertyImpl(String nameSpace,
                        String localName,
                        int ordinal,
                        Model m) throws RDFException {
        super(nameSpace + localName, m);
        this.nameSpace = nameSpace;
        this.localName = localName;
        this.ordinal = ordinal;
    }
     
    public String getNameSpace() {
      return nameSpace;
    }
    public String getLocalName() {
      return localName;
    }
    
    public int getOrdinal() {
        return ordinal;
    }
    
    protected void splitNameSpace() throws RDFException {
        int splitPoint = Util.splitNamespace(uri);
        if (splitPoint == 0) {
            throw new RDFException(RDFException.INVALIDPROPERTYURI);
        } else {
            nameSpace = uri.substring(0, splitPoint);
            localName = uri.substring(splitPoint);
        }
    }
    
    protected void checkOrdinal()
    {
        char c;
        
        // check for an rdf:_xxx property
        
        if (   localName.charAt(0) == '_' && nameSpace.equals(RDF.getURI()) 
            && nameSpace.equals(RDF.getURI())
            && localName.length() > 1) {
            for (int i=1; i<localName.length(); i++) {
                c = localName.charAt(i);
                if (c < '0'  || c > '9') {
                    return;
                }
            }
            try {
              ordinal = Integer.parseInt(localName.substring(1));
            } catch (NumberFormatException e) {
                ErrorHelper.logInternalError(this.getClass().getName(), 1);
            }
        }
    }    
    
    public Resource port(Model m) throws RDFException {
        if (model == m) {
            return this;
        } else {
            PropertyImpl p = new PropertyImpl(nameSpace, localName, ordinal, m);
            p.id = id;
            return p;
        }
    }
}
