/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * RDFLoader.java Uses David Megginson's RDF Filter
 *                to parse an xml file and add to a model
 *
 * Created on 20 August 2000, 10:10
 */

package Freenet.client.rdf.util;



import Freenet.client.rdf.*;
import Freenet.client.rdf.vocabulary.RDF;

import Freenet.client.rdf.RDFHandler;
import Freenet.client.rdf.RDFReader;

import Freenet.contrib.xmlrpc.xml.sax.Attributes;
import Freenet.contrib.xmlrpc.xml.sax.ContentHandler;
import Freenet.contrib.xmlrpc.xml.sax.Parser;
import Freenet.contrib.xmlrpc.xml.sax.XMLReader;
import Freenet.contrib.xmlrpc.xml.sax.helpers.ParserAdapter;
import Freenet.contrib.xmlrpc.xml.sax.helpers.ParserFactory;
import Freenet.contrib.xmlrpc.xml.sax.helpers.XMLReaderFactory;

import java.io.Reader;
import java.util.HashMap;

/** Uses David Megginson's RDF Filter to load an XML serialization into a model.
 *
 * <p>Uses the Sax2 parser specified by the Freenet.contrib.xmlrpc.xml.sax.driver system property.
 *    If this is not set, uses the Apache Xerces parser by default.</p>
 *
 * @author  bwm
 */
public class RDFLoader extends Object implements RDFHandler {
    
    // next available error code = 3
        
    static final String saxDriverPropName = "sax.driver";
    static final String defaultSaxDriverProp = "Freenet.contrib.xmlrpc.xml.sax.helpers.ParserAdapter";
    static final String namespacePrefixPropName = "http://org.xml.sax/features/namespace-prefixes";
    static final String namespacePropName = "http://org.xml.sax.features/namespace";
    
    XMLContentHandler contentHandler = new XMLContentHandler();
    int subjectType;
    String subject;
    String predicate;
    String language;
    String base;

    Model model;                            // the model being loaded
    HashMap anonResources = new HashMap();

    RDFLoader(Model model, String base) {
        this.model = model;
        this.base = base;
    } 
    
    protected static RDFReader createRDFReader(Model model, String base)
      throws Exception {
        String parserClassName = System.getProperty(saxDriverPropName,
                                                     defaultSaxDriverProp);    
        XMLReader xmlReader = XMLReaderFactory.createXMLReader(parserClassName);
        RDFReader rdfReader = new RDFReader(xmlReader);
        RDFLoader rdfLoader = new RDFLoader(model, base);
        rdfReader.setRDFHandler(rdfLoader);
        rdfReader.setContentHandler(rdfLoader.getContentHandler());
        return rdfReader;
    }
   
    protected XMLContentHandler getContentHandler() {
        return contentHandler;
    }
    
    public static void load(Model model, Reader reader, String base)
      throws RDFException {
        try {
            createRDFReader(model, base).readRDF(reader);
        } catch (Exception e) {
            throw new RDFException(e);
        }
    } 
    
    public static void load(Model model, String url)
      throws RDFException {
        try {
            createRDFReader(model, url).readRDF(url);
        } catch (Exception e) {
            throw new RDFException(e);
        }
    }
    
    public void literalStatement(int subjectType, String subject,
                                  String predicate, String object,
                                  String language) {
        if (language==null) language="";
        try {
            if (subjectType==SUBJECT_URI) {
              model.add(model.createResource(translateURI(subject)),
                        model.createProperty(translateURI(predicate)),
                        object, language);
            } else if (subjectType==SUBJECT_GENERATED) {
                model.add(getAnonResource(subject),
                          model.createProperty(translateURI(predicate)),
                          object, language);
            } else {
                ErrorHelper.logWarning(
                              "subject type not supported: " + subject);
            }
        } catch (RDFException e) {
            ErrorHelper.logInternalError("RDFLoader", 1, e);
        }
    }
    
    public void resourceStatement(int subjectType, String subject, 
                                    String predicate, String object) {
        try {
            Resource s;
            Resource o;
            if (subjectType==SUBJECT_URI) {
                s = model.createResource(translateURI(subject));
            } else if (subjectType==SUBJECT_GENERATED) {
                s = getAnonResource(subject);
            } else {
                ErrorHelper.logWarning(
                               "subject type not supported: " + subject);
                return;
            }
            
                                              // FIXME: kludge here to deal
                                              // with RDFFilter problem.  RDFF
                                              // doesn't tell us if the object
                                              // is an anon resource id
                                              // so we judge by looking at
                                              // the first character.
            if (object.charAt(0) == '{') {
                o = getAnonResource(object);
            } else {
                o = model.createResource(translateURI(object));
            }
            model.add(s, model.createProperty(translateURI(predicate)), o);
        } catch (RDFException e) {
            ErrorHelper.logInternalError("RDFLoader", 2, e);
        }
    }
    
    public void startXMLStatement(int subjectType, String subject,
                                   String predicate, String language) {
        this.subjectType = subjectType;
        this.subject = subject;
        this.predicate = predicate;
        this.language = language;
        contentHandler.activate();
    }
    
    public void endXMLStatement()
    {
        literalStatement(subjectType, subject, predicate, 
                            contentHandler.deActivate(), language);        
    }

    protected Resource getAnonResource(String id) throws RDFException {
        Resource resource;
        
        resource = (Resource) anonResources.get(id);
        if (resource==null) {
            resource = model.createResource();
            anonResources.put(id, resource);
        }
        return resource;
    }
    
    protected String translateURI(String uri) {
        if (uri.charAt(0) == '#') {
            return base + uri.substring(1);
        } else {
            return uri;
        }
    }
}

class XMLContentHandler implements ContentHandler {
    
    String content;                     // accumulator of the XML content
    boolean active;                     // whether the accumulator is active
    boolean elementEmpty=false;         // whether current element is empty
    
    public void activate() {
        active=true;
        content="";
    }
    
    public String deActivate() {
        String result = content;
        active = false;
        content = null;
        return result;
    }
    
    public void characters(char[] cb, int start, int length) {
        if (!active) return;
        closeOpenElement();
        content = content + 
            Util.substituteStandardEntities(String.valueOf(cb, start, length));
    }
    
    public void ignorableWhitespace(char[] cb, int start, int length) {
        characters(cb, start, length);
    }    
    
    public void startElement(String nameSpaceURI, String localName,
                               String qname, Attributes atts) {
                                
       if (!active) return;
       closeOpenElement();
       content = content + "<" + qname;
       for (int i=0; i<atts.getLength(); i++) {
           content = content
                       + " " + atts.getQName(i) + "=\""
                       + Util.replace(atts.getValue(i), "\"", "&quote;") + "\"";
       }
       elementEmpty = true;
    }
    
    public void endElement(String namespaceURI, String localName,
                             String qname) {
        if (!active) return;
        if (elementEmpty) {
            content = content + "/>";
        } else {
            content = content + "</" + qname + ">";
        }
        elementEmpty = false;
    }
    
    public void processingInstruction(String target, String data) {
        if (!active) return;
        closeOpenElement();
        content = content + "<?" + target + " " + data + "?>";
    }
    
    public void startDocument() {}
    public void endDocument() {}
    public void startPrefixMapping(String prefix, String uri) {}
    public void endPrefixMapping(String prefex) {}
    public void skippedEntity(java.lang.String name) {}
    public void setDocumentLocator(Freenet.contrib.xmlrpc.xml.sax.Locator locator) {}
    
    void closeOpenElement() {
        if (elementEmpty) {
            content = content + ">";
            elementEmpty = false;
        }
    }
}
