/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * RDFWriter.java
 *
 * Created on 29 July 2000, 07:28
 */

package Freenet.client.rdf.util;


import Freenet.client.rdf.*;
import Freenet.client.rdf.vocabulary.RDF;
import java.io.PrintWriter;
import java.util.Vector;
import java.util.Enumeration;
import java.util.HashSet;

/** Writes out an XML serialization of a model.
 *
 * @author  bwm
 */
public class RDFWriter extends Object {

    /** Write out an XML serialization of a model.
     * @param model the model to be serialized
     * @param writer the PrintWriter to receive the serialization
     * @throws IOException if an io error occurs
     * @throws RDFException if any other exception occurs
     */
    public static void write(Model model, PrintWriter writer)
    throws java.io.IOException, RDFException {

        Vector nsv = listNameSpaces(model);
        
        writeRDFHeader(nsv, writer);
        writeRDFStatements(model, nsv, writer);
        writeRDFTrailer(writer);
        writer.flush();
    }
    
    protected static Vector listNameSpaces(Model model) throws RDFException {
        Vector nsv = new Vector();
        NsIterator nsIter = model.listNameSpaces();
        while (nsIter.hasNext()) {
          nsv.add(nsIter.next());
        }
        return nsv;
    }
    
    protected static void writeRDFHeader(Vector nsv, PrintWriter writer) {        
        Enumeration nsEnum = nsv.elements();
        String ns;
        
        writer.print("<rdf:RDF");
        writer.print("\r\n  xmlns:rdf='" + RDF.getURI() + "'");
        while (nsEnum.hasMoreElements()) {
            ns = (String) nsEnum.nextElement();
            // another kludge for rdf namespace
            if (!ns.equals(RDF.getURI())) {
                writer.print("\r\n  xmlns:" + nsId(nsv, ns) 
                                               + "='" + ns + "'");
            }
        }
        writer.print(" >\r\n");
    }
    
    protected static void 
        writeRDFStatements(Model model, Vector nsv, PrintWriter writer)
            throws RDFException {
        ResIterator rIter = model.listSubjects();
        while (rIter.hasNext()) {
            writeRDFStatements(model, rIter.next(), nsv, writer);
        }
    }
    
    protected static void writeRDFTrailer(PrintWriter writer) {
        writer.print("</rdf:RDF>\r\n");
    }
    
    protected static void 
      writeRDFStatements(Model model, Resource subject,
                              Vector nsv, PrintWriter writer) 
      throws RDFException {
        StmtIterator sIter = 
          model.listStatements(new SelectorImpl(subject, null, (RDFNode) null));
        
        writeDescriptionHeader(subject, writer);
        if ( (subject instanceof Statement) 
           && ! model.contains((Statement)subject)) {
            // an unasserted reified statement
            writeReifiedProperties((Statement) subject, writer);
        }
        while (sIter.hasNext()) {
            writePredicate(sIter.next(), nsv, writer);
        }
        writeDescriptionTrailer(writer); 
        
        // if the subject of subject is a reified statement not in the model
        // need to write it out too
        if (subject instanceof Statement) {
            Resource innerSubject = ((Statement)subject).getSubject();
            if (innerSubject instanceof Statement
             && ! model.contains((Statement) innerSubject)) {
                writeRDFStatements(model, innerSubject, nsv, writer);
            }
            RDFNode innerObject = ((Statement)subject).getObject();
            if (innerObject instanceof Statement
             && ! model.contains((Statement) innerObject)) {
                writeRDFStatements(model, (Resource) innerObject, nsv, writer);
            }
        }
    }
    
    protected static void
        writeDescriptionHeader(Resource subject, PrintWriter writer) 
            throws RDFException {
        writer.print("  <rdf:Description ");
        writeResourceId(subject, writer);
        writer.print(">\r\n");
    }
    
    protected static void 
        writePredicate(Statement stmt, Vector nsv, PrintWriter writer)
            throws RDFException {
                
        Property predicate = stmt.getPredicate();
        RDFNode object = stmt.getObject();
    
        writer.print("    <" + transUri(nsv, predicate));
        
        if (stmt.isReified()) {
            writer.print(" ID='" + anonId(stmt) + "'");
        }
      
        if (object instanceof Resource) {
            writer.print(" ");
            writeResourceReference(((Resource)object), writer);
            writer.print("/>\r\n");
        } else {
            writeLiteral((Literal) object, writer);
            writer.print("</" + transUri(nsv, predicate) + ">\r\n");
        }
    }
    
    protected static void writeDescriptionTrailer(PrintWriter writer) {
        writer.print("  </rdf:Description>\r\n");
    }
    
    protected static void 
      writeReifiedProperties(Statement stmt, PrintWriter writer) 
      throws RDFException {
        writer.print("    <rdf:type rdf:resource='" 
                       + RDF.getURI() + "Statement'/>\r\n");
        writer.print("    <rdf:subject ");
        writeResourceReference(stmt.getSubject(), writer);
        writer.print("/>\r\n");
        writer.print("    <rdf:predicate rdf:resource='" 
                             + stmt.getPredicate().getURI() + "'/>\r\n");
        writer.print("    <rdf:object ");
        
        RDFNode object = stmt.getObject();
        if (object instanceof Resource) {
            writeResourceReference((Resource) stmt.getObject(), writer);
            writer.print("/>\r\n");
        } else {
            writeLiteral((Literal) object, writer);
            writer.print("</rdf:object>\r\n");
        }
    }
    
    static protected void writeResourceId(Resource r, PrintWriter writer) 
      throws RDFException {
        writer.print("rdf:about='");
        if (r.isAnon()) {
            writer.print("#" + anonId(r));
        } else {
            writer.print(r.getURI());
        }
        writer.print("'");
    }
    
    static protected void writeResourceReference(Resource r, PrintWriter writer) 
      throws RDFException {
        writer.print("rdf:resource='");
        if (r.isAnon()) {
            writer.print("#" + anonId(r));
        } else {
            writer.print(r.getURI());
        }
        writer.print("'");
    }
    
    static protected void writeLiteral(Literal l, PrintWriter writer) {
        String lang = l.getLanguage();
        if (! lang.equals("")) {
            writer.print(" xml:lang=\'" + lang + "'");
        }
        writer.print(">");
        writer.print(Util.substituteStandardEntities(l.toString()));
    }
        
  
    static protected String anonId(Resource r) throws RDFException {
      return "RDFAnonId" +  r.getId();
    }
  
    static protected String transUri(Vector nsv, Property p) {
      String t = p.getNameSpace();
      t = nsId(nsv, t) + ":";
      return t + p.getLocalName();
    }
  
    static protected String nsId(Vector nsv, String ns) {
      // got a kludge here to fix up rdf:
      if (ns.equals(RDF.getURI())) {
        return "rdf";
      } else {
        return "RDFNsId" + nsv.indexOf(ns);
      }
    }
}
