/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ResourceImpl.java
 *
 * Created on 03 August 2000, 13:45
 */

package Freenet.client.rdf.util;

import Freenet.client.rdf.*;

import java.util.Random;

/** An implementation of Resource.
 *
 * @author  bwm
 */
public class ResourceImpl implements Resource, ResourceI {
    
    String     uri;    
    Model      model = null;
    int        id    = 0;
    static int nextId = 1;
    
    /** Creates new ResourceImpl */
    public ResourceImpl() {
        uri = "";
        assignId();
    }
    
    public ResourceImpl(String uri)
    {
        this.uri = checkURI(uri);
    }
    
    public ResourceImpl(Model m) throws RDFException {
        if (m == null) {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
        uri = "";
        model = m;
        assignId();
    }
    
    public ResourceImpl(String uri, Model m) throws RDFException {
        if (m == null) {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
        this.uri = checkURI(uri);
        model = m;
        if (uri == null || uri.equals("")) {
            assignId();
        }
    }
    
    public ResourceImpl(Resource r, Model m) throws RDFException {
        uri = r.getURI();
        id = r.getId();
        model = m;
    }
    
    protected String checkURI(String uri) {
        if (uri == null) {
            return "";
        } else {
            return uri;
        }
    }
    
    protected void assignId() {
        id = nextId++;
    }
    
    public int getId() throws RDFException {
        return id;
    }
    
    public String  getURI() {
        return uri;
    }
    public String  toString() {
        if (uri == null || uri.equals("")) {
          return "[" + id + "]";
        } else {
            return getURI();
        }
    }
    
    public boolean equals(Object o) {
        if (o instanceof Resource) {
            if (uri == null || uri.equals("")) {
                Resource r = ((ResourceI) o).getEmbeddedResource();
                if (r instanceof ResourceImpl) {
                    return id == ((ResourceImpl)r).id;
                } else {
                    return false;
                }
            }
            return uri.equals(((Resource) o).getURI());
        } else {
            return false;
        }
    }
        
    public int hashCode() {
        if (isAnon()) {
            return id;
        } else {
            return uri.hashCode();
        }
    }
    
    public boolean isAnon() {
        return (uri == null || uri.equals(""));
    }
    
    public Statement getProperty(Property p) throws RDFException {
        if (model != null) {
            return model.getProperty(this, p);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public StmtIterator listProperties(Property p) throws RDFException {
        if (model != null) {
            return model.listStatements(
                                     new SelectorImpl(this, p, (RDFNode) null));
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }        
    }
    
    public StmtIterator listProperties() throws RDFException {
        if (model != null) {
            return model.listStatements(
                                  new SelectorImpl(this, null, (RDFNode) null));
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Resource addProperty(Property p, boolean o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, long o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, char o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, float o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, double o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, String o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, String o, String l)
      throws RDFException {
        if (model != null) {
            model.add(this, p, o, l);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, Object o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource addProperty(Property p, RDFNode o) throws RDFException {
        if (model != null) {
            model.add(this, p, o);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public boolean hasProperty(Property p) throws RDFException {
        if (model != null) {
            return model.contains(this, p);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, boolean o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, long o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, char o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, float o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, double o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, String o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, String o, String l)
      throws RDFException {
        if (model != null) {
            return model.contains(this, p, o, l);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, Object o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public boolean hasProperty(Property p, RDFNode o) throws RDFException {
        if (model != null) {
            return model.contains(this, p, o);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Resource removeProperties() throws RDFException {
        if (model != null) {
            StmtIterator iter = null;
            iter = model.listStatements(
                                 new SelectorImpl(this, null, (RDFNode) null));
            while (iter.hasNext()) {
                iter.next();
                iter.remove();
            }
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Resource begin() throws RDFException {
        if (model != null) {
            model.begin();
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource abort() throws RDFException {
        if (model != null) {
            model.abort();
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    public Resource commit() throws RDFException {
        if (model != null) {
            model.commit();
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Resource port(Model m) throws RDFException {
        if (model == m) {
            return this;
        } else {
            return new ResourceImpl(this, m);
        }
    }
    public Model getModel() {
        return model;
    }
    public Resource getEmbeddedResource() {
        return this;
    }
}
