/*
 *  (c) Copyright Hewlett-Packard Company 2000 
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * StatementImpl.java
 *
 * Created on 03 August 2000, 13:58
 */

package Freenet.client.rdf.util;

import Freenet.client.rdf.*;
import Freenet.client.rdf.vocabulary.RDF;

import java.lang.Byte;
import java.lang.Short;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Character;
import java.lang.Float;
import java.lang.Double;
import java.lang.NumberFormatException;

/** An implementation of Statement.
 *
 * @author  bwm
 */
public class StatementImpl extends ResourceImpl implements Statement {
    
    protected Resource subject;
    protected Property predicate;
    protected RDFNode  object;
    
    public StatementImpl(Resource subject, Property predicate, RDFNode object) {
        super();
        this.subject = subject;
        this.predicate = predicate;
        this.object = object;
    }
    
    /** Creates new StatementMem */
    public StatementImpl(Resource subject,
                         Property predicate,
                         RDFNode object,
                         Model model) throws RDFException {
        super(model);
        this.subject=((ModelI)model).convert(subject);
        this.predicate = ((ModelI)model).convert(predicate);
        this.object = ((ModelI)model).convert(object);
    }
    
    public int hashCode() {
        return subject.hashCode() ^ predicate.hashCode() ^ object.hashCode();
    }
    
    public Resource getSubject() {
        return subject;
    }
    public Property getPredicate() {
        return predicate;
    }
    public RDFNode getObject() {
        return object;
    }    
    public Statement getProperty(Property p) throws RDFException {
        if (object instanceof Resource) {
            return ((Resource) object).getProperty(p);
        } else {
            throw new RDFException(RDFException.OBJECTNOTRESOURCE);
        }
    }    
    public Statement getStatementProperty(Property p) throws RDFException {
        return super.getProperty(p);
    }
    public Resource getResource() throws RDFException {
        if (object instanceof Resource) {
            return (Resource) object;
        } else {
            throw new RDFException(RDFException.OBJECTNOTRESOURCE);
        }
    }
    public Literal getLiteral() throws RDFException {        
        if (object instanceof Literal) {
            return (Literal) object;
        } else {    
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public boolean getBoolean() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getBoolean();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public byte getByte() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getByte();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public short getShort() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getShort();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public int getInt() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getInt();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public long getLong() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getLong();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public char getChar() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getChar();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public float getFloat() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getFloat();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public double getDouble() throws RDFException {
        
        if (object instanceof Literal) {
            return ((Literal) object).getDouble();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public String getString() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).toString();
        } else {
            throw new RDFException(RDFException.OBJECTNOTLITERAL);
        }
    }
    public Resource getResource(ResourceF f) throws RDFException {
        if (object instanceof Resource) {
            try {
                return
                    f.createResource(((ResourceI)object).getEmbeddedResource());
            } catch (Exception e) {
                throw new RDFException(e);
            }
        } else {
            throw new RDFException(RDFException.OBJECTNOTRESOURCE);
        }
    }
    public Object getObject(ObjectF f) throws RDFException {
        try {
            return f.createObject(((Literal) object).toString());
        } catch (Exception e) {
            throw new RDFException(e);
        }
    }
    public Bag getBag() throws RDFException {
        if (object instanceof Bag) {
            return (Bag) object;
        } else if (object instanceof Resource
                && model.contains((Resource)object, RDF.type, RDF.Bag)) {
            return new BagImpl((Resource) object, model);
        } else {
            throw new RDFException(RDFException.OBJECTWRONGTYPE);
        }
    }
    public Alt getAlt() throws RDFException {
        if (object instanceof Alt) {
            return (Alt) object;
        } else if (object instanceof Resource
                && model.contains((Resource)object, RDF.type, RDF.Alt)) {
            return new AltImpl((Resource) object, model);
        } else {
            throw new RDFException(RDFException.OBJECTWRONGTYPE);
        }
    }
    public Seq getSeq() throws RDFException {
        if (object instanceof Seq) {
            return (Seq) object;
        } else if (object instanceof Resource
                && model.contains((Resource)object, RDF.type, RDF.Seq)) {
            return new SeqImpl((Resource) object, model);
        } else {            
            throw new RDFException(RDFException.OBJECTWRONGTYPE);
        }
    }    
    public String getLanguage() throws RDFException {
        if (object instanceof Literal) {
            return ((Literal) object).getLanguage();
        } else {
            throw new RDFException(RDFException.OBJECTWRONGTYPE);
        }
    }    
    public Statement set(boolean o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }
    public Statement set(long o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }
    public Statement set(char o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }
    public Statement set(float o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }
    public Statement set(double o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }
    public Statement set(String o) throws RDFException {
        replace(new LiteralImpl(o));
        return this;
    }  
    public Statement set(String o, String l) throws RDFException {
        replace(new LiteralImpl(o,l));
        return this;
    }      
    public Statement set(RDFNode o) throws RDFException {
        replace(o);
        return this;
    }    
    public Statement set(Object o) throws RDFException {
        replace(new LiteralImpl(o.toString()));
        return this;
    }
    
    protected void replace(RDFNode n) throws RDFException {
        if (model != null) {
            model.remove(this);
            model.add(subject, predicate, n);
            object = n;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
        
    public String toString() {
        String result = "[" + subject.toString() + ", "
                            + predicate.toString() + ", ";
        if (object instanceof Resource) {
            result = result + "Resource<" + ((Resource)object).toString() + ">";
        } else {
            result = result + "Literal<" + ((Literal)object).toString() + ">";
        }
        return result + "]";
    }
    
    public boolean equals(Object o) {
        return (o instanceof Statement)
            && (subject.equals(((Statement) o).getSubject()))
            && (predicate.equals(((Statement) o).getPredicate()))
            && (object.equals(((Statement) o).getObject()));
    }

    public Resource removeProperties() throws RDFException {
        if (model != null) {
            super.removeProperties();
            model.remove(this);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Statement remove() throws RDFException {
        if (model != null) {
            model.remove(this);
            return this;
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
            
    public boolean isReified() throws RDFException {
        if (model != null) {
            return model.isReified(this);
        } else {
            throw new RDFException(RDFException.NOTRELATEDTOMODEL);
        }
    }
    
    public Resource port(Model m) throws RDFException {
        if (model == m) {
            return this;
        } else {
            StatementImpl s = new StatementImpl(subject, predicate, object, m);
            s.id = id;
            return s;
        }
    }
}
