#!/usr/bin/perl -w

my $PARSEDATE;
use strict;
use IO::File;
use IO::Socket;
use IO::Select;
BEGIN {
    eval "use Time::ParseDate";
    if ($@) {
	warn "Time::ParseDate not found.  Disabling DateRedirects";
	$PARSEDATE = 0;
    } else {
	$PARSEDATE = 1;
    }
}
use Carp;
use POSIX qw(setsid);
use FCP;
use Data::Dumper;

my $SPLITSIZE = 256 * 1024; #the number of bytes to split files larger than
my $REDIRSIZE = 32 * 1024;  #the number of bytes to redirect to a CHK for
my $PARTSIZE = 256 * 1024;  #the size of parts to split into

my $HOST = "localhost";  #ip address of the FCP server
my $PORT = 8481;             #port the FCP server is running on
my $LISTENPORT = 7892;       #port the proxy should listen on

my $BUFSIZE = 65535; #size of buffer to use for reading/writing
my $CONCURRENCY = 5;  #the number of pieces of a splitfile to get at a time
my $LOGFILE = "/dev/null";
my $DEBUG = 1;
my $DAEMONIZE = 0;
my $TIMEZONE = 86400 * -6/24;
################### SHOULDN'T NEED TO CHANGE ANYTHING BELOW############

sub usage {
    print "$0 version 0.4
Copyright (C) 2001 Eric Norige
This is free software and may be redistributed under the terms of the GNU GPL.

usage: $0 [options]
  --help             - print this help message
  --port (-p)        - the port of the FCP server  [Def: 8481]
  --listenport (-l)  - the port this proxy listens on  [Def: 7892]
  --concurrency (-c) - the number of concurrent requests to make [Def: 4]
  --daemonize (-D)   - use this to make the proxy fork into the background
";
}

my @nextargv;
for (@ARGV) {
    /^--help$/     and do { &usage; exit 1 };
    /^--host$/     and do { push @nextargv, \$HOST; next };
    /^--port$/     and do { push @nextargv, \$PORT; next };
    /^--listenport$/ and do { push @nextargv, \$LISTENPORT; next };
    /^--concurrency$/ and do { push @nextargv, \$CONCURRENCY; next };
    /^--daemonize$/ and do { $DAEMONIZE = 1; next };
    /^-(.*)$/ and do {foreach (split //, $1) {
	/[\?]/ and do { &usage; exit 1 };
	/h/ and do {push @nextargv, \$HOST; next };
	/p/ and do {push @nextargv, \$PORT; next };
	/l/ and do {push @nextargv, \$LISTENPORT; next };
	/c/ and do {push @nextargv, \$CONCURRENCY; next };
	/D/ and do { $DAEMONIZE = 1; next };
#	/v/ and do {$VERB++; next };  #fix me
#	/q/ and do {$VERB--; next };  #fix me
	die "$0: unknown option\"-$1\", check usage with --help\n"; } next };
    @nextargv and do {my $r = shift @nextargv; $r and $$r = $_; next };
    die "$0: Too many arguments, check usage with $0 --help";
}

&daemonize () if $DAEMONIZE;

sub daemonize {
    my $out = $LOGFILE || "/dev/null";
    chdir '/'                 or die "Can't chdir to /: $!";
    open STDIN, '/dev/null'   or die "Can't read /dev/null: $!";
    open STDOUT, '>>/dev/null'or die "Can't write to $out: $!";
    open STDERR, '>>/dev/null'or die "Can't write to $out: $!";
    defined(my $pid = fork)   or die "Can't fork: $!";
    exit if $pid;
    setsid                    or die "Can't start a new session: $!";
    umask 0;
}

print "Starting the proxy with parameters:
SPLITSIZE = $SPLITSIZE
REDIRSIZE = $REDIRSIZE
PARTSIZE  = $PARTSIZE
HOST      = $HOST
PORT      = $PORT
LISTENPORT= $LISTENPORT
BUFSIZE   = $BUFSIZE
CONCURRENCY = $CONCURRENCY
DEBUG     = $DEBUG
DAEMONIZE = $DAEMONIZE
TIMEZONE  = $TIMEZONE\n\n";

&init($HOST, $PORT); #sets %fields to the response fields

my $PROTOCOL = $FCP::fields{Protocol} || 1;
my $NODETYPE = $FCP::fields{Node} || " ";

&log("Protocol version: $PROTOCOL");
&log("Server Node Type: $NODETYPE");

my $socket = &startListening; 

my $waitedpid = 0;
my $paddr;

sub REAPER {
    $waitedpid = wait;
    $SIG{CHLD} = \&REAPER;  # loathe sysV
    &log ("reaped $waitedpid" , ($? ? "with exit $?\n" : '\n'));
}

$SIG{CHLD} = \&REAPER;

for ( $waitedpid = 0;
      ($paddr = accept(Client,Server)) || $waitedpid;
      $waitedpid = 0, close Client)
{
    next if $waitedpid and not $paddr;
    my($listenport,$iaddr) = sockaddr_in($paddr);
    my $name = gethostbyaddr($iaddr,AF_INET);
    
    &log ("connection from $name [". inet_ntoa($iaddr). "] at port $listenport");
    
    &spawn(\&server);
}

sub spawn {
    my $coderef = shift;
    
    unless (@_ == 0 && $coderef && ref($coderef) eq 'CODE') {
	confess "usage: spawn CODEREF";
#	die "spawning: $!";
    }
    
    my $pid;
    if (!defined($pid = fork)) {
	&log("cannot fork: $!");
	return;
    } elsif ($pid) {
	&log("begat $pid");
	return; # I'm the parent
    }
    # else I'm the child -- go spawn
    
    open(STDIN,  "<&Client")   || die "can't dup client to stdin";
    open(STDOUT, ">&Client")   || die "can't dup client to stdout";
    ## open(STDERR, ">&STDOUT") || die "can't dup stdout to stderr";
    exit &$coderef();
}

sub server {
    select(STDOUT); $|++;  #need to turn off buffering
    my $zerozerozerotwo;
    read STDIN, $zerozerozerotwo, 4;
    return 1 unless $zerozerozerotwo eq "\0\0\0\2";
    
    my $quit = 0;
    while(!$quit) {
	$quit = &proto1 if $PROTOCOL == 1;
    }
    return $quit;
}

sub proto1 ($$) {
    my ($request, $params) = &parseComField(\*STDIN);
    &log("Message type: <$request>");
    if ($request eq "ClientHello") {
	&printField(*STDOUT, "NodeHello", 
		   { Protocol => "1", 
		     Node => "Split-o-matic proxy"
		     } );
	&log("Sent Hello");
	return 0;
    } elsif ($request eq "ClientGet") {
	&clientGet($params->{URI}, $params->{HopsToLive});
	return 0;
    } elsif ($request eq "ClientPut") {
	my $URI = $params->{URI};
	my $metalen = hex $params->{MetadataLength} || 0; #if there's no metadata, set the len = 0
	my $datalen = hex $params->{DataLength};
	my $HTL = hex $params->{HopsToLive};
	my $metadata = substr $params->{Data}, 0, $metalen, "";
	my $info;
	if ($metadata =~ /([^=]*=[^=]*\n)+/) {
	    &log("FNP metadata");
	    $info = "EndPart\nInfo\n$metadata\n";
	} elsif ($metadata) {
	    &log("Non-FNP metadata, inserting seperately and losing");
	    $info = &putURI("CHK@", $HTL, "", $metadata);
	} else {
	    $info = "";
	}

	&log("Handling file of size $datalen");
	if ($datalen > $SPLITSIZE) {
	    &log("Splitting");
	    my $metadatapart = &insertSplit(\$params->{Data},$HTL);
	    my $meta = "Version\nVersionNo=0\nEndPart\n$metadatapart";
	    $meta .= $info;
	    $meta .= "End\n";
	    &putURI($URI, $HTL, $meta, "");
	    $FCP::fields{URI} .= "//";
	} elsif ($datalen > $REDIRSIZE && $URI !~ /^CHK/) {
	    &log("Redirecting");
	    my $reuri = &putURI("CHK@", $HTL, $metadata, $params->{Data});
	    &log("Redir to: $reuri");
	    my $meta = "Version\nVersionNo=0\nEndPart\nRedirect\nDocumentName=\nTarget=$reuri\nEnd\n";
	    &putURI($URI, $HTL, $meta, "");
	    $FCP::fields{URI} .= "//";
	} else {
	    &log("Inserting");
	    &putURI($URI, $HTL, $metadata, $params->{Data});
	}
      respond:
	&log("Inserted as ".$FCP::fields{URI});
	$_ = $FCP::response;
	my $rdata = \%FCP::fields;
	/^FormatError$/ and    do {die "formatting error"};
	/^URIError$/ and       do {&cliBadURI($URI); return 0};
#	/^KeyCollision$/ and   do {&printField(*STDOUT, $_, $rdata); return 0};
#	/^RouteNotFound$/ and  do {&printField(*STDOUT, $_, $rdata); return 0};
#	/^DataRejected$/ and   do {&printField(*STDOUT, $_, $rdata); return 0};
#	/^Success$/ and        do {&printField(*STDOUT, $_, $rdata); return 0};
	&printField(*STDOUT, $_, $rdata);
	return 0;
    } elsif ($request eq "CloseConnection") {
	return 1;
    } else {
	&log("unknown message");
	my $serfh = &connectToServer;
	&printField($serfh, $request, $params);
	my ($resp, $rdata) = &parseComField($serfh);
	&printField(*STDOUT, $resp, $rdata);
	&closeServer($serfh);
	return 0;
    }	
}

sub cliBadURI { print "BadURI\nURI=$_[0]\nEndMessage\n"; }

sub cliFailed {
    my $reason = shift;

    print "Failed\n";
    print "Reason=$reason\n" if $reason;
    print "EndMessage\n";
}

sub startListening {
    my $port = shift || $LISTENPORT;
    my $proto = getprotobyname('tcp');
    $port = $1 if $port =~ /(\d+)/; # untaint port number
    
    socket(Server, PF_INET, SOCK_STREAM, $proto)        || die "socket: $!";
    setsockopt(Server, SOL_SOCKET, SO_REUSEADDR,
	       pack("l", 1))   || die "setsockopt: $!";
    bind(Server, sockaddr_in($port, INADDR_ANY))        || die "bind: $!";
    listen(Server,SOMAXCONN)                            || die "listen: $!";
    
    &log("server started on port $port");
    return *Server;
}

sub log {
    print STDERR $_[0], "\n";
}

sub handleMetadata {
    my $metadata = shift;
    my $data = shift;
    my $docname = shift;
    my $htl = shift;

    my $internalformat = &parseMetadata($metadata);
    &log(Dumper($internalformat));
    my $target = $internalformat->{$docname};
    my $meta = "";
    $meta .= $_ for @{$target->{INFO}};
    my $actions;
    while ($actions = shift @{$target->{CDC}}) {
	$_ = $actions->[0];
	my $fields = $actions->[1];
	last unless $_;
	&log("Metadata followed: $_");
	/^Redirect$/ and return &clientGet( $fields->{Target}, $htl );
	/^SplitFile$/ and do { 
	    my @URI = ();
	    my @chunkfields = map {sprintf "Chunk.%x", $_} (0..((hex $fields->{"Chunks"}) - 1));
	    push @URI, map {$fields->{$_}} @chunkfields;
	    printField(*STDOUT, "DataFound", 
		       {DataLength => $fields->{FileSize}} );
	    return &splitGet( $htl, @URI ) 
	    };
	/^DateRedirect$/ and do {
	    my $inc = $fields->{Increment} || 86400;
	    my $baseline = $fields->{BaseLine} || die "No Baseline in DateRedirect";
	    my $now = time();
	    my $delta = $now - $baseline;
	    my $deltainc = $inc * int ($delta/$inc);
	    my $date = $baseline + $deltainc;
	    $date = &fromunix($date);
	    my $uri = $fields->{Target};
	    $uri =~ s|/|/$date-|;
	    &log("Requesting $uri");
	    return &clientGet ($uri, $htl)
	    };
    }
    &log("Couldn't find match for $docname");
    &log("DocumentNames found: " . join ",", keys %$internalformat);

    my $dl = (length $data) + (length $metadata);
    my $ml = length $metadata;
    &printField(*STDOUT, "DataFound", { DataLength => (sprintf "%x", $dl),
				       MetadataLength => (sprintf "%x", $ml) } );
    &printField(*STDOUT, "DataChunk", { Length => (sprintf "%x", $dl),
				       Data => $metadata.$data } );
    return;
}

sub parseMetadata ($$) {
    my $meta = shift;

    my ($ver) = $meta =~ /^Version\nVersionNo=(\d+)\n/;
    $ver = -1 if ($meta =~ /^Mapfile\n/ or $meta =~ /^Redirect\n/);

    my $int = {};
    
    if ($ver == 0) {
	foreach (split /EndPart\n/, $meta) {
	    my @lines = split /\n/;
	    my $command = shift @lines;
	    next if $command eq "Version";
	    my ($doc) = /DocumentName=(.*)\n/;
	    $doc = "" unless defined $doc;
	    if ($command eq "Info") {
		my $info = "";
		foreach my $line (@lines) {
		    $info .= $line unless /DocumentName=/;
		}
		push @{ $int->{$doc}{INFO} }, $info;
	    } else {
		my $fields = {};
		foreach my $text (@lines) {
		    last if $text =~ /End/;
		    my ($key, $value) = split /=/, $text, 2;
		    next unless $key;
		    $fields->{$key} = $value;
		}
		push @{ $int->{$doc}{CDC} }, [$command, $fields];
	    }
	}
    } elsif ($ver == -1) {
	if ($meta =~ /^Redirect\nEnd\n(.*)\n$/) { 
	    $int->{""}{CDC}[0] = ["Redirect", {Target => $1}];
	} elsif ($meta =~ /^Redirect\nincrement=(\d+)\nbaseline=(\d+)\nEnd\n(.*)\n$/) {
	    $int->{""}{CDC}[0] = ["DateRedirect", {Increment => $1, BaseLine=>&tounix($2), Target => $3}];
	} elsif ($meta =~ /^Mapfile\n(.*)End\n/) {
	    &log("Found Mapfile, using old school processing");
	    my $fields = $1;
	    my $default = "";
	    $default = $1 if $fields =~ /^default=(.*)$/;
	    my @lines = split /\n/, $meta;
	    chomp @lines;
	    my $line = "";
	    $line = shift @lines while ($line ne "End");
	    foreach my $line (@lines) {
		my ($doc, $key) = split /=/, $line;

		push @{$int->{$doc}{CDC}}, ["Redirect", {Target => $key}];
		if ($doc eq $default) {
		    push @{$int->{""}{CDC}}, ["Redirect", {Target => $key}];
		}
	    }
	}
    } else {
	warn "Unknown metadata version: $ver";
    }
    return $int;
}

sub clientGet ($$) {
    my $requested = shift;
    my $htl = shift;
    
    if($requested =~ /\/\//) {
	my ($URI,$docname) = split /\/\//, $requested, 2;
	my $metadata = "";
	my $data = "";
	&getURI($URI, $htl, \$metadata, \$data);

	$_ = $FCP::response;
	my $rdata = \%FCP::fields;
#	&log("Server response: $_");
	/^FormatError$/ and   do {die "formatting error"};
	/^URIError$/ and      do {&cliBadURI($URI); return 0};
	/^DataNotFound$/ and   do {&printField(*STDOUT, $_, $rdata); return 0};
	/^RouteNotFound$/ and  do {&printFIeld(*STDOUT, $_, $rdata); return 0};
	return &handleMetadata($metadata, $data, $docname, $htl);
    }
#    if ($requested =~ /^CHK/) {
	&getURI($requested, $htl, sub { &printField(*STDOUT, $_[0], $_[1]); } );
#    } else {
#	&log("Buffering the entire piece before sending it through");
#	my ($metadata, $data);
#	&getURI($requested, $htl, \$metadata, \$data);
#	my $dl = (length $data) + (length $metadata);
#	my $ml = length $metadata;
#	&printField(*STDOUT, "DataFound", 
#		    { DataLength => (sprintf "%x", $dl),
#		      MetadataLength => (sprintf "%x", $ml) } );
#	&printField(*STDOUT, "DataChunk", 
#		    { Length => (sprintf "%x", $dl),
#		      Data => $metadata.$data } );
#    }
    return 0;
}

#calculates the partsize of a splitfile and works out the metadata
#$data must be a reference to the data, $htl the htl in decimal
sub insertSplit ($$) {
    my $data = shift;
    my $htl = shift;

    my $datalen = length $$data;
#    my $numparts = 1+ int ((sqrt ($datalen/1024)) / 4);
#    my $partsize = &pow2($datalen/$numparts);
    my $partsize = $PARTSIZE;
    my $numparts = 1 + int ($datalen / $partsize);
    &log("Decomposing $datalen sized file into $numparts parts of $partsize each");
    my $s;
    my @keys = &insertParts($data, $partsize, $htl);
    die "Problem inserting all the parts" unless @keys == $numparts;
    
    my $meta = sprintf "SplitFile\nFileSize=%x\nChunks=%x\n", $datalen, $numparts;
    my $partnum = 0;
    foreach my $part (@keys) {
	$meta .= sprintf "Chunk.%x=%s\n", $partnum++, $part;
    }
    return $meta;
}

sub insertParts($$$) {
    my $data = shift;
    my $partsize = shift;
    my $htl = shift;

    my @queue = ();
    my @uqueue = ();
    my @keys;

    my $sent = 0;
    while (@queue > 0 || length $$data > $sent) {
	while (@queue < 2 * $CONCURRENCY && length $$data > $sent) {
	    &log("Sending block ". $sent / $partsize. " Queue: " . scalar @queue);
	    my $buffer = substr $$data, $sent, $partsize;
	    my $actsize = length $buffer;
	    push @uqueue, $sent/$partsize;
	    $sent += $actsize;
	    warn "Part size $actsize<$partsize" if $partsize>$actsize;
	    my $conn = &FCP::connectToServer($PORT, $HOST);
	    push @queue, $conn;
	    &printField($conn, "ClientPut", 
			{ URI => "CHK@", 
			  HopsToLive => (sprintf "%x", $htl),
			  DataLength => (sprintf "%x", length $buffer),
			  MetadataLength => 0,
			  Data => $buffer } );
	}
	my $fields;
	my $conn = shift @queue;
	my $count = shift @uqueue;
	&log("Waiting for response on insert $count, number queued: ". scalar @queue);
	do {  #get the request
	    ($_, $fields) = &parseComField($conn);
	} while (/^Restarted$/);  #try again if the response was "Restarted"

	/^DataRejected$/ and do {warn "Piece $count rejected"; 
				 return @keys };
	/^KeyCollision$/ and do {warn "Piece $count collided" };
	/^RouteNotFound$/ and do {warn "Piece $count couldn't be routed";
				  return @keys };
	my $uri = $fields->{URI};
	&log("Part $count=". $uri);
	die "Missing URI" unless $uri;
	push @keys, $uri;
#	&FCP::closeServer($conn);
    }
    return @keys;
}

#gets the pieces of a splitfile (the "DataFound" must have already been sent)
sub splitGet ($@) {
    my $htl = shift;
    my @uris = @_;
    my @queue;
    my $s = IO::Select->new();
    my ($acted, $fields) = (0, {});
    while (@uris || @queue) {
#	my $ready = $s->can_read(0.01) || 0;
	while (@queue < $CONCURRENCY && @uris > 0) { #queue up waiting pieces
	    my $uri = shift @uris;
	    die "Missing chunk URI" unless defined $uri;
	    &log("Requesting $uri\nQueued:". scalar @queue . " Left: " . scalar @uris);
	    my $conn = &FCP::connectToServer($PORT, $HOST);
#	    $conn->blocking(0);  #turn on non-blocking IO
	    $s->add($conn);      #watch it to see when it's readable
	    push @queue, { NET => $conn, URI => $uri , BUFFER => "", 
			   DSIZE => 1, DREAD => 0,
			   WAITING => 1, DATA => ""};
	    &printField($conn, "ClientGet", {URI => $uri, HopsToLive => $htl});
	    $acted = 1;
	}

	foreach my $request ($s->can_read(10)) {  #get any data in buffers
	    foreach my $pos (0..$#queue) {
		if ($request->fileno == $queue[$pos]{NET}->fileno) {
#		    print STDERR "W$pos ";
		    my $buf = "";
		    $request->sysread($buf, 10240);
		    $queue[$pos]{BUFFER} .= $buf;
		    $acted=2;
		    last;
		}
	    }
	    die "didn't put data anywhere" unless $acted == 2;
	}
#	print STDERR "Buffers: ", (join "\t", map {length $_->{BUFFER}} @queue), "\n";
	my $request = $queue[0]->{NET};
	unless ($request->connected) {
	    &log("Reopening connection");
	    &FCP::closeServer($request->{NET});
	    $request = &FCP::connectToServer($PORT, $HOST);
	    &printField($request->{NET}, "ClientGet", 
			{URI => $request->{URI}, HopsToLive => $htl});
	    my $rew = $request->{DREAD} - length $request->{DATA};
	    &printField(*STDOUT, "Rewind", {Length=>$rew});
	    $request->{WAITING}=1; $request->{DREAD}=0;
	    $request->{DATA} = "";
	    &log("Transfer Restarted");
	    $acted=5;
	}

	foreach my $req (@queue) {
	    while ($req->{BUFFER}) {
		($_, $fields) = &stringPCF(\$req->{BUFFER}); #parse the buffer
		last unless defined $_;
		$req->{WAITING} and do {
		    /^DataFound$/ and do { #start the xfer
			$req->{DSIZE} = hex $fields->{DataLength}; 
			&log("Chunk size: ".$req->{DSIZE});
			$req->{WAITING} = 0; next };
		    /^Restarted$/ and next;
		    /^DataNotFound$/ and do { #abort the xfer, sending DNF
			my $uri = $req->{URI};
			&log("Missing: $uri");
			&printField(*STDOUT, "DataNotFound", {Missing=>$uri});
			return 0 };
		    &printField(*STDOUT, $_, $fields);
		    die "Got message $_ while waiting";
		};
		/^DataChunk$/ and do {  #throw the data in DATA
		    $req->{DATA} .= $fields->{Data};
		    $req->{DREAD} += hex $fields->{Length}; 
		    print STDERR "."; next };
		/^Restarted$/ and do {   ##FINISH UP CODE HERE
		    $req->{REWIND} = $req->{DREAD} - length $req->{DATA};
		    $req->{DREAD} = 0;
		    $req->{DATA} = "";
		    $req->{WAITING} = 1;
		    &log("Transfer Restarted");
		    next };
		die "Got message $_ while not waiting";
	    }
	}
	if ($queue[0]{DATA}) {
	    my $h = $queue[0]; #keep track of the head of the queue
	    if ($h->{REWIND}) {
		my $len = sprintf "%x", $h->{REWIND};
		&printfield(*STDOUT, "Rewind", {Length => $len});
		$h->{REWIND} = 0;
	    }
	    my $len = sprintf "%x", length $h->{DATA};
	    &printField(*STDOUT, "DataChunk", 
		       {Length => $len, Data => $h->{DATA}} );
	    $h->{DATA} = "";
	    if ($h->{DREAD }== $h->{DSIZE}) { #we've finished the current piece
		my $done = shift @queue;  #pull it out of the queue
		$s->remove($done->{NET}); #stop watching it
		&FCP::closeServer($done->{NET}); #close that connection
		&log("\nCompleted: ".$done->{URI});
		$acted=4;
	    } elsif ($h->{DREAD} > $h->{DSIZE}) {
		warn "Data read more than expected";
		return 0;
	    }
	}
	unless ($acted) {
	    print STDERR "Did nothing.  read:", $queue[0]{DREAD}, "\n";
	    sleep 10;
	}
	$acted = 0;
    }
    print STDERR "Returning";
    return 1;
}

#rounds up its argument to the next power of 2
sub pow2($) { return 2 ** (1 + int ((CORE::log $_[0]) / (CORE::log 2))); }

#converts YYYYMMDDHHMMSS time format to unix time
sub tounix($) {
    my $in = shift;
    return 0 unless $PARSEDATE;
    return 0 unless $in =~ /(\d\d\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)/;
    my ($year, $month, $day, $hour, $minute, $second)=($1, $2, $3, $4, $5, $6);
    $year -= 2000;
    my $out = $second + &parsedate(sprintf "%02d/%02d/%02d.%02d:%02d", $year, $month, $day, $hour, $minute);
    $out += $TIMEZONE;
    print STDERR "IN: $in  OUT: $out\n";
    return $out;
}

#converts unix time to YYMMDDHHMMSS
sub fromunix ($) {
    my $unix = shift;
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime($unix);
    $year += 1900;
    $mon += 1;
    my $out = sprintf "%04d%02d%02d%02d%02d%02d", $year,$mon,$mday,$hour,$min,$sec;
    print STDERR "IN: $unix  OUT: $out\n";
    return $out;
}

=head1
#makes the filehandle bits for select
sub fhbits {
#    my @fhlist = split(' ',$_[0]);
    my $bits=0;
    return unless @_;
    while ($_ = shift @_) {
	vec($bits,$_->fileno,1) = 1;
    }
    return $bits;
}

=cut
