package Freenet.contrib.fcp;

import Freenet.*;
import Freenet.client.*;
import Freenet.client.events.*;
import Freenet.presentation.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.servlet.*;
import Freenet.servlet.util.*;
import java.io.*;
import java.net.MalformedURLException;

/**
 * An evil force in the universe.
 * @author Misses Good
 *
 **/

public class FCPHandler extends GenericServlet implements ClientEventListener {

    private static ClientFactory cf;
    private static int chunkSize;

    // servlet context
    private FNPRawMessage m;
    private InputStream in;
    private OutputStream out;

    // state info from monitoring ClientEvents
    private TransferStartedEvent xferStartedEvent;
    private boolean timedOut    = false;
    private boolean requesting  = false;
    private boolean xferStarted = false;
    private boolean finished    = false;
    private boolean cleanFail   = false;
    private boolean nodeError   = false;
    private boolean collision   = false;

    /** called by the ServletConstructor to handle a connection
      */
    public FCPHandler(Params params) throws Exception {
        synchronized (this.getClass()) {
            if (cf == null) {
                chunkSize = params.getint("chunkSize", 2048);
                ClientCore cc = ClientUtil.getServiceCore(0);
                cf = new FNPClient(cc, ClientUtil.getAddress(params.getParam("serverAddress")));
            }
        }
    }

    /** called by the Servlethandler to handle a connection
      */
    public void service(ServletRequest req, ServletResponse resp)
        throws ServletException, IOException
    {
        FreenetServletRequest freq;
        FreenetServletResponse fresp;
        try {
            freq = (FreenetServletRequest) req;
            fresp = (FreenetServletResponse) resp;
        } catch(ClassCastException e) {
            throw new ServletException(e);
        }
        this.in = new BufferedInputStream(freq.getNormalInputStream());
        this.out = fresp.getNormalOutputStream();

        try {
            // pull off the 00 00 00 02
            int b = 0;
            for (int i=0; i<4; ++i) {
                b <<= 8;
                b |= in.read();
            }
            if (b != 2)
                throw new IOException("bad session/presentation ID");
            m = new FNPRawMessage(in);
            if (m.messageType.equals("ClientHello"))            hello();
            else if (m.messageType.equals("GenerateCHK"))       generateCHK();
            else if (m.messageType.equals("GenerateSVKPair"))   generateSVKPair();
            else if (m.messageType.equals("ClientGet"))         get();
            else if (m.messageType.equals("ClientPut"))         put();
            else sendMessage("FormatError", null);
        }
        catch (InvalidMessageException e) {
            throw new ServletException(e);
        }
        finally {
            // don't need to close streams, it's automatic
            try { out.flush(); }
            catch (Throwable e) {}
        }
    }
 
    private void hello() throws IOException {
        FieldSet fs = new FieldSet();
        fs.put("Protocol", "1");
        fs.put("Node", "Freenet 0.3");
        sendMessage("NodeHello", fs);
    }

    private void generateCHK() throws IOException {

        long dlen = m.trailingFieldLength, mlen;
        try { mlen = Long.parseLong(m.fs.get("MetadataLength"), 16); }
        catch (NumberFormatException e) { mlen = 0; }

        if (dlen <= 0) {
            sendMessage("FormatError", null);
            return;
        }

        Bucket meta, data;
        if (mlen > 0) {
            meta  = new FileBucket();
            readAll(meta, mlen);
        }
        else {
            meta = new NullBucket();
        }
        data = new FileBucket();
        readAll(data, dlen - mlen);
       
        ComputeCHKRequest req = new ComputeCHKRequest(
            m.fs.get("CipherName"), meta, data, new FileBucket()
        );
        req.addEventListener(this);
        req.addEventListener(new EventLogger(Core.logger));
        
        cf.obtainClient(req).execute();
        if (waitRequesting()) waitDone();

        if (req.state() < Request.DONE) {
            sendMessage("Failed", null);
            return;
        }
        FieldSet fs = new FieldSet();
        fs.put("URI",  ""+req.getURI());
        sendMessage("Success", fs);
    }

    private void generateSVKPair() throws IOException {
        ComputeSVKRequest req = new ComputeSVKRequest();
        req.addEventListener(this);
        req.addEventListener(new EventLogger(Core.logger));
        cf.obtainClient(req).execute();
        if (waitRequesting()) waitDone();
        if (req.state() < Request.DONE) {
            sendMessage("Failed", null);
            return;
        }
        FieldSet fs = new FieldSet();
        fs.put("PublicKey",  Base64.encode(req.getPublicKeyFingerPrint()));
        fs.put("PrivateKey", Base64.encode(req.getPrivateKey()));
        sendMessage("Success", fs);
    }

    private void get() throws IOException {

        String uri = m.fs.get("URI");
        String htl = m.fs.get("HopsToLive");
        if (uri == null || htl == null) {
            sendMessage("FormatError", null);
            return;
        }

        Bucket meta = new PipedBucket(), data = new PipedBucket();   
        DataRequest req;
        try {
            req = new DataRequest(Integer.parseInt(htl, 16), uri, meta, data);
        } catch (MalformedURLException e) {
            sendMessage("URIError", null);
            return;
        }

        req.addEventListener(this);
        req.addEventListener(new EventLogger(Core.logger));

        cf.obtainClient(req).execute();
        if (waitRequesting()) waitTransfer();

        if (req.state() < Request.TRANSFERRING) {
            // have I got this right?
            String message;
            if      (cleanFail)  message = "DataNotFound";
            else if (nodeError)  message = "Failed";
            else                 message = "RouteNotFound";
            sendMessage(message, null);
            return;
        }

        while (true) {
            long dlen = xferStartedEvent.getDataLength();
            long mlen = xferStartedEvent.getMetadataLength();

            FieldSet fs = new FieldSet();
            fs.put("DataLength", Long.toHexString(mlen + dlen));
            fs.put("MetadataLength", Long.toHexString(mlen));

            try {
                sendMessage("DataFound", fs);
                if (mlen > 0) sendAll(meta, mlen);
                sendAll(data, dlen);
                return;
            } catch (IOException e) {
                sendMessage("Restarted", null);
                synchronized (this) { waitTransfer(); }
                if (req.state() < Request.TRANSFERRING) break;
            }
        }

        sendMessage("Failed", null);
    }

    private void put() throws IOException {

        long dlen = m.trailingFieldLength, mlen;
        try { mlen = Long.parseLong(m.fs.get("MetadataLength"), 16); }
        catch (NumberFormatException e) { mlen = 0; }
        String uri = m.fs.get("URI");
        String htl = m.fs.get("HopsToLive");

        if (uri == null || htl == null || dlen <= 0) {
            sendMessage("FormatError", null);
            return;
        }

        Bucket meta, data;
        if (mlen > 0) {
            meta  = new FileBucket();
            readAll(meta, mlen);
        }
        else {
            meta = new NullBucket();
        }
        data = new FileBucket();
        readAll(data, dlen - mlen);

        InsertRequest req;
        try {
            req = new InsertRequest(
                Integer.parseInt(htl, 16), uri, m.fs.get("CipherName"), meta, data, new FileBucket()
            );
        } catch (InsertSizeException e) {
            sendMessage("SizeError", null);
            return;
        }

        req.addEventListener(this);
        req.addEventListener(new EventLogger(Core.logger));

        Client c;
        try {
            c = cf.obtainClient(req);
        } catch (KeyException e) {
            sendMessage("URIError", null);
            return;
        }
        c.execute();
        
        if (waitRequesting()) waitDone();

        // build the reply message
        FieldSet fs = new FieldSet();
        FreenetURI furi = req.getURI();
        if (furi.getKeyType().equals("SVK") || furi.getKeyType().equals("SSK")) {
            fs.put("PublicKey",  Base64.encode(req.getPublicKeyFingerPrint()));
            fs.put("PrivateKey", Base64.encode(req.getPrivateKey()));
        }
        fs.put("URI", ""+furi);

        String message;
        if      (req.state() == Request.DONE)  message = "Success";
        else if (collision)                    message = "KeyCollision";
        else if (nodeError)                    message = "Failed";
        else                                   message = "RouteNotFound";

        sendMessage(message, fs);
    }


    //=== utility methods =====================================================

    private int readChunk(InputStream i, byte[] buf, int len) throws IOException {
        int n = 0, m;
        while (n < len) {
            m = i.read(buf, n, len-n);
            if (m == -1) return n;
            n += m;
        }
        return n;
    }

    private void readAll(Bucket b, long len) throws IOException {
        byte[] buf = new byte[chunkSize];
        OutputStream bout = b.getOutputStream();
        try {
            while (len > 0) {
                int n = (int) (len < chunkSize ? len : chunkSize);
                if (readChunk(in, buf, n) != n) throw new IOException("EOF from client.");
                bout.write(buf, 0, n);
                len -= n;
            }
        }
        finally {
            bout.close();
        }
    }
    
    private void sendAll(Bucket b, long len) throws IOException {
        WriteOutputStream writer = new WriteOutputStream(out);
        byte[] buf = new byte[chunkSize];
        InputStream bin = b.getInputStream();
        try {
            while (len > 0) {
                int n = (int) (len < chunkSize ? len : chunkSize);
                if (readChunk(bin, buf, n) != n) throw new IOException("EOF from node.");
                writer.writeUTF("DataChunk\nLength="+Long.toHexString(n)+"\nData\n");
                writer.flush();
                out.write(buf, 0, n);
                len -= n;
            }
        }
        finally {
            bin.close();
        }
    }

    private void sendMessage(String messageType, FieldSet fs) throws IOException {
        WriteOutputStream writer = new WriteOutputStream(out);
        writer.writeUTF(messageType, '\n');
        if (fs == null)
            writer.writeUTF("EndMessage", '\n');
        else
            fs.writeFields(writer, "EndMessage", '\n', '=', '.');
        writer.flush();
    }


    //=== event handling ======================================================

    private synchronized boolean waitRequesting() {
        while (!finished && !requesting) {
            timedOut = true;
            try { wait(20000); } catch (InterruptedException e) {}
            if (timedOut) {
                nodeError = true;
                return false;
            }
        }
        return true;
    }

    private synchronized void waitTransfer() {
        while (!finished && !xferStarted) {
            try { wait(); } catch (InterruptedException e) {}
        }
        xferStarted = false;
    }

    private synchronized void waitDone() {
        while (!finished) {
            try { wait(); } catch (InterruptedException e) {}
        }
    }
    
    public synchronized void receive(ClientEvent ce) {
        if (ce instanceof StateReachedEvent) {
            int state = ((StateReachedEvent) ce).getState();
            if (state == Request.FAILED || state == Request.DONE)
                finished = true;
            else if (state == Request.REQUESTING)
                requesting = true;
        }
        else if (ce instanceof RestartedEvent) {
            // this happens during the REQUESTING phase
            try { sendMessage("Restarted", null); }
            catch (IOException e) {}
        }
        else if (ce instanceof TransferStartedEvent) {
            xferStarted = true;
            xferStartedEvent = (TransferStartedEvent) ce;
        }
        else if (ce instanceof CollisionEvent) {
            collision = true;
        }
        else if (ce instanceof RequestFailedEvent) {
            cleanFail = true;
        }
        else if (ce instanceof ExceptionEvent) {
            // hmm.. what conditions should constitute a node error ?
            try { ((ExceptionEvent) ce).rethrow(); }
            catch (Exception e) {
                if (e instanceof ConnectFailedException ||
                    e instanceof SendFailedException    ||
                    e instanceof InterruptedIOException)
                    nodeError = true;
                System.err.println(((ExceptionEvent) ce).getDescription());
            }
        }
        timedOut = false;
        notify();
    }
}


