package Freenet.contrib.fproxy;

import Freenet.*;
import Freenet.client.*;
import Freenet.keys.*;
import Freenet.message.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.contrib.fproxy.mumail.mime.*;
import Freenet.contrib.fproxy.filter.*;
import java.io.*;
import java.net.*;
import java.util.*;
import Freenet.servlet.*;
import Freenet.servlet.util.*;


/*
  This code is part of fproxy, an HTTP proxy server for Freenet.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/


/**
 * Thread to handle incoming HTTP connections for the ProxyServer
 *
 * @author <a href="http://www.doc.ic.ac.uk/~twh1/">Theodore Hong</a>
 **/

public class HttpHandlerServlet extends GenericServlet
{
    // class variables
    private static final char[] lf = {'\012'};
    private static final long dataStreamTimeout = 10000000;
    private static final long MAX_FORCE_KEYS = 1000;
    // is anyone really going to keep waiting for their data after 3 hours?

    // local instance variables
    private InputStream in;
    private OutputStream out;
    private PrintWriter pw;
    private Logger saveLogger;
    private String typeExpected = "";
    private Params params;

    private Hashtable attrs;
    private String url;
    private ContentFilter filter = null;
    private static Random random = new Random();
    private static Hashtable randkeys = new Hashtable();
    private static Object lastForceKey = null;
    private static Object firstForceKey = null;
    private boolean isLocalConnection;

    public HttpHandlerServlet()
    {
      this(new Params());
    }

    public HttpHandlerServlet(Params params)
    {
	if(ProxyServer.sc==null) {
	    ProxyServer.sc=new SimplifiedClient(params);
	}
	if (params.getboolean("doFiltering", false)) {
	    filter = ContentFilterFactory.newInstance(params.getParam("passthroughMimeTypes",""));
	}
    }

    /**
     * Creates a new HttpHandler communicating on Socket s
     **/
    public void service(ServletRequest req, ServletResponse resp) throws ServletException {
      FreenetServletRequest freq;
      FreenetServletResponse fresp;
      try
      {
        freq=(FreenetServletRequest)req;
        fresp=(FreenetServletResponse)resp;
      }
      catch(ClassCastException e)
      { throw new ServletException("HttpHandlerServlet needs FreenetServletRequest and FreenetServletResponse so that it can access raw InputStream and OutputStream objects."); }

	this.in = new BufferedInputStream(freq.getNormalInputStream());
	this.out = fresp.getNormalOutputStream();
	this.pw = new PrintWriter(this.out);

	this.isLocalConnection = freq.isLocal();

	ProxyServer.sc.getLogger().log(this, "New HttpHandler", Logger.MINOR);
        run();
    }

    private void parseUrl(String url) throws URLEncodedFormatException {
	attrs = new Hashtable();
	StringTokenizer st = new StringTokenizer(url, "?");
	if (!st.hasMoreTokens())
	    return;
	this.url = st.nextToken();
	if (!st.hasMoreTokens())
	    return;
	String attrs_string = st.nextToken();
	st = new StringTokenizer(attrs_string, "&");
	while (st.hasMoreTokens()) {
	    String name = st.nextToken();
	    String value = "";
	    int pos = name.indexOf('=');
	    if (pos > 0) {
		value = name.substring(pos+1);
		name = name.substring(0, pos);
		value = Freenet.support.URLDecoder.decode(value);
	    }
	    attrs.put(name, value);
	}
    }

    /**
     * Main routine to process an HTTP request
     **/
    public void run() {
	try {
	    // read HTTP request
	    UnbufferedReader r = new UnbufferedReader(in);
	    String line = r.readLine();
	    ProxyServer.sc.getLogger().log(this, line, Logger.MINOR);
	    StringTokenizer st = new StringTokenizer (line, " ");
	    String method = st.nextToken();
	    String url = st.nextToken();

	    // well, well, what have we here?
	    if (method.equalsIgnoreCase("GET")) {
		try {

		    //System.err.println("HttpHandlerServelet.run -- url: " + url);

		    // Handle checked jumps out of freenet.
		    if (handleCheckedJump(url)) {
			return;
		    }

		    // Parse any parameters
		    parseUrl(url);
		    // Slurp up the headers
		    do { line = r.readLine(); } while (line != null && !"".equals(line));
		    // Perform the get
            if (attrs.containsKey("key")) {
                String loc = "/" + attrs.get("key");
                attrs.remove("key");
                String qs = "";
                for (Enumeration e = attrs.keys() ; e.hasMoreElements() ;) {
                    String k = (String) e.nextElement();
                    qs += k + "=" + attrs.get(k) + "&";
                }
                if (!qs.equals("")) {
                    loc += "?"+qs;
                }
                System.out.println("Got key in query string, issuing a 301.. qs="+qs);
                pw.print("HTTP/1.0 301 Moved Permanently\015\012");
                pw.print("Connection: close\015\012");
                pw.print("Location: "+loc+"\015\012");
                pw.print("\015\012");
                pw.flush();
                out.flush();
                out.close();
            }
            else get();
		}
		catch (URLEncodedFormatException e) {
		    malformed(url);
		}
	    } else if (method.equalsIgnoreCase("POST")) {
		post(in);	// key will be in the form body
	    } else {
		unimplemented(method);
	    }
	}
	catch (EOFException e) {
	    ProxyServer.sc.getLogger().log(this, "Encountered EOFException", Logger.ERROR);
	    e.printStackTrace(ProxyServer.sc.getLogger().getOut());
	}
	catch (IOException e) {
	    ProxyServer.sc.getLogger().log(this, "Encountered IOException", Logger.ERROR);
	    e.printStackTrace(ProxyServer.sc.getLogger().getOut());
	}
	catch (RuntimeException e) {
	    ProxyServer.sc.getLogger().log(this, "Encountered runtime exception",
			    Logger.ERROR);
	    e.printStackTrace(ProxyServer.sc.getLogger().getOut());
	}
    }

    ////////////////////////////////////////////////////////////
    // Support checked jumps out of Freenet.
    protected final boolean handleCheckedJump(String url) throws IOException {

	String decodedURL = decodeCheckedJumpURL(url);
	if (decodedURL == null) {
	    return false;
	}

	pw.print("HTTP/1.0 200 OK\015\012");
	pw.print("Connection: close\015\012");
	pw.println("Content-type: text/html");
	pw.println();

	pw.println("<html>");
	pw.println("<head>");

	pw.println("<title>");
	pw.println("External link " + decodedURL );
	pw.println("</title>");
	pw.println("</head>");

	pw.println("<h1>Warning: External link " + decodedURL + "</h1>");
	pw.println("<p>");
	pw.println("Browsing external links <em>is not</em> anonymous.");
	pw.println("<p>");
	pw.println("Click on the link below to continue or hit the");
	pw.println("back button on your browser to abort.");
	pw.println("<p>");
	pw.println("<a href=\"");
	pw.println( decodedURL );
	pw.println("\">" + decodedURL + "</a>");
	pw.println("</body>");
	pw.println("</html>");
	pw.flush();
	out.flush();
	out.close();

	return true;
    }
    
    protected final static String MSG_BADURL =  "Couldn't decode checked jump url.";

    protected final static String ESCAPED_HTTP = "/__CHECKED_HTTP__";
    protected final static String UNESCAPED_HTTP = "http://";

    protected final static String ESCAPED_FTP = "/__CHECKED_FTP__";
    protected final static String UNESCAPED_FTP = "ftp://";

    protected final static String decodeCheckedJumpURL(String url) {
	String ret = null;
	if (url.startsWith(ESCAPED_HTTP)) {
	    if (url.length() < ESCAPED_HTTP.length() + 1) {
		throw new IllegalArgumentException(MSG_BADURL);
	    }

	    ret = UNESCAPED_HTTP + url.substring(ESCAPED_HTTP.length());
	}

	if (url.startsWith(ESCAPED_FTP)) {
	    if (url.length() < ESCAPED_FTP.length() + 1) {
		throw new IllegalArgumentException(MSG_BADURL);
	    }

	    ret = UNESCAPED_FTP + url.substring(ESCAPED_FTP.length());
	}

	return ret;
    }
    ////////////////////////////////////////////////////////////

    protected static boolean checkForceKey(Object key) {
	return randkeys.get(key) != null;
    }

    protected static String makeForceKey() {
	synchronized (randkeys) {
	    String forceKey = null;
	    do {
		forceKey = Integer.toHexString(random.nextInt());
	    }
	    while (randkeys.containsKey(forceKey));

	    randkeys.put(forceKey, forceKey);

	    if (lastForceKey != null) {
		randkeys.remove(lastForceKey);
		randkeys.put(lastForceKey, forceKey);
	    }
	    else {
		firstForceKey = forceKey;
	    }

	    lastForceKey = forceKey;
	    if (randkeys.size() > MAX_FORCE_KEYS) {
		Object newForceKey = randkeys.get(firstForceKey);
		randkeys.remove(firstForceKey);
		firstForceKey = newForceKey;
	    }
	    return forceKey;
	}
    }

    protected static void dumpForceKeys() {
	Enumeration keys = randkeys.keys();
	while (keys.hasMoreElements()) {
	    Object key = keys.nextElement();
	    System.out.println("   " + key + " -> " + randkeys.get(key));
	}
    }

    public static void main(String[] args) {
	for (int i = 0 ; i < 1000 ; i++) {
	    String key = makeForceKey();
	    System.out.println("KEY: " + key);
	    System.out.println("DUMP:");
	    dumpForceKeys();
	}
    }

    /**
     * Initiate a Freenet request (GET method)
     **/
    protected void get() throws IOException, URLEncodedFormatException {
	// strip leading slash from absolute URI
	String key = Freenet.support.URLDecoder.decode(url);
	if (key.startsWith("/")) {
	    key = key.substring(1);
	}

	// a normal request

	FProxyDocument doc = null;
	try {
          if(key.equals(""))
          {
            System.out.println("No key, loading gateway.html.");
            pw.print("HTTP/1.0 200 OK\015\012");
            pw.print("Connection: close\015\012");
            pw.println("Content-type: text/html");
            pw.println();
            BufferedReader br=new BufferedReader(new InputStreamReader(HttpHandlerServlet.class.getResourceAsStream("gateway.html")));
            String s = br.readLine();
            while(s!=null)
            {
              pw.println(s);
              s=br.readLine();
            }
            pw.flush();
            out.flush();
            out.close();
          }
	  else if (key.equals("robots.txt"))
	  {
	      if (!isLocalConnection) {
		  pw.print("HTTP/1.0 200 OK\015\012");
	          pw.print("Connection: close\015\012");
	          pw.print("Content-type: text/plain\015\012\015\012");
	          pw.println("Disallow: *");
	      } else {
		  pw.print("HTTP/1.0 404 Not Found\015\012");
	          pw.print("Connection: close\015\012");
	          pw.println("Content-type: text/html");
	          pw.println("");
		  pw.println("Okay, little robot, give me your best shot!");
	      }
	      pw.flush();
	      out.flush();
	      out.close();
	  }
          else
          {
	    // do freenet request
            pw.flush();
	    Object htl = attrs.get("htl");
	    if (htl!=null) {
		ProxyServer.sc.setHtl(Integer.parseInt(htl.toString()));
	    }
	    ProxyClient pc = new ProxyClient(out, ProxyServer.sc);
	    doc = pc.request(key);
            if(doc.in==null) throw new FileNotFoundException();
            if(doc.mimeType==null) doc.mimeType=guessMimeType(key);
	    if (attrs.get("mime") != null) {
		doc.mimeType = (String)attrs.get("mime");
	    }
	    Object force = attrs.get("force");
	    Conduit conduit = null;

	    if (filter == null) {
		conduit = new Conduit(doc.in, out, null);
	    } 
	    else {
		if (force != null) {
		    if (checkForceKey(force)) {
			filter.setAllowSecurityWarnings(true);
			filter.setAllowSecurityErrors(true);
		    }
		}
		else {
		    filter.setAllowSecurityWarnings(false);
		    filter.setAllowSecurityErrors(false);
		}
		conduit = new Conduit(filter.run(doc.in, doc.mimeType), out, null);
	    }
            pw.print("HTTP/1.0 200 OK\015\012");
            pw.print("Connection: close\015\012");
            if(doc.mimeType!=null)
              pw.println("Content-type: "+doc.mimeType);
	    // This following line is a workaround for an IE bug
	    if (doc.mimeType.equals("application/octet-stream"))
	      pw.println("Content-Disposition: attachment; filename="+key);
            pw.println();
            pw.flush();
            conduit.run();
            out.flush();
            out.close();
          }
        } catch (Exception e) {
	    // something went wrong - send error message to browser
            pw.print("HTTP/1.0 404 Not Found\015\012");
            pw.print("Connection: close\015\012");
            pw.println("Content-type: text/html");
            pw.println("");
	    pw.println("<html>");
	    pw.println("<head>");
	    pw.println("<title> Couldn't retrieve " + key + "</title>");
	    pw.println("</head>");
	    pw.println("");
	    pw.println("<body>");
	    if (e instanceof FilterException) {
		FilterException fe = (FilterException)e;
		pw.println("<H1>Warning: "+e.getMessage()+"</H1>");
		pw.println(((FilterException) e).explanation);
                // Why do we have to encode the key here?  It's because this
                // HTML is generated by the gateway, and we cannot trust the
                // URL.  It might have characters which will break the security
                // of the page (e.g. close quotes).
                //
                // So we encode it, and pass it as part of the query string.
                // If the user clicks on the "Retrieve anyway", we will get the
                // URL as a form argument, decode it and redirect to it.  The
                // redirect is hopefully secure against funny characters...
                //
		String encKey = java.net.URLEncoder.encode(key);
		String forceKey = makeForceKey();
		pw.println("<p><a href=\"/?key=" + encKey + "&force=" + forceKey + "\">Retrieve anyway</A>, see the <a href=\"/?key=" + encKey + "&mime=text/plain\">source</A> or <A HREF=\"/\">return</A> to gateway page");
		if (fe.analysis != null) {
		    Enumeration errors = fe.analysis.getDisallowedElements();
		    if (errors != null) {
			pw.println("<H3>Disallowed elements</H3>");
			while (errors.hasMoreElements()) {
			    pw.println("<BR> - " + errors.nextElement());
			}
		    }
		    Enumeration warnings = fe.analysis.getWarningElements();
		    if (warnings != null) {
			pw.println("<H3>External links</H3>");
			while (warnings.hasMoreElements()) {
			    pw.println("<BR> - " + warnings.nextElement());
			}
		    }
		}
	    }
	    else {
		pw.println("<H1>Network Error</H1>");
                pw.println("<p>Couldn't retrieve key: <b>" + key + "</b> <br>Hops To Live: <b>" + ProxyServer.sc.getHtl() + "</b><br>");
                String encKey = java.net.URLEncoder.encode(key);
                pw.println("<form name=\"changeHTL\" action=\"/" + encKey + "\"> <p>Change Hops To Live <input type=\"text\" size=\"3\" name=\"htl\" value=\"" + ProxyServer.sc.getHtl() + "\"/> <input type=\"submit\" value=\"Retry\"/> </form>");
	    }
	    sendHTMLFooter();
            
            if (e instanceof FileNotFoundException) {
                ProxyServer.sc.getLogger().log(this, "File not found in Freenet", Logger.DEBUGGING);
            }
            else if (e instanceof NoDataException) {
                ProxyServer.sc.getLogger().log(this, "File not found in Freenet", Logger.DEBUGGING);
            }
	    else if (e instanceof URLEncodedFormatException) {
		ProxyServer.sc.getLogger().log(this, "Invalid urlencoded key: " + key,
				Logger.ERROR);
	    }
	    else if (e instanceof MalformedURLException) {
		ProxyServer.sc.getLogger().log(this, "Invalid Freenet URI: " + key,
				Logger.ERROR);
	    }
	    else if (e instanceof BadAddressException) {
		ProxyServer.sc.getLogger().log(this, "Invalid Freenet server address: "
			  + ProxyServer.serverAddress, Logger.ERROR);
	    }
	    else {
		ProxyServer.sc.getLogger().log(this, "Client object signalled: " + e,
				Logger.ERROR);
                e.printStackTrace();
	    }
	} finally {
	    // terminate page
//	    logIn();

	    ProxyServer.sc.getLogger().log(this, "Finished handling request, user's turn now",
			    Logger.MINOR);
	}
    }

    private String guessMimeType(String key)
    {
      if (key.toLowerCase().indexOf("html") != -1)
        return "text/html";
      if (key.toLowerCase().indexOf("text") != -1)
        return "text/plain";
      if (key.toLowerCase().indexOf("txt") != -1)
        return "text/plain";
      if (key.toLowerCase().indexOf("gif") != -1)
        return "image/gif";
      if (key.toLowerCase().indexOf("jpeg") != -1)
        return "image/jpeg";
      if (key.toLowerCase().indexOf("jpg") != -1)
        return "image/jpeg";
      if (key.toLowerCase().indexOf("mp3") != -1)
        return "audio/mpeg3";
      if (key.toLowerCase().indexOf("mpeg") != -1)
        return "video/mpeg";
      if (key.toLowerCase().indexOf("mpg") != -1)
        return "video/mpeg";
      return "application/octet-stream";
    }

    /**
     * Perform a Freenet insert (POST method)
     **/
    protected void post(InputStream in) throws IOException {
	// send response header
	pw.print("HTTP/1.0 200 OK\015\012");
	pw.print("Connection: close\015\012");
        logOut();
        pw.println("Content-type: text/html\n");
        pw.println("<html><head><title>Freenet insert</title></head><body>");
        pw.flush();
	// read POST data
	mimeMail message = new mimeMail(in, pw);
	String key = "<no key specified>";
	String contentType = "application/octet-stream";

	try {
	    typeExpected = "multipart/form-data";
	    MIME_multipart formData = (MIME_multipart) message.getBody();
	    ProxyServer.sc.getLogger().log(this, "Content-type: "
		       + message.getHeader().getContent_Type()
		       + " (" + formData.getPartCount() + " parts)",
		       Logger.DEBUGGING);
	    for (int i = 0; i < formData.getPartCount(); i++) {
		ProxyServer.sc.getLogger().log(this, i + " content-type: "
			   + formData.getPart(i).getHeader().getContent_Type(),
			   Logger.DEBUGGING);
	    }

	    if (formData.getPartCount() < 3) {
		throw new RuntimeException
		    ("Form data must contain at least three parts");
	    }
	    // COMEBAK: check the part names instead of just assuming the order
	    MIME part0 = formData.getPart(0);
	    MIME part1 = formData.getPart(1);
	    MIME part2 = formData.getPart(2);

	    // this should be MIMEtext by default, but Netscape seems to send 
	    // unknown binary without content-types, so we need to use 
	    // application/octet-stream as the default
	    typeExpected = "application/octet-stream";
	    key = new String (((MIME_application) part0).getBody());
	    contentType = new String (((MIME_application) part2).getBody());
	    ProxyServer.sc.getLogger().log(this, "User specified content-type: "+contentType,
			    Logger.DEBUGGING);

	    // make contents into byte array, if necessary
	    byte[] insertData = new byte[0];
	    if (part1 instanceof MIMEtext) {
		MIMEtext t = (MIMEtext) part1;
		// COMEBAK: this uses our location's line terminator
		// really we should just keep the file as-submitted
		insertData = MIMEcoder.decode7Bit(t.getLines(), 0,
						  t.getLines().length);
	    }
	    else if (part1 instanceof MIMEbinary) {
		MIMEbinary b = (MIMEbinary) part1;
		insertData = b.getBody();
	    }

	    // detect content-type if not specified
	    if (contentType == null || contentType.equalsIgnoreCase("auto")) {
		contentType = part1.getHeader().getContent_Type();
		if (contentType.equalsIgnoreCase("application/unspecified")) {
		    contentType = null;
		}
	    }
	    ProxyServer.sc.getLogger().log(this, "Inserting with content-type: "+contentType,
			    Logger.DEBUGGING);

	    // check a file was sent
	    if (insertData.length == 0) {
		pw.println("Inserting key: <b>" + key + "</b>");
		pw.println("<p>You didn't send any data!</p>");
		pw.flush();
		return;
	    }

	    // show user we're doing something
	    //logOut();
	    pw.println("<p>Inserting key: <b>" + key + "</b></p>");
	    pw.println("<p>");
	    pw.flush();

	    // do freenet insert
	    InputStream datastream = new ByteArrayInputStream(insertData);
	    ProxyClient pc = new ProxyClient(out, ProxyServer.sc);
	    key = pc.insert(key, datastream, contentType);

	    // show success message
	    pw.println("</p>");
	    pw.println("<p>The insert completed successfully.<br>");

	    // print key confirmation
	    pw.println("To retrieve this file in the future, use this URI: "
		       + "<b>" + key + "</b></p");
            String s=(String)StringUtils.split(key, ':').elementAt(1);
            String type=(String)StringUtils.split(s, '@').elementAt(0);
	    if (type.equals("SVK")) {
		ClientSVK svk = new ClientSVK(new FreenetURI(s));
		pw.println("<p>The SVK private key used was: <code>"
			   + Base64.encode(((SVK) svk.getKey()).getKeyPair()
					   .getX().toByteArray())
			   + "</code>.<br>");
		pw.println("Hang on to this key if you wish to update it in "
			   + "the future, or if you wish to insert subspace "
			   + "keys (SSKs) under this SVK root.</p>");
	    }
	    pw.flush();

	    // send to key indexes, if requested
	    boolean firstPass = true;
	    for (int i = 3; i < formData.getPartCount(); i++) {
		MIME nextPart = formData.getPart(i);
		String name = nextPart.getHeader().getContent_DispositionParameter("name");
		if (!name.equalsIgnoreCase("index")) {
		    continue;
		}
		typeExpected = "application/octet-stream";
		String index = new String (((MIME_application) nextPart).getBody());
		
		// obligatory warning
		if (firstPass) {
		    pw.println("<hr>Submitting key to indexes: <b>" + key + "</b>");
		    pw.println("<p>The keyindex server is only a temporary "
			       + "tool for alpha testers. It is not a "
			       + "permanent feature of the system. It will "
			       + "be removed in a future release.</p>");
		    pw.println("<p>");
		    pw.flush();
		    firstPass = false;
		}

		// do it
		pw.println("Notifying " + index + "... ");
		pw.flush();
		try {
		    URL keyserver = new URL(index + "?key="
					    + URLEncoder.encode(key));
		    InputStream kstream = keyserver.openStream();

		    // wait until the server returns something before closing
		    // this way we make sure not to kill the server-side 
		    // script too soon
		    kstream.read();
		    kstream.close();

		    pw.println("ok.<br>");
		    pw.flush();
		}
		catch (Exception e) {
		    // note error and keep going
		    ProxyServer.sc.getLogger().log(this, "Inform connection to " + index
				    + " failed: "+e, Logger.ERROR);
		}
	    }

	    // terminate page
	    if (!firstPass) {
		pw.println("<p>done.</p>");
		pw.flush();
	    }
	    sendHTMLFooter();
	    logIn();

	    ProxyServer.sc.getLogger().log(this, "Finished handling insert", Logger.MINOR);
	}
	catch (Exception e) {
	    // something went wrong - send error message to browser
	    pw.println("</p>");
	    pw.println("<p>Couldn't insert key: <b>" + key + "</b><br>");
	    
	    if (e instanceof ClassCastException) {
		throw new RuntimeException
		    ("Unexpected form data type: "
		     + message.getHeader().getContent_Type()
		     + " (expecting " + typeExpected + ")");
	    }
	    else if (e instanceof MalformedURLException) {
		ProxyServer.sc.getLogger().log(this, "Invalid Freenet URI: " + key,
				Logger.ERROR);
	    }
	    else if (e instanceof BadAddressException) {
		ProxyServer.sc.getLogger().log(this, "Invalid Freenet server address: "
				+ ProxyServer.serverAddress, Logger.ERROR);
	    }
	    else {
		ProxyServer.sc.getLogger().log(this, "Client object signalled: " + e,
				Logger.ERROR);
                e.printStackTrace();
	    }
	}
    }


    /**
     * Trap an unimplemented method
     **/
    protected void unimplemented(String method) throws IOException {
	ProxyServer.sc.getLogger().log(this, "Sending HTTP 501 response", Logger.MINOR);
	pw.print("HTTP/1.0 501 Not Implemented\015\012\015\012");
	pw.println("<HTML><HEAD>");
	pw.println("<TITLE>501 Not Implemented</TITLE>");
	pw.println("</HEAD><BODY>");
	pw.println("<H1>" + method + " Method Not Implemented</H1>");
	pw.println("</BODY></HTML>");
	pw.flush();
	throw new RuntimeException(method + " method not implemented");
    }

    /**
     * Trap an unimplemented method
     **/
    protected void malformed(String error) throws IOException {
	ProxyServer.sc.getLogger().log(this, "Sending HTTP 501 response", Logger.MINOR);
	pw.print("HTTP/1.0 404 Not Implemented\015\012\015\012");
	pw.println("<HTML><HEAD>");
	pw.println("<TITLE>404 Not Implemented</TITLE>");
	pw.println("</HEAD><BODY>");
	pw.println("<H1>" + error + " malformed request</H1>");
	pw.println("</BODY></HTML>");
	pw.flush();
	throw new RuntimeException(error + " method not implemented");
    }

    /**
     * Send HTML footer
     **/
    private void sendHTMLFooter() throws IOException {
	pw.println("</body>");
	pw.println("</html>");
	pw.flush();
    }


    /**
     * Transfer log out to user's browser
     **/
    private void logOut() throws IOException {
	ProxyServer.sc.getLogger().log(this, "Transferring log messages to HTTP stream",
			Logger.MINOR);

	// swap out logger object
	saveLogger = ProxyServer.sc.getLogger();
	ProxyServer.sc.setLogger(new HtmlLogger(out, 0, saveLogger.threshold));
    }


    /**
     * Transfer log back in to logfile
     **/
    private void logIn() {
	// restore old logger
	ProxyServer.sc.setLogger(saveLogger);
    }
}
