package Freenet.contrib.fproxy;
import Freenet.*;
import Freenet.client.*;
import Freenet.support.*;
import java.io.*;
import java.net.*;


/*
  This code is part of fproxy, an HTTP proxy server for Freenet.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/


/**
 * The main ProxyServer class.  It listens for incoming HTTP connections on 
 * -listenPort (default 8081), extracts the Freenet key, and issues a proxy 
 * request on the web client's behalf to the Freenet node at -serverAddress 
 * (default tcp/localhost:19114).  The reply is streamed directly over the
 * HTTP connection back to the web client, with the appropriate MIME type
 * Any errors are also reported to the web client.
 *
 * Note: This class is never instantiated; there is only one ProxyServer.
 *
 * @author <a href="http://www.doc.ic.ac.uk/~twh1/">Theodore Hong</a>
 **/

public class ProxyServer
{
    // defaults
    private static final int defaultListenPort = 8081;
    private static final String defaultAllowedHosts = "127.0.0.1";
    private static final String defaultServerAddress = "tcp/localhost:19114";
    private static final long defaultHopsToLive = 5;
    private static final String defaultLogFile = "fproxy.log";
    private static final String defaultLogging = "normal";
    private static final int defaultVerbosity = 1;

    // server parameters
    public static String allowedHosts;
    public static int listenPort;
    public static String serverAddress=defaultServerAddress;
    public static long hopsToLive;
    static public SimplifiedClient sc;


    static public void main(String[] args) {
	// parse command line
	Params params = new Params(args);

	// check for help
	if (params.getParam("help") != null) {
	    usage();
	    return;
	}

	listenPort = params.getint("listenPort", defaultListenPort);
	params.setParam("listenPort", "0");
	Core.init(params);
	
        sc = new SimplifiedClient(params);
	// process options
	allowedHosts = params.getParam("allowedHosts",defaultAllowedHosts);
	//listenPort = params.getint("listenPort", defaultListenPort);
	serverAddress = params.getParam("serverAddress", defaultServerAddress);
	hopsToLive = params.getlong("htl", defaultHopsToLive);

	// set logging
	int threshold = StandardLogger.priorityOf
	    (params.getParam("logging", defaultLogging));
	int verbosity = params.getint("verbosity",defaultVerbosity);
	String fname = params.getParam("logFile",defaultLogFile);
	try {
	    if (!fname.equalsIgnoreCase("NO"))
		sc.setLogger(new StandardLogger(fname, verbosity, threshold));
	    else
		sc.setLogger(new StandardLogger(System.err, verbosity,
						 threshold));
	} catch (IOException e) {
	    sc.setLogger(new SuppressedLogger());
	}
	
        // In the future, should pass core.
        try {
          ServletConstructor constructor=new ServletConstructor("Freenet.contrib.fproxy.HttpHandlerServlet", params);
          ServletHandler sh=new ServletHandler(new ListeningAddress("tcp/"+listenPort), constructor, allowedHosts);

          sh.acceptConnections();

        } catch(Exception e)
        {
          sc.getLogger().log(null, "Could not listen on port "+listenPort, Logger.ERROR);
          System.exit(1);
        }
    }


    /**
     * Print usage information
     **/
    public static void usage() {
	System.out.println("fproxy version 0.3");
	System.out.println("Usage: fproxy [options]");
	System.out.println("");
	System.out.println("  -listenPort port           Local port to listen on");
	System.out.println("  -allowedHosts list         Comma-delimited list of allowed hosts");
	System.out.println("                                 or `all' to accept all connections");
	System.out.println("  -serverAddress address     Server node to connect to");
	System.out.println("  -htl hops-to-live          Hops-to-live for request");
	System.out.println("  -logFile file              Name of log file (`no' to log to console)");
	System.out.println("  -logging error|normal|minor|debugging");
	System.out.println("                             Logging level");
	System.out.println("  -verbosity 0-5             Verbosity of log messages");
	System.out.println("  -help                      Display this help and exit");
	System.out.println("");
	System.out.println("Send bug reports to Theodore Hong <twh1@doc.ic.ac.uk>.");
    }
}
