// SAXDriver.java: The SAX driver for AElfred.
// NO WARRANTY! See README, and copyright below.

package Freenet.contrib.xmlrpc.xml;

import Freenet.contrib.xmlrpc.xml.sax.Parser;
import Freenet.contrib.xmlrpc.xml.sax.InputSource;
import Freenet.contrib.xmlrpc.xml.sax.Locator;
import Freenet.contrib.xmlrpc.xml.sax.AttributeList;
import Freenet.contrib.xmlrpc.xml.sax.EntityResolver;
import Freenet.contrib.xmlrpc.xml.sax.DTDHandler;
import Freenet.contrib.xmlrpc.xml.sax.DocumentHandler;
import Freenet.contrib.xmlrpc.xml.sax.ErrorHandler;
import Freenet.contrib.xmlrpc.xml.sax.HandlerBase;
import Freenet.contrib.xmlrpc.xml.sax.SAXException;
import Freenet.contrib.xmlrpc.xml.sax.SAXParseException;

import Freenet.contrib.xmlrpc.xml.XmlParser;
import Freenet.contrib.xmlrpc.xml.XmlHandler;

import java.io.IOException;
import java.io.Reader;
import java.util.Enumeration;
import java.util.Stack;
import java.util.Locale;
import java.util.Vector;


/**
  * A SAX driver for Microstar's &AElig;lfred XML parser.
  *
  * <p>This driver acts as a front-end for &AElig;lfred, and 
  * translates &AElig;lfred's events into SAX events.  It 
  * implements the SAX parser interface, and you can use it without 
  * directly calling &AElig;lfred at all:</p>
  *
  * <pre>
  * Freenet.contrib.xmlrpc.xml.sax.Parser parser = new Freenet.contrib.xmlrpc.xml.SAXDriver();
  * </pre>
  *
  * <p>When you are using SAX, you do not need to use the
  * <code>XmlParser</code> or <code>XmlHandler</code> classes at
  * all: this class is your entry point.</p>
  *
  * <p>This driver is based on the 1.0gamma version of SAX,
  * available from http://www.megginson.com/SAX/</p>
  *
  * @author Copyright (c) 1998 by Microstar Software Ltd.
  * @author written by David Megginson &lt;dmeggins@microstar.com&gt;
  * @version 1.1
  * @see Freenet.contrib.xmlrpc.xml.sax.Parser
  * @see XmlParser
  */
public class SAXDriver
  implements XmlHandler, Locator, AttributeList, Parser
{


  //
  // Variables.
  //

  private HandlerBase base = new HandlerBase();
  private XmlParser parser;

  private boolean seenDTDEvents = false;

				// Encapsulate the default behaviour
				// from HandlerBase
  private EntityResolver entityResolver = base;
  private DTDHandler dtdHandler = base;
  private DocumentHandler documentHandler = base;
  private ErrorHandler errorHandler = base;

  private String elementName = null;
  private Stack entityStack = new Stack();

  private Vector attributeNames = new Vector();
  private Vector attributeValues = new Vector();



  //
  // Constructor.
  //

  public SAXDriver ()
  {
  }



  //
  // Implementation of Freenet.contrib.xmlrpc.xml.sax.Parser.
  //

  /**
    * Set the locale.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#setLocale
    */
  public void setLocale (Locale locale)
    throws SAXException
  {
    throw new SAXException("AElfred driver does not yet have locale support.");
  }


  /**
    * Set the entity resolver for this parser.
    * @param handler The object to receive entity events.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#setEntityResolver
    */
  public void setEntityResolver (EntityResolver resolver) 
  {
    this.entityResolver = resolver;
  }


  /**
    * Set the DTD handler for this parser.
    * @param handler The object to receive DTD events.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#setDTDHandler
    */
  public void setDTDHandler (DTDHandler handler) 
  {
    this.dtdHandler = handler;
  }


  /**
    * Set the document handler for this parser.
    * @param handler The object to receive document events.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#setDocumentHandler
    */
  public void setDocumentHandler (DocumentHandler handler) 
  {
    this.documentHandler = handler;
  }


  /**
    * Set the error handler for this parser.
    * @param handler The object to receive error events.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#setErrorHandler
    */
  public void setErrorHandler (ErrorHandler handler) 
  {
    this.errorHandler = handler;
  }


  /**
    * Parse a document.
    * <p>If you want anything useful to happen, you should set
    * at least one type of handler.
    * @param source The XML input source.
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#parse(InputSource)
    * @see #setEntityResolver
    * @see #setDTDHandler
    * @see #setDocumentHandler
    * @see #setErrorHandler
    * @exception SAXException The handlers may throw any exception.
    */
  public void parse (InputSource source)
    throws SAXException
  {


    parser = new XmlParser();

    parser.setHandler(this);

    try {
      if (source.getCharacterStream() != null) {
	parser.parse(source.getSystemId(),
		     source.getPublicId(),
		     source.getCharacterStream());
      } else if (source.getByteStream() != null) {
	parser.parse(source.getSystemId(),
		     source.getPublicId(),
		     source.getByteStream(),
		     source.getEncoding());
      } else {
	parser.parse(source.getSystemId(),
		     source.getPublicId(),
		     source.getEncoding());
      }
    } catch (SAXException e) {
      throw e;
    } catch (Exception e) {
			//			e.printStackTrace(System.err);
      throw new SAXException(e);
    } finally {
      try {
	closeStreams(source);
      } catch (Exception e) {};
    }

    try {
      closeStreams(source);
    } catch (IOException e) {
      throw new SAXException(e);
    }
  }


  /**
    * Parse an XML document from a system identifier (URI).
    * @see Freenet.contrib.xmlrpc.xml.sax.Parser#parse(String)
    */
  public void parse (String systemId)
    throws SAXException
  {
    parse(new InputSource(systemId));
  }


  /**
    * Close any streams provided.
    */
  private void closeStreams (InputSource source)
    throws IOException
  {
    if (source.getCharacterStream() != null) {
      source.getCharacterStream().close();
    }
    if (source.getByteStream() != null) {
      source.getByteStream().close();
    }
  }



  //
  // Implementation of Freenet.contrib.xmlrpc.xml.XmlHandler.
  // This is where the driver receives AElfred events and translates
  // them into SAX events.
  //


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#startDocument.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#startDocument
    * @exception SAXException May throw any exception.
    */
  public void startDocument ()
    throws SAXException
  {
    documentHandler.setDocumentLocator(this);
    documentHandler.startDocument();
    attributeNames.removeAllElements();
    attributeValues.removeAllElements();
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#endDocument.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#endDocument
    * @exception SAXException May throw any exception.
    */
  public void endDocument ()
    throws SAXException
  {
    documentHandler.endDocument();
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler.resolveSystemId
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#resolveEntity
    * @exception SAXException May throw any exception.
    */
  public Object resolveEntity(String publicId, String systemId)
    throws SAXException, IOException
  {
    InputSource source = entityResolver.resolveEntity(publicId,
						      systemId);

    if (source == null) {
      return null;
    } else if (source.getCharacterStream() != null) {
      return source.getCharacterStream();
    } else if (source.getByteStream() != null) {
      return source.getByteStream();
    } else {
      return source.getSystemId();
    }
				// FIXME: no way to tell AElfred
				// about a new public id.
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#startExternalEntity.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#startExternalEntity
    * @exception SAXException May throw any exception.
    */
  public void startExternalEntity (String systemId)
    throws SAXException
  {
    entityStack.push(systemId);
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#endExternalEntity.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#endExternalEntity
    * @exception SAXException May throw any exception.
    */
  public void endExternalEntity (String systemId)
    throws SAXException
  {
    entityStack.pop();
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#doctypeDecl.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#doctypeDecl
    * @exception SAXException May throw any exception.
    */
  public void doctypeDecl (String name, String publicId, String systemId)
    throws SAXException
  {
    // no op
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#attribute.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#attribute
    * @exception SAXException May throw any exception.
    */
  public void attribute (String aname, String value, boolean isSpecified)
    throws SAXException
  {
    if (value != null) {
      attributeNames.addElement(aname);
      attributeValues.addElement(value);
    }
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#startElement.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#startElement
    * @exception SAXException May throw any exception.
    */
  public void startElement (String elname)
    throws SAXException
  {
				// We should deliver all DTD events
				// before the first startElement event.
    if (!seenDTDEvents) {
      deliverDTDEvents();
      seenDTDEvents = true;
    }

    elementName = elname;
    documentHandler.startElement(elname, this);
    elementName = null;
    attributeNames.removeAllElements();
    attributeValues.removeAllElements();
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#endElement.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#endElement
    * @exception SAXException May throw any exception.
    */
  public void endElement (String elname)
    throws SAXException
  {
    documentHandler.endElement(elname);
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#charData.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#charData
    * @exception SAXException May throw any exception.
    */
  public void charData (char ch[], int start, int length)
    throws SAXException
  {
    documentHandler.characters(ch, start, length);
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#ignorableWhitespace.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#ignorableWhitespace
    * @exception SAXException May throw any exception.
    */
  public void ignorableWhitespace (char ch[], int start, int length)
    throws SAXException
  {
    documentHandler.ignorableWhitespace(ch, start, length);
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#processingInstruction.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#processingInstruction
    * @exception SAXException May throw any exception.
    */
  public void processingInstruction (String target, String data)
    throws SAXException
  {
    documentHandler.processingInstruction(target, data);
  }


  /**
    * Implement Freenet.contrib.xmlrpc.xml.XmlHandler#error.
    * <p>Translate to the SAX interface.
    * <p>Users should never invoke this method directly.
    * @see Freenet.contrib.xmlrpc.xml.XmlHandler#error
    * @exception SAXException May throw any exception.
    */
  public void error (String message, String url, int line, int column)
    throws SAXException
  {
      errorHandler.fatalError(new SAXParseException(message, null,
						    url, line, column));
  }


  /**
    * Before the first startElement event, deliver all notation
    * and unparsed entity declarations.
    */
  private void deliverDTDEvents ()
    throws SAXException
  {
    String ename;
    String nname;
    String publicId;
    String systemId;

    Enumeration notationNames = parser.declaredNotations();
    Enumeration entityNames = parser.declaredEntities();

				// First, report all notations.
    while (notationNames.hasMoreElements()) {
      nname = (String)notationNames.nextElement();
      publicId = parser.getNotationPublicId(nname);
      systemId = parser.getNotationSystemId(nname);
      dtdHandler.notationDecl(nname, publicId, systemId);
    }

				// Next, report all unparsed entities.
    while (entityNames.hasMoreElements()) {
      ename = (String)entityNames.nextElement();
      if (parser.getEntityType(ename) == XmlParser.ENTITY_NDATA) {
	publicId = parser.getEntityPublicId(ename);
	systemId = parser.getEntitySystemId(ename);
	nname = parser.getEntityNotationName(ename);
	dtdHandler.unparsedEntityDecl(ename, publicId, systemId, nname);
      }
    }
  }



  //
  // Implementation of Freenet.contrib.xmlrpc.xml.sax.AttributeList.
  //

  public int getLength ()
  {
    return attributeNames.size();
  }

  public String getName (int i)
  {
    return (String)(attributeNames.elementAt(i));
  }


  public String getType (int i)
  {
    switch (parser.getAttributeType(elementName, getName(i))) {

    case XmlParser.ATTRIBUTE_UNDECLARED:
    case XmlParser.ATTRIBUTE_CDATA:
      return "CDATA";
    case XmlParser.ATTRIBUTE_ID:
      return "ID";
    case XmlParser.ATTRIBUTE_IDREF:
      return "IDREF";
    case XmlParser.ATTRIBUTE_IDREFS:
      return "IDREFS";
    case XmlParser.ATTRIBUTE_ENTITY:
      return "ENTITY";
    case XmlParser.ATTRIBUTE_ENTITIES:
      return "ENTITIES";
    case XmlParser.ATTRIBUTE_NMTOKEN:
    case XmlParser.ATTRIBUTE_ENUMERATED:
      return "NMTOKEN";
    case XmlParser.ATTRIBUTE_NMTOKENS:
      return "NMTOKENS";
    case XmlParser.ATTRIBUTE_NOTATION:
      return "NOTATION";

    }

    return null;
  }


  public String getValue (int i)
  {
    return (String)(attributeValues.elementAt(i));
  }


  public String getType (String name)
  {
    for (int i = 0; i < getLength(); i++) {
      if (name.equals(getName(i))) {
	return getType(i);
      }
    }
    return null;
  }


  public String getValue (String name)
  {
    for (int i = 0; i < getLength(); i++) {
      if (name.equals(getName(i))) {
	return getValue(i);
      }
    }
    return null;
  }



  //
  // Implementation of Freenet.contrib.xmlrpc.xml.sax.Locator.
  //

  public String getPublicId ()
  {
    return null;		// TODO
  }

  public String getSystemId ()
  {
    return (String)(entityStack.peek());
  }

  public int getLineNumber ()
  {
    return parser.getLineNumber();
  }

  public int getColumnNumber ()
  {
    return parser.getColumnNumber();
  }

}
