package Freenet.crypt;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

import java.io.*;

/**
 * Implements a Java InputStream that is encrypted with any symmetric block
 * cipher (implementing the BlockCipher interface).
 * 
 * This stream operates in Periodic Cipher Feedback Mode (PCFB), allowing 
 * byte at a time encryption with no additional encryption workload.
 */
public class CipherInputStream extends InputStream {
    static final int BUFFER_SIZE = 4 * 1024;

    private PCFBMode ctx;
    private InputStream in;
    private byte [] buf;
    private int bufOff, bufLen;

    public CipherInputStream(BlockCipher c, InputStream in) throws IOException {
	this(c, in, true);
    }

    public CipherInputStream(BlockCipher c, InputStream in, byte[] iv)
        throws IOException {
	this(c, in, false);
	ctx.acceptIV(iv);
    }

    public CipherInputStream(BlockCipher c, InputStream in, boolean readIV) 
	throws IOException {
	ctx=new PCFBMode(c);
	if (readIV)
            ctx.readIV(in);
        this.in = in;
    }

    public int available() throws IOException {
        if (bufOff == bufLen)
            return in.available();
        return bufLen - bufOff;
    }

    public void close() throws IOException {
	in.close();
    }

    public boolean markSupported() {
	return false;
    }

    public int read() throws IOException {
        int bytesRead;

        /* if buf empty, read some more */
        if (bufOff == bufLen) {
           if (buf == null)
               buf = new byte[BUFFER_SIZE];

           bytesRead = in.read(buf, 0, buf.length);
           if (bytesRead <= 0)
               return bytesRead;
           bufOff = 0;
           bufLen = bytesRead;

           /* decipher the bytes read */
           ctx.decipher(buf, 0, bufLen);
        }

        return buf[bufOff++] & 0xff;
    }

    public int read(byte[] b, int offset, int length) throws IOException {
        int bytesRead;

        if (length == 0)
            return 0;

        /* if some data left in the buffer, return it */
        if (bufOff < bufLen) {
            bytesRead = bufLen - bufOff;
            if (bytesRead > length)
                bytesRead = length;
            System.arraycopy(buf, bufOff, b, offset, bytesRead);
            bufOff += bytesRead;

            return bytesRead;
        }

        /* read directly into the user buffer and decipher */
        bytesRead = in.read(b, offset, length);
        if (bytesRead > 0)
            ctx.decipher(b, offset, bytesRead);

        return bytesRead;
    }
}



