package Freenet.crypt;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/
import java.io.*;

/**
 * Implements a Java OutputStream that is encrypted with any symmetric block
 * cipher (implementing the BlockCipher interface).
 * 
 * This stream operates in Periodic Cipher Feedback Mode (PCFB), allowing 
 * byte at a time encryption with no additional encryption workload.
 */
public class CipherOutputStream extends OutputStream {
    static final int BUFFER_SIZE = 4 * 1024;

    private PCFBMode ctx;
    private OutputStream out;
    private byte [] buf;
    private int bufOff;

    public CipherOutputStream(BlockCipher c, OutputStream out)
        throws IOException {
	ctx = new PCFBMode(c);
	this.out = out;
    }

    public CipherOutputStream(BlockCipher c, OutputStream out, byte[] iv)
        throws IOException {
	this(c, out);
	ctx.acceptIV(iv);
    }

    public CipherOutputStream(RandomSource rs, BlockCipher c, OutputStream out) 
	throws IOException {
        this(c, out);
	ctx.writeIV(rs, out);
    }

    public void flush() throws IOException {
        if (bufOff > 0) {
            out.write(buf, 0, bufOff);
            bufOff = 0;
        }
	out.flush();
    }

    public void close() throws IOException {
        if (bufOff > 0) {
            out.write(buf, 0, bufOff);
            bufOff = 0;
        }
	out.close();
    }

    public void write(byte[] b, int offset, int length) throws IOException {
        if (length == 0)
            return;

        /* alloc buffer space */
        if (buf == null) {
            if (length > BUFFER_SIZE)
                buf = new byte[length];
            else
                buf = new byte[BUFFER_SIZE];
        }

        /* if length of data is more than buf space left, empty buffer */
        if (bufOff > 0 && length > buf.length - bufOff) {
            out.write(buf, 0, bufOff);
            bufOff = 0;
        }

        /* if length is bigger than current buffer size, make buf bigger */
        if (length > buf.length) {
            if (bufOff > 0) {
                out.write(buf, 0, bufOff);
                bufOff = 0;
            }
            buf = new byte[length];
        }

        /* copy and encipher */
        System.arraycopy(b, offset, buf, bufOff, length);
        ctx.encipher(buf, bufOff, length);
        bufOff += length;
    }

    public void write(int b) throws IOException {
        if (buf == null) {
            buf = new byte[BUFFER_SIZE];
        }

        /* flush if buffer full */
        if (bufOff == buf.length) {
            out.write(buf, 0, bufOff);
            bufOff = 0;
        }

        /* add byte to buffer */
	buf[bufOff++] = (byte) ctx.encipher(b);
    }
}






