package Freenet.crypt;

import java.math.BigInteger;
import java.util.Random;
import java.io.*;

/**
 * Implements the Digital Signature Algorithm (DSA) described in FIPS-186
 */
public class DSA {

    /**
     * Returns a DSA signature given a group, private key (x), a random nonce
     * (k), and the hash of the message (m).
     */
    public static DSASignature sign(DSAKeyPair kp,
				    BigInteger k, 
				    BigInteger m) {
	BigInteger r=kp.getG().modPow(k, kp.getP()).mod(kp.getQ());

	BigInteger kInv=k.modInverse(kp.getQ());
	return sign(kp, r, kInv, m);
    } 

    /**
     * Precalculates a number of r, kInv pairs given a random source
     */
    public static BigInteger[][] signaturePrecalculate(DSAKeyPair kp,
						       int count, Random r) {
	BigInteger[][] result=new BigInteger[count][2];

	for (int i=0; i<count; i++) {
	    BigInteger k;
	    do {
		k=Util.generateLargeRandom(80, 160, r);
	    } while (k.compareTo(kp.getQ())>-1 || k.compareTo(Util.ZERO)==0);
	    
	    result[i][0] = kp.getG().modPow(k, kp.getP()); // r 
	    result[i][1] = k.modInverse(kp.getQ()); // k^-1 
	}
	return result;
    }

    /**
     * Returns a DSA signature given a group, private key (x), 
     * the precalculated values of r and k^-1, and the hash
     * of the message (m)
     */
    public static DSASignature sign(DSAKeyPair kp,
				    BigInteger r, BigInteger kInv, 
				    BigInteger m) {
	BigInteger s1=m.add(kp.getX().multiply(r)).mod(kp.getQ());
	BigInteger s=kInv.multiply(s1).mod(kp.getQ());
	return new DSASignature(r,s);
    }

    /**
     * Verifies the message authenticity given a group, the public key
     * (y), a signature, and the hash of the message (m).
     */
    public static boolean verify(DSAKeyPair kp,
				 DSASignature sig,
				 BigInteger m) {
	BigInteger w=sig.getS().modInverse(kp.getQ());
	BigInteger u1=m.multiply(w).mod(kp.getQ());
	BigInteger u2=sig.getR().multiply(w).mod(kp.getQ());
	BigInteger v1=kp.getG().modPow(u1, kp.getP());
	BigInteger v2=kp.getY().modPow(u2, kp.getP());
	BigInteger v=v1.multiply(v2).mod(kp.getP()).mod(kp.getQ());
	return v.equals(sig.getR());
    }
	
    public static void main(String[] args) throws Exception {
	Yarrow r=new Yarrow();
	SHA1 ctx=new SHA1();
	File f=new File(args[1]);
	File o=new File(args[2]);
	FileInputStream keyf=new FileInputStream(args[0]);
	DSAKeyPair key=(DSAKeyPair)CryptoKey.read(keyf);

	byte[] hash=Util.hashFile(ctx, f);
	BigInteger m=Util.byteArrayToMPI(hash);

	if (args.length==4 && 
	    args[3].equals("sign")) {
	    BigInteger k=new BigInteger(159, r);
	    BigInteger[][] precal=signaturePrecalculate(key, 2000, r);

	    DSASignature sig=null;

	    long start=System.currentTimeMillis();
	    for (int i=0; i<2000; i++) {
		sig=sign(key, 
			 //			 Util.generateLargeRandom(80,160,r),
			 precal[i][0], precal[i][1],
			 m);
	    }
	    System.out.println("Time:"+(System.currentTimeMillis()-start));
	    FileOutputStream out=new FileOutputStream(o);
	    sig.write(out);
	    out.close();
	} else {
	    FileInputStream sigf=new FileInputStream(o);
	    DSASignature sig=new DSASignature(sigf);
	    long start=System.currentTimeMillis();
	    for (int i=0; i<1000; i++) {
		verify(key, sig, m);
	    }
	    System.out.println("Time:"+(System.currentTimeMillis()-start));
	    System.err.println(verify(key, sig, m));
	}
    }
}

	

