package Freenet.crypt;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/
import java.io.*;

/**
 * Control mechanism for the Period Cipher Feed Back mode.  This is
 * a CFB variant used apparently by a number of programs, including PGP. 
 * Thanks to Hal for suggesting it.
 */
public class PCFBMode {
    
    private BlockCipher c;
    private byte[] feedback_register, buffer;
    protected int registerPointer, blockSizeBytes;
    
    public PCFBMode(BlockCipher c) {
	this.c=c;
	blockSizeBytes=c.getBlockSize()>>3;
	feedback_register=new byte[blockSizeBytes];
	buffer=new byte[feedback_register.length];
	registerPointer=feedback_register.length;
    }

    /**
     * Writes the initialization vector to the stream.  Though the IV
     * is transmitted in the clear, this gives the attacker no additional 
     * information because the registerPointer is set so that the encrypted
     * buffer is empty.  This causes an immediate encryption of the IV,
     * thus invalidating any information that the attacker had.
     */
    void writeIV(RandomSource rs, OutputStream out) throws IOException {
	rs.nextBytes(feedback_register);
	out.write(feedback_register);
    }
    
    /**
     * Reads the initialization vector from the given stream.  
     */
    void readIV(InputStream in) throws IOException {
	for (int i=0; i<feedback_register.length; i++)
	    feedback_register[i]=(byte)in.read();
    }

    public void acceptIV(byte[] iv) {
	for (int i=0; i<iv.length; i++) 
	    feedback_register[i]=iv[i];
    }

    /**
     * Deciphers one byte of data, by XOR'ing the ciphertext byte with
     * one byte from the encrypted buffer.  Then places the received
     * byte in the feedback register.  If no bytes are available in 
     * the encrypted buffer, the feedback register is encrypted, providing
     * block_size/8 new bytes for decryption
     */
    int decipher(int b) {
	if (registerPointer<feedback_register.length) {
	    int rv=(feedback_register[registerPointer] ^ (byte)b) & 0xff;
	    feedback_register[registerPointer++]=(byte)b;
	    return rv;
	} else {
	    refillBuffer();
	    return decipher(b);
	}
    }

    /**
     * Enciphers one byte of data, by XOR'ing the plaintext byte with
     * one byte from the encrypted buffer.  Then places the enciphered 
     * byte in the feedback register.  If no bytes are available in 
     * the encrypted buffer, the feedback register is encrypted, providing
     * block_size/8 new bytes for encryption
     */
    int encipher(int b) {
	if (registerPointer<feedback_register.length) {
	    feedback_register[registerPointer]=
		(byte)(feedback_register[registerPointer] ^ b);
	    return feedback_register[registerPointer++] & 0xff;
	} else {
	    refillBuffer();
	    return encipher(b);
	}

    }

    /**
     * Deciphers length bytes of data, by XOR'ing the ciphertext bytes with
     * bytes from the encrypted buffer.  This routine will fail if
     * feedback register is less than 8 bytes or if it is not modulus 8.
     */
    void decipher(byte[] buff, int offset, int length) {
        int endOffset;
        byte swap;

        endOffset = offset + length;

        if ( offset >= endOffset )
          return;

        loop: for (;;) {
            if (registerPointer == feedback_register.length)
                refillBuffer();

            if (endOffset - offset >= feedback_register.length) {

                /* this assumes that feedback is at least 8 bytes and
                   has a length modulus 8 */
                switch (registerPointer & 0x7) {
                    case 0:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 1:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 2:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 3:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 4:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 5:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 6:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                    case 7:
                        swap = buff[offset];
                        buff[offset++] ^= feedback_register[registerPointer];
                        feedback_register[registerPointer++] = swap;
                        break;
                }

                if (offset == endOffset)
                    break loop;
            }
            else {
                do {
                    swap = buff[offset];
                    buff[offset++] ^= feedback_register[registerPointer];
                    feedback_register[registerPointer++] = swap;

                    if (offset == endOffset)
                        break loop;
                } while (registerPointer < feedback_register.length);
            }
        }
    }

    /**
     * Enciphers length byte of data, by XOR'ing the plaintext bytes with
     * bytes from the encrypted buffer.  This routine will fail if
     * feedback register is less than 8 bytes or if it is not modulus 8.
     */
    void encipher(byte[] buff, int offset, int length) {
        int endOffset;

        endOffset = offset + length;

        if ( offset >= endOffset )
          return;

        loop: for (;;) {
            if (registerPointer == feedback_register.length)
                refillBuffer();

            if (endOffset - offset >= feedback_register.length) {

                /* this assumes that feedback is at least 8 bytes and
                   has a length modulus 8 */
                switch (registerPointer & 0x7) {
                    case 0:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 1:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 2:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 3:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 4:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 5:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 6:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                    case 7:
                        feedback_register[registerPointer] ^= buff[offset];
                        buff[offset++] = feedback_register[registerPointer++];
                        break;
                }

                if (offset == endOffset)
                   break loop;
            }
            else {
                do {
                    feedback_register[registerPointer] ^= buff[offset];
                    buff[offset++] = feedback_register[registerPointer++];

                    if (offset == endOffset)
                      break loop;
                } while (registerPointer < feedback_register.length);
            }
        }
    }

    byte[] blockEncipher(byte[] buff, int offset) {
	for (int i=0; i<blockSizeBytes; i++) {
	    feedback_register[i]=buffer[i]=
		(byte)(buff[i+offset] ^ feedback_register[i]);
	}
	refillBuffer();
	return buffer;
    }

        // Refills the encrypted buffer with data.
    protected void refillBuffer() {
	feedback_register=c.encipher(feedback_register);
	registerPointer=0;
    }
}
	
