package Freenet.crypt;
import Freenet.support.io.*;
import Freenet.support.FileBucket;
import Freenet.Presentation;
import Freenet.Key;
import Freenet.Address;

import java.io.*;
import java.util.Stack;
import java.io.IOException;


/**
 * A progressive hash stream is a stream of data where each part is preceded
 * by a hash of that part AND the hash of the next part. This means
 * that only the hash of the first part (and second hash) needs to be known, 
 * but one can be sure that each part is valid after reading it.
 * 
 * This class provides a VerifyingInputStream that verifies each part of the 
 * hash on reading.
 *
 * The design of the progressive hash as used in Freenet is taken from:
 * Gennaro, R and Rohatgi, P; "How to Sign Digital Streams", 
 * Advances in Cryptology - CRYPTO '97, 1997.
 *
 * @author oskar
 **/

public class ProgressiveHashInputStream extends VerifyingInputStream {

     public static void main(String[] args) throws Exception {
	 File data = new File(args[0]);
	 FileInputStream fin = new FileInputStream(data); 
	 long partSize = Long.parseLong(args[1]);
	 ProgressiveHashOutputStream hout = new 
	     ProgressiveHashOutputStream(partSize, new SHA1Factory(), new FileBucket());
	 int left = (int) data.length(), n = 0;
	 byte[] b = new byte[1024];
	 while (left > 0) {
	     n = fin.read(b,0,Math.min(left,b.length));
	     hout.write(b,0,n);
	     left -= n;
	 }

	 hout.finish();

	 byte[] init = hout.getDigest();
	 for (int i = 0; i < init.length ; System.err.print((init[i++] & 0xff) + " ")) {} 

	 long totalLength = hout.getLength();
	 System.err.println("\nTotalLength = " + totalLength);

	 InputStream in = hout.getData();

	 VerifyingInputStream vin = new ProgressiveHashInputStream(in, partSize,totalLength, new SHA1(),init);
	 //	 VerifyingInputStream vin = new ControlInputStream(in, partSize+20,totalLength);

	 vin.stripControls(args.length < 3 || Integer.parseInt(args[2]) == 0 ?
			   true :
			   false);
	 InputStream bin = new BufferedInputStream(vin);
	 // uncomment to test fucking up
	 // in.read();    
	 //	while (vin.available() > 0) {
	 int i = bin.read(b);
	 while (i > 0) {
	     System.out.write(b,0,i);
	     i = bin.read(b);
	 }
    }

    private long partSize;
    private long read = 0;
    //    private DigestFactory df;
    private Digest dig;
    private byte[] expect;
    private byte[] buf;
    private byte[] res;
    private int pos = 0;
    private int endsWith = -1;
    private int dnvDiedCode = -1;
    

    /**
     * Create a new InputStream that verifies a stream of Serially hashed data
     * @param in           The inputstream to read.
     * @param partSize     The amount of data preceding each digest value and
     *                     control character.
     * @param dataLength   The total length of the data.
     * @param dig          A digest of the type needed.
     * @param firstDig     The Digest value to expect for the first part (and
     *                     second digest value). The length of the digest bites
     *                     will be copied starting from the first.
     * @exception DataNotValidIOException is thrown if the partSize combined
     *            with the datalength produces an impossible EOF.
     **/
    public ProgressiveHashInputStream(InputStream in, long partSize, 
				      long dataLength, Digest dig, 
				      byte[] firstDig) 
	throws DataNotValidIOException {
	super(in,dataLength);
	this.partSize = partSize;
	this.dig = dig;
	int ds = dig.digestSize() >> 3;
	// Sanity check
	long lastPart = dataLength % (partSize + ds + 1);
	//System.out.println ("partSize = " + partSize + " dataLength = " + 
	//		    dataLength + " lastPart = " + lastPart);
	if (partSize < 0 
	    || dataLength < 2
	    || (partSize != 0 
		&& (partSize > dataLength - 23 
		    || lastPart > partSize + 1
		    || lastPart < 2)))
	    throw new DataNotValidIOException(Presentation.CB_OK);
	// ok so this isn't elegant. Fuck you. I'm tired of these streams!
	if (partSize == 0)
	    this.partSize = ((long) 1) << 63 - 1;
	this.expect = new byte[ds];
	this.buf = new byte[ds];
	this.res = new byte[ds];
	System.arraycopy(firstDig,0,expect,0,ds);
    }

    public int read() throws DataNotValidIOException, IOException {
	if (dnvDiedCode != -1)
	    throw new DataNotValidIOException(dnvDiedCode);
	int i = priv_read();
	if (stripControls && allRead) {
	    priv_read();
	}
	return i;
    }

    private int priv_read() throws DataNotValidIOException, IOException {
	if (endsWith != -1) {
	    valPart(endsWith);
	    return endsWith;
	}
	    
	if (finished)
	    return -1;
	int n = super.read();
	if (n == -1) {
	    throw new EOFException("Unexpected end of stream");
	}
	if (read < partSize) {
	    read++;
	    if (!finished)
		dig.update((byte) n);
	    else {
		valPart(n);
		if (stripControls)
		    n = -1;
	    }
	} else {
	    if (pos < buf.length) {
		if (finished)
		    throw new EOFException("End of stream during control read");
		buf[pos++] = (byte) n;
		dig.update((byte)n);
	    } else {
		valPart(n);
	    }
	    if (stripControls)
		n = priv_read();  // this will keep on going until we run out of controls
	}
	return n;
    }

    public int read(byte[] b,int off, int length) throws IOException,
    DataNotValidIOException {
	if (dnvDiedCode != -1)
	    throw new DataNotValidIOException(dnvDiedCode);
	int i = priv_read(b,off,length);
	if (stripControls && allRead) {
	    priv_read();
	}
	return i;
    }

    private int priv_read(byte[] b, int off, int length) throws IOException,
    DataNotValidIOException {
	if (length == 0)
	    return 0;
	if (endsWith != -1) {
	    valPart(endsWith);
	    b[off] = (byte) endsWith;
	    return 1;
	}
	if (finished)
	    return -1;
	
	if (read == partSize) {
	    if (pos < buf.length) {
		int l = length < (buf.length-pos) ? length : (buf.length-pos);
		int n = super.read(b,off,l);
		if (n == -1)
		    throw new EOFException("EOF while reading controls");
		dig.update(b,off,n);
		System.arraycopy(b,off,buf,pos,n); // copy into buffer
		pos += n;
		if (stripControls) {
		    return priv_read(b,off,length);			
		} else
		    return n;
	    } else {
		int n = super.read();
		valPart(n);
		if (stripControls) {
		    return priv_read(b,off,length);
		} else {
		    b[off] = (byte) n;
		    return 1 + priv_read(b,off+1,length-1);
		}
	    }    
	} else {
	    int newlen = (read + length > partSize ?
			  (int) (partSize - read) :
			  length);
	    int n = super.read(b,off,newlen);
	    if (n == -1)
		throw new EOFException("Unexpected end of stream");
	    read += n;
	    if (!finished) {
		dig.update(b,off,n);
		return n + (newlen < length && read == partSize ? 
			    priv_read(b,off+n,length - newlen) :
			    0);
	    } else {
		if (n == 1) {
		    valPart(b[off + n - 1] & 0xff);
		    return !stripControls ? 1 : -1;
		} else {
		    endsWith = b[off + n -1] & 0xff;
		    n--;
		    return n;
		}
	    }
	}
    }

    private void valPart(int n) throws DataNotValidIOException {
	// take digest
	dig.digest(true,res,0);
	// check against expected

	if (!same(res,expect) || n != (Presentation.CB_OK & 0xff)) {
	    dnvDiedCode = n;
	    throw new DataNotValidIOException(n);
	}

	// copy in next expected
	System.arraycopy(buf,0,expect,0,buf.length);
	// reset
	read = 0;
	pos = 0;
    }

    private boolean same(byte[] a, byte[] b) {
	if (a.length != b.length)
	    return false;
	//boolean bo = true;
	for (int i = 0; i < a.length ; i++) {
	    //	    System.err.println(a[i] + " = " + b[i]);
	    if (a[i] != b[i])
		return false;
	}
	return true;
    }

    public void finish() throws DataNotValidIOException, IOException {
	finished = true;
    }
}








