package Freenet.crypt;
import Freenet.support.io.*;
import Freenet.support.Bucket;
import Freenet.Presentation;
import java.io.*;
import java.util.Stack;

/**
 * A progressive hash stream is a stream of data where each part is preceded
 * by a hash of that part AND the hash of the next part. This means
 * that only the hash of the first part (and second hash) needs to be known, 
 * but one can be sure that each part is valid after reading it.
 * 
 * This class provides an OutputStream to which the data must be written,
 * after which it can be streamed out again with the correct hash values
 * and control bytes interleaved. It will require enough temporary 
 * diskspace to write the data.
 *
 * The design of the progressive hash as used in Freenet is taken from:
 * Gennaro, R and Rohatgi, P; "How to Sign Digital Streams", 
 * Advances in Cryptology - CRYPTO '97, 1997.
 *
 * @author oskar
 **/

public class ProgressiveHashOutputStream extends OutputStream {

    private Bucket b;
    private OutputStream out;
    private DigestFactory df;
    private Digest dig;
    private Stack digests;
    private long written;
    private long partSize;

    private long totalLength;
    private InputStream in;
    private byte[] value;
    private Stack dvals;

    /**
     * Create a SerialHashOutputStream.
     *
     * @param partSize  The amount of data in each part. After each part the 
     *                  digest value and one control byte will be written.
     * @param df        A DigestFactory for creating Digest objects of the
     *                  kind in question.
     * @param b         The bucket to put the data in when it is processed.
     *                  Note that you can't simply take the inputstream from
     *                  this bucket after writing to this class, since it 
     *                  will not contain the interleaved hash values,
     *                  getData() provides that.
     * @return an InputStream of the original data, interleaved after every
     *         partSize bytes with the next digest value and a control char.
     **/
    public ProgressiveHashOutputStream(long partSize, 
				       DigestFactory df,
				       Bucket b) throws IOException {

	//	InputStream in = new FileInputStream(data);
	this.b = b;
	b.resetWrite();
	this.out = new BufferedOutputStream(b.getOutputStream());
	this.df = df;
	this.dig = df.getInstance();
	this.partSize = (partSize != 0 ? partSize : 
			 (((long) 1) << 63 - 1));
	this.digests  = new Stack();
    }

    public void write(int i)  throws IOException {
	priv_write(i);
    }

    private void priv_write(int i) throws IOException {
	if (written < partSize) {
	    dig.update((byte) i);
	    out.write(i);
	    written++;
	} else {
	    digests.push(dig);
	    dig = df.getInstance();
	    written = 0;
	    priv_write(i);
	}
    }
    
    public void write(byte[] b) throws IOException {
	write(b,0,b.length);
    }

    public void write(byte[] b, int off, int length) throws IOException {
	priv_write(b,off,length);
    }

    private void priv_write(byte[] b, int off, int length) throws IOException {
	if (length == 0)
	    return;
	else if (written + length <= partSize) {
	    dig.update(b,off,length);
	    out.write(b,off,length);
	    written += length;
	} else if (written == partSize) {
	    priv_write(b[off] & 0xff);
	    priv_write(b,off+1,length-1);
	} else {
	    int newlen = (int) (partSize - written);
	    priv_write(b, off, newlen);
	    priv_write(b, off + newlen, length - newlen);
	}	   
    }
    
    /**
     * Finalizes the stream, creating the data InputStream, digest value,
     * and total length count.
     */
    public synchronized void finish() throws IOException {
	close();

	// set hash values
	dvals = new Stack();
	byte[] dval = {};
	Digest d;

	while (!digests.empty()) {
	    d = (Digest) digests.pop();
	    d.update(dval,0,dval.length);
	    dval = d.digest();
	    dvals.push(dval);
	}

	// set digest value as last
	value = (byte[]) dvals.pop();

	// calculate length
	int parts = (int) (b.size() / partSize);
	long lastPart = b.size() % partSize;

	totalLength = parts * (partSize + 21) + lastPart + 1;
    }

    public void close() throws IOException {
	digests.push(dig);
	out.close();
    }

    /**
     * Returns the initial hash value of the Progressive hash.
     * @return the hash value if finish() has been called, null otherwise.
     **/
    public synchronized byte[] getDigest() {
	return value;
    }

    /**
     * Returns the total length of the new stream, with the interleaved
     * hash values and control bytes counted.
     * @return total length of resulting stream if finish() has been called,
     *         0 otherwise.
     **/
    public synchronized long getLength() {
	return totalLength;
    }
    
    /**
     * Return the an inputstream of the data with interleaved hash values
     * and control bytes.
     * @return The resulting stream if finish() has been called, null
     *         otherwise.
     **/
    public synchronized InputStream getData() throws IOException {
	return dvals == null ? 
	    null :
	    new InterleaveInputStream(b);
    }

    /**
     * A simple stream that pushes byte arrays (digest values) off the 
     * stack and adds them and a control byte to the stream after every 
     * partSize.
     **/    
    protected class InterleaveInputStream extends FilterInputStream {
	private long read;	
	private byte[] buf;
	private int pos;
	private boolean ended;
	private int dSize;
	private int dig;

	public InterleaveInputStream(Bucket b) throws IOException {
	    super(b.getInputStream());
	    read = 0;
	    pos = 0;
	    dig = dvals.size() - 1;
	    if (dig == -1)
		buf = null;
	    else
		buf = (byte[]) dvals.elementAt(dig--);
	    dSize = buf == null ? 0 : buf.length;
	}

	public int read() throws IOException {
	    return priv_read();
	}

	private int priv_read() throws IOException {
	    if (ended)
		return -1;
	    if (read == partSize) {
		if (buf == null) {
		    ended = true;
		    return Presentation.CB_OK;
		} else if (pos < buf.length) {
		    int t = buf[pos++] & 0xff; // int conversion
		    return t;
		} else {
		    pos = 0;
		    read = 0;
		    if (dig >= 0)
			buf = (byte[]) dvals.elementAt(dig--);
		    else
			buf = null;
		    return Presentation.CB_OK;
		}
	    } else {
		read++;
		int i = super.read();
		if (i == -1) {
		    ended = true;
		    return Presentation.CB_OK;
		} else
		    return i;
	    }
	}

	public int read(byte[] b, int off, int length) throws IOException {
	    return priv_read(b,off,length);
	}
	private int priv_read(byte[] b, int off, int length) 
	    throws IOException {
	    if (length == 0) 
		return 0;
	    else if (ended)
		return -1;
	    else if (read == partSize) {
		int n;
		if (pos < buf.length) {
		    n = length < (buf.length-pos) ? 
			length : (buf.length-pos);
		    System.arraycopy(buf,pos,b,off,n);
		    pos = pos + n;
		} else {
		    b[off] = (byte) priv_read();
		    n = 1;
		}
		return n;
	    } else {
		int n = (length < (partSize - read) ?
			 length : (int) (partSize - read));
		n = super.read(b,off,n);
		if (n == -1) {
		    b[off] = Presentation.CB_OK;
		    read += 1;
		    ended = true;
		    return 1;
		} else {
		    read += n;
		    return n;
		}
	    }
	}

	public int available() throws IOException {
	    //	    int i = super.available();
	    return ended ? 0 : super.available() + 1;
	}
    }









}
