package Freenet.crypt;

import java.math.BigInteger;
import java.util.Random;
import java.io.*;
import Freenet.Core;
import Freenet.support.Logger;

/**
 * Implements the RSA algorithm for encryption and digital signatures
 *
 * @author Scott G. Miller
 */
public class RSA {

    /**
     * Encrypts/Decrypts/Signs/Verifies a message, M, with the RSA key k.
     * If 'inverse' is set, the decryption key is used to encrypt (sign)
     */
    public static BigInteger rsa_algo(RSAKeyPair k, BigInteger M, 
				      boolean inverse) {
	if (M.bitLength() > k.getN().bitLength())
	    Core.logger.log(new RSA(), "Warning in RSA: Message size exceeds key size\n", Logger.ERROR);
	
	if (inverse) {
	    if (k.getP()!=null) {
		// Solve by Chinese Remainder Theorum
		BigInteger M1, M2, C;
		M1=M.modPow(k.getDmodPsub1(), k.getP());
		M2=M.modPow(k.getDmodQsub1(), k.getQ());
		C=M1.multiply(k.getQinvmodP()).multiply(k.getQ())
		.add(M2.multiply(k.getPinvmodQ()).multiply(k.getP()))
		.mod(k.getN());
		return C;
	    } else 
		return M.modPow(k.getD(), k.getN());
	} else 
	    return M.modPow(k.getE(), k.getN());
    }

    public static BigInteger encrypt(RSAKeyPair k, BigInteger M) {
	return rsa_algo(k, M, false);
    }

    public static BigInteger decrypt(RSAKeyPair k, BigInteger C) {
	return rsa_algo(k, C, true);
    }

    public static BigInteger sign(RSAKeyPair k, Random r, byte[] h) {
	BigInteger M=wrap(k,r,h);
	return decrypt(k, M);
    }

    public static boolean verify(RSAKeyPair k, BigInteger S, byte[] h) {
	BigInteger M=encrypt(k, S);
	byte[] m=unwrap(M);
	return Util.byteArrayEqual(m, h);
    }

    public static BigInteger wrap(RSAKeyPair k, Random r, byte[] m) {
	int bl=k.getN().bitLength();
	byte[] tmp=new byte[bl>>3];
	r.nextBytes(tmp);
	tmp[0]=(byte)m.length;
	System.arraycopy(m, 0, tmp, 1, m.length);
	return Util.byteArrayToMPI(tmp);
    }

    public static byte[] unwrap(BigInteger M) {
	byte[] data=M.toByteArray();
	int len=data[0];
	byte[] rv=new byte[len];
	System.arraycopy(data, 1, rv, 0, len);
	return rv;
    }

    public static void main(String[] args) throws Exception {
	Yarrow r=new Yarrow();
	SHA1 ctx=new SHA1();
	File f=new File(args[1]);
	File o=new File(args[2]);
	FileInputStream keyf=new FileInputStream(args[0]);
	RSAKeyPair key=(RSAKeyPair)CryptoKey.read(keyf);

	byte[] hash=Util.hashFile(ctx, f);
	if (args.length==4 && args[3].equals("test")) {
	    BigInteger m=Util.TWO;
	    System.err.println(m.toString(16));
	    BigInteger c=encrypt(key, m);
	    System.err.println(c.toString(16));
	    m=decrypt(key, c);
	    System.err.println(m.toString(16));
	    return;
	} if (args.length==4 && 
	    args[3].equals("sign")) {
	    long start=System.currentTimeMillis();
	    RSASignature sig=null;
	    for (int i=0; i<2000; i++) {
		sig=new RSASignature(sign(key, r, hash));
	    }
	    System.out.println("Time:"+(System.currentTimeMillis()-start));
	    FileOutputStream out=new FileOutputStream(o);
	    sig.write(out);
	    out.close();
	} else {
	    FileInputStream sigf=new FileInputStream(o);
	    RSASignature sig=new RSASignature(sigf);
	    long start=System.currentTimeMillis();
	    for (int i=0; i<2000; i++) {
		verify(key, sig.getS(), hash);
	    }
	    System.out.println("Time:"+(System.currentTimeMillis()-start));
	    System.err.println(verify(key, sig.getS(), hash));
	}
    }
}

	

