package Freenet.crypt;

import java.math.BigInteger;
import java.io.*;
import java.util.Random;

/**
 * Holds a RSA key pair.  The private key (d) is used for signing/encrypting
 * while the public key (e, n) is used for verification.
 */
public class RSAKeyPair extends KeyPair {

    /**
     * Generate a new RSA key pair
     */
    public static RSAKeyPair generate(int bitlength, Random r, boolean crt) {
	BigInteger p=new BigInteger(bitlength/2, 12, r);
	BigInteger q=new BigInteger(bitlength/2, 12, r);

	RSAKeyPair kp=null;
	while (kp==null) {
	    kp=generate(new BigInteger(bitlength/2, 15, r),
			new BigInteger(bitlength/2, 15, r), 
			crt);
	}
	return kp;
    }

    /**
     * Calculate a new RSA key pair given two primes
     */
    public static RSAKeyPair generate(BigInteger p, BigInteger q, 
				      boolean crt) {
	BigInteger pq=p.subtract(Util.ONE)
	    .multiply(q.subtract(Util.ONE));
	BigInteger e=BigInteger.valueOf(65537L);
	if (Util.ONE.compareTo(e.gcd(pq))!=0)
	    return null;

	BigInteger n=p.multiply(q);
	BigInteger d=e.modInverse(pq);
	return (crt ? new RSAKeyPair(d, e, n, p, q) :
		new RSAKeyPair(d, e, n));
    }

    public RSAKeyPair(BigInteger d, BigInteger e, BigInteger n) {
	this(d, e, n, null, null);
    }

    public RSAKeyPair(BigInteger d, BigInteger e, BigInteger n,
		      BigInteger p, BigInteger q) {
	super((p!=null ? 
	       new BigInteger[] {d, p, q, 
				 d.mod(p.subtract(Util.ONE)),
				 d.mod(q.subtract(Util.ONE)),
				 q.modInverse(p),
				 p.modInverse(q)}
	       : new BigInteger[] {d}),
	      new BigInteger[] {e, n});
    }

    public KeyPair read(DataInputStream dis) throws IOException {
	BigInteger[][] keys=KeyPair.readKeys(dis);
	
	return (keys[0].length > 1 ?
		new RSAKeyPair(keys[0][0], keys[1][0], keys[1][1],
			       keys[0][1], keys[0][2]) :
		new RSAKeyPair(keys[0][0], keys[1][0], keys[1][1]));
    }

    public void write(OutputStream out) throws IOException {
	super.write(new DataOutputStream(out), getClass().getName());
    }
			       
    public String keyType() {
	return "RSA-1024";
    }

    public BigInteger getD() {
	return x[0];
    }

    public BigInteger getE() {
	return y[0];
    }

    public BigInteger getN() {
	return y[1];
    }

    public BigInteger getP() {
	return (x.length > 1 ? x[1] : null);
    }

    public BigInteger getQ() {
	return (x.length > 1 ? x[2] : null);
    }

    public BigInteger getDmodPsub1() {
	return (x.length > 1 ? x[3] : null);
    }

    public BigInteger getDmodQsub1() {
	return (x.length > 1 ? x[4] : null);
    }

    public BigInteger getQinvmodP() {
	return (x.length > 1 ? x[5] : null);
    }

    public BigInteger getPinvmodQ() {
	return (x.length > 1 ? x[6] : null);
    }

    public static void main(String[] args) throws Exception {
	RSAKeyPair kp=null;
	if (args[0].equals("test")) {
	    kp=generate(BigInteger.valueOf(47),
			BigInteger.valueOf(71),
			true);
	    BigInteger c=RSA.encrypt(kp, BigInteger.valueOf(688));
	    System.err.println("e(688)="+c);
	    System.err.println("d("+c+")="+RSA.decrypt(kp, c));
	    return;
	} else if (args[0].equals("read")) {
	    File f=new File(args[1]);
	    kp=(RSAKeyPair)CryptoKey.read(new FileInputStream(f));
	} else {
	    File f=new File(args[0]);
	    FileOutputStream out=new FileOutputStream(f);
	    
	    long s=System.currentTimeMillis();
	    for (int i=0; i<1; i++) {
		kp=generate(1024, new Yarrow(), true);
	    }
	    System.err.println(System.currentTimeMillis()-s);
	    kp.write(out);
	    out.close();
	}
	System.err.println(kp.verboseToString());
    }
}
